# OAuth tokens

This document explains how are the OAuth tokens implemented and their database
layout in Redis.

OAuth tokens are used for authorization, and are generated by an external system
and sent to us so that we check them when processing OAuth authorization calls.

Basically they map a token to a given application identifier and, in addition
and optionally, to a user identifier. This way, instead of providing an
application identifier (or a user key) or a user identifier, they provide an
access token that the backend has to check against.

# Design

The system works on the following assumptions:

* Tokens are expected to be *UNIQUE* per service. Duplicated tokens between
  applications or users are explicitly *not* supported.

* Tokens map to a tuple (application, user) for each service. Different tokens
  can map to the same tuple.

* Tokens can *NOT* be shared between users.

* Tokens can be application-wide or user-specific *for each application*.

* Application and user identifiers specified in the endpoint parameters take
  precedence over those provided by a token.

# CR(U)D

## Reading ONE token (aka mapping a token to an application)

The Redis layout for mapping a token to a service is a single key-value:

* `oauth_access_tokens/service:#{service_id}/#{token}`

So it depends only on a service. This key is known as the `token_key`. The value
of this key *must* contain an application identifier.

The application identifier is 'app_id' and the value has the form:

* `app_id`

Appropriate checks are in place to parse the above possibilities.

## Creating and Deleting tokens

Creation and deletion are more complex than just creating or deleting
`token_key`s. This is so because we have to support an extra feature, token
listing, without resorting to scanning the database for key pattern matching
(because that would be a no-no in production). Token listing is used by services
to manage and assign tokens.

This is why we track the list of tokens that the combination of a given service
and application have. Since tokens are unique within that combination, we use
Redis sets for storage. These sets are known as `token_set_key`.

Their format is:

* `"oauth_access_tokens/service:#{service_id}/app:#{app_id}/"`

### Redis steps

When storing a new token, we add `token_key` in Redis and assign an optional TTL
to it if the token expires after a certain amount of time. The value of such key
is the application id referred.

We also add the token to the `token_set_key` so that we can list the tokens for
that application. We do the reverse steps when deleting.

## Listing all tokens for a service and application

This is the feature that requires walking the token set.

> The current implementation uses Redis operations that could lead to potential
> DoSes, because it obtains the members of an arbitrarily big set in a single go.
>
> We expect this limitation to be solved incrementally in the future as set sizes
> grow. The reasoning behind this is that we lack a way to ask for certain ranges
> in the endpoint (or a way to stream results) and we don't yet use Redis' scan
> support anywhere in the codebase to avoid blocking the database.

Additionally, we want to take the opportunity to make some janitorial work.
Because tokens can expire, we check that they are still valid, and if not, do
all the related housekeeping.

When we are done, we return the tokens in an array.

### Redis steps

First thing we do is getting the members of the `token_set_key` set.

For each token we get, we build its `token_key` and get it from Redis. If the
result was not nil, it means we got a correct application id, and thus create a
new `OAuth::Token` object ready to be returned.

If the result *was* nil, it means that token is no longer valid (expired). In
that case, we remove the token from the `token_set_key` set.

# Authorization

Authorization is granted or denied based on the application id that the token
points to. Obviously, no authorization will occur if there is no such mapping.
However, *all and any* application identifiers provided through parameters will
take precedence over identifiers found through the token. (this, inexplicably,
totally bypasses `user_key` at the moment).
