require File.expand_path(File.dirname(__FILE__) + '/../../test_helper')

class Abilities::BuyersTest < ActiveSupport::TestCase
  def setup
    @provider = Factory(:provider_account)
    @buyer    = Factory(:buyer_account, :provider_account => @provider)
    @not_admin = Factory(:user, :account => @buyer, :role => :member)
  end

  test "provider admin can read, update and destroy buyer accounts" do
    ability = Ability.new(@provider.admins.first)

    assert ability.can?(:read, @provider => Account)
    assert ability.can?(:read, @buyer)

    assert ability.can?(:update,  @buyer)
    assert ability.can?(:destroy, @buyer)
  end

  test "provider admin can reject and approve buyer accounts" do
    ability = Ability.new(@provider.admins.first)

    assert ability.can?(:approve, @buyer)
    assert ability.can?(:reject,  @buyer)
  end

  test "provider members can read buyer accounts" do
    ability = Ability.new(Factory(:user, :account => @provider))

    assert ability.can?(:read, @provider => Account)
    assert ability.can?(:read, @buyer)
  end

  test "provider members can't create/update/destroy buyer accounts" do
    ability = Ability.new(Factory(:user, :account => @provider))

    assert !ability.can?(:create,  @provider => Account)
    assert !ability.can?(:update,  @buyer)
    assert !ability.can?(:destroy, @buyer)
  end

  test "provider memebers can't reject nor approve buyer accounts" do
    ability = Ability.new(Factory(:user, :account => @provider))

    assert !ability.can?(:approve, @buyer)
    assert !ability.can?(:reject,  @buyer)
  end

  test "provider admin can't configure buyer accounts" do
    ability = Ability.new(@provider.admins.first)

    assert !ability.can?(:configure, @buyer)
  end

  test "superadmin can configure provider accounts" do
    ability = Ability.new(Account.master.admins.first)

    assert ability.can?(:configure, @provider)
  end

  test "provider admin can manage his buyer users" do
    ability = Ability.new(@provider.admins.first)

    assert ability.can?(:read, @buyer => User)

    assert ability.can?(:read,      @buyer.users.first)
    assert ability.can?(:update,    @buyer.users.first)
    assert ability.can?(:destroy,   @buyer.users.first)
    assert ability.can?(:suspend,   @buyer.users.first)
    assert ability.can?(:unsuspend, @buyer.users.first)
  end

  test "provider admin can update role of his buyer users" do
    buyer_user = Factory(:user, :account => @buyer)

    ability = Ability.new(@provider.admins.first)
    assert ability.can?(:update_role, buyer_user)
  end

  test "provider admin can update role of his buyer users even when advanced cms is enabled" do
    admin = @provider.admins.first
    buyer_user = Factory(:user, :account => @buyer)

    ability = Ability.new(admin)
    assert ability.can?(:update_role, buyer_user)
  end

  test "provider admin can't update role of a buyer user if that user is the only admin of his account" do
    @buyer.users.first.update_attribute(:role, :admin)
    @buyer.users[1..-1].each { |user| user.update_attribute(:role, :member) }

    ability = Ability.new(@provider.admins.first)
    assert !ability.can?(:update_role, @buyer.users.first)
  end

  test "provider admin can't impersonate buyer users" do
    ability = Ability.new(@provider.admins.first)
    assert !ability.can?(:impersonate, @buyer.users.first)
  end

  test "provider admin can't impersonate other provider users" do
    other_provider = Factory(:provider_account)

    ability = Ability.new(@provider.admins.first)
    assert !ability.can?(:impersonate, other_provider.users.first)
  end

  test "buyer can update their user details" do
    ability = Ability.new(@buyer.admins.first)
    assert ability.can?(:update, @buyer.admins.first)
  end

  test "non admin buyer can update their user details" do
    ability = Ability.new(@not_admin)
    assert ability.can?(:update, @not_admin)
  end

  test "buyer cant update their user details if useraccountarea disabled" do
    @provider.settings.update_attribute(:useraccountarea_enabled, false)
    ability = Ability.new(@buyer.admins.first)
    assert !ability.can?(:update, @buyer.admins.first)
  end

  test "provider can always update their user details" do
    @provider.settings.update_attribute(:useraccountarea_enabled, true)
    ability = Ability.new(@provider.admins.first)
    assert ability.can?(:update, @provider.admins.first)

    @provider.settings.update_attribute(:useraccountarea_enabled, false)
    ability = Ability.new(@provider.admins.first)
    assert ability.can?(:update, @provider.admins.first)
  end

  test "buyers can manage alerts if they are enable in cinstance" do
    @cinstance = Factory(:cinstance, :name => "cde", :user_account => @buyer)

    @cinstance.stubs(:buyer_alerts_enabled?).returns(false)
    ability = Ability.new(@buyer.admins.first)
    assert ability.cannot?(:manage_alerts, @cinstance)

    @cinstance.stubs(:buyer_alerts_enabled?).returns(true)
    ability = Ability.new(@buyer.admins.first)
    assert ability.can?(:manage_alerts, @cinstance)
  end

end
