package com.redhat.installer.tests.validator;

import java.util.*;
import java.io.File;

import com.redhat.installer.tests.mock.MockProcessingClient;
import com.redhat.installer.tests.testers.DataValidatorTester;
import org.junit.*;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.ArrayUtils;

import com.izforge.izpack.installer.DataValidator;
import com.redhat.installer.tests.TestUtils;
import com.redhat.installer.tests.constants.CommonStrings;
import com.redhat.installer.validator.DirectoryValidator;


import static junit.framework.TestCase.assertEquals;


@RunWith(Parameterized.class)
public class DirectoryValidatorTest extends DataValidatorTester implements CommonStrings
{
    private String os;
    private String rootDir;
    private String separator;
    private DirectoryValidator directoryValidator;
    private MockProcessingClient mockProcessingClient;

    private static String [] invalidWindowsCharacters;
    private static String [] invalidUnixCharacters;
    private static String [] invalidInstallerCharacters;

    private ArrayList<String> testPatterns = new ArrayList<String>();
    private static Map<String, String[]> validPathsTable = new HashMap<String, String[]>();
    private static Map<String, String[]> invalidPathsTable = new HashMap<String, String[]>();

    private final static String [] validUnixPaths = {
            "/foo bar/with spaces/", "/foo/bar/dir/is/fine"
    };
    private final static String [] validWindowsPaths = {
            "C:\\foo bar\\with spaces\\", "C:\\Program Files\\", "C:\\Program Files (x86)\\"
    };

    private final static String [] invalidUnixPaths = {
            "/foo bar/two  space/", "/noescaping \\please"
    };
    private final static String [] invalidWindowsPaths = {
            "C:\\foo bar\\\\WithEmptyDir\\", "C:\\HasLeadingSpace\\ "
    };

    private String [] invalidCharacters(String os)
    {
        if (os.toLowerCase().contains(WINDOWS)) return invalidWindowsCharacters;
        else                                    return invalidUnixCharacters;
    }

    public DirectoryValidatorTest(String os, String separator, String rootDir)
    {
        this.os = os;
        this.rootDir = rootDir;
        this.separator = separator;
    }

    private DataValidator.Status getPathStatus(String path)
    {
        idata.setInstallPath(path);
        DataValidator.Status status = directoryValidator.validateData(idata);
        return status;
    }

    private boolean isPathValid(String path)
    {
        boolean result;
        mockProcessingClient.addToFields(path);
        result = directoryValidator.validate(mockProcessingClient);
        mockProcessingClient.clear();
        return result;

    }

    @Parameterized.Parameters
    public static Collection<Object[]> data()
    {
        String fileSeperator;
        String rootDirectory;
        List<Object[]> params = new ArrayList<Object[]>();

        for (String os : OSES)
        {
            if (os.equals(WINDOWS))
            {
                rootDirectory = "C:\\";
                fileSeperator = "\\";
            }
            else
            {
                rootDirectory = "/";
                fileSeperator = "/";
            }

            Object[] param = { os , fileSeperator, rootDirectory};
            params.add(param);
        }

        return params;
    }

    @BeforeClass
    public static void init() throws Exception
    {
        idata.langpack = TestUtils.createMockLangpack("TargetPanel.invalid"); //Dependency of DirectoryValidator
        invalidInstallerCharacters = DirectoryValidator.invalidCharacters;
        invalidWindowsCharacters = (String[]) ArrayUtils.addAll(invalidInstallerCharacters, DirectoryValidator.invalidWindows);
        invalidUnixCharacters= (String[]) ArrayUtils.addAll(invalidInstallerCharacters, DirectoryValidator.invalidUnix);
        validPathsTable.put(UNIX, validUnixPaths);
        validPathsTable.put(WINDOWS, validWindowsPaths);
        invalidPathsTable.put(UNIX, invalidUnixPaths);
        invalidPathsTable.put(WINDOWS, invalidWindowsPaths);
    }

    @Before
    public void setUp()
    {
        System.setProperty("os.name", os);

        testPatterns.add(rootDir + "%sstarting" + separator);
        testPatterns.add(rootDir + "trailing%s" + separator);
        testPatterns.add(rootDir + "%s" + separator);

        directoryValidator = new DirectoryValidator();
        mockProcessingClient = new MockProcessingClient();
    }

    @After
    public void tearDown()
    {
        testPatterns.clear();
        idata.getVariables().clear();
        directoryValidator = null;
        mockProcessingClient = null;
    }

    @AfterClass
    public static void specificDestroy() throws Exception{
        FileUtils.deleteDirectory(new File(TestUtils.INSTALL_PATH));
        TestUtils.destroyIdataSingleton();
    }

    @Test
    public void testInvalidCharacters()
    {
        String path;
        DataValidator.Status panelResult;
        boolean userInputResult;

        for (String pattern : testPatterns)
        {
            for (String s : invalidCharacters(os))
            {
                path = String.format(pattern, s);
                panelResult = getPathStatus(path);
                userInputResult = isPathValid(path);

                assertEquals("testInvalidPatterns|"+os+"|"+path,
                             DataValidator.Status.ERROR, panelResult);
                assertEquals("testInvalidPatterns|"+os+"|"+path,
                             false, userInputResult);
            }
        }
    }


    @Test
    public void testValidPaths()
    {
        String [] specificPaths = validPathsTable.get(os);
        DataValidator.Status panelResult;
        boolean userInputResult;

        for (String path : specificPaths)
        {
            panelResult = getPathStatus(path);
            userInputResult = isPathValid(path);

            assertEquals("testValidPaths | " + os + "'" + path + "'",
                         DataValidator.Status.OK, panelResult);
            assertEquals("testValidPaths | " + os + "'" + path + "'",
                         true, userInputResult);
        }
    }

    @Test
    public void invalidPaths()
    {
        String [] specificPaths = invalidPathsTable.get(os);
        DataValidator.Status panelResult;
        boolean userInputResult;

        for (String path : specificPaths)
        {
            panelResult = getPathStatus(path);
            userInputResult = isPathValid(path);

            assertEquals("invalidPaths | " + os + "'" + path + "'",
                    DataValidator.Status.ERROR, panelResult);
            assertEquals("invalidPaths | " + os + "'" + path + "'",
                    false, userInputResult);
        }
    }
}
