package com.redhat.installer.asconfiguration.processpanel.postinstallation;

import com.izforge.izpack.LocaleDatabase;
import com.izforge.izpack.installer.AutomatedInstallData;
import com.izforge.izpack.util.AbstractUIProcessHandler;
import com.redhat.installer.asconfiguration.ascontroller.EmbeddedServerCommands;
import com.redhat.installer.logging.InstallationLogger;
import org.apache.commons.lang.StringUtils;
import org.jboss.dmr.ModelNode;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.ArgumentCaptor;
import org.mockito.Captor;
import org.mockito.Mock;
import org.mockito.ScopedMock;
import org.mockito.junit.MockitoJUnitRunner;
import org.wildfly.security.credential.PasswordCredential;
import org.wildfly.security.credential.store.CredentialStore;
import org.wildfly.security.credential.store.CredentialStore.CredentialSourceProtectionParameter;
import org.wildfly.security.credential.store.CredentialStoreException;
import org.wildfly.security.password.interfaces.ClearPassword;

import java.io.IOException;
import java.security.NoSuchAlgorithmException;
import java.security.Provider;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;

import static org.junit.Assert.assertEquals;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyString;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.mockStatic;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import static org.mockito.internal.verification.VerificationModeFactory.times;

@RunWith(MockitoJUnitRunner.class)
public class InstallCredentialStoreTest {

    private static final String JBOSS_SERVER_CONFIG_DIR = "/random/folder/path";

    @Mock
    ModelNode modelNodeMock;

    @Mock
    EmbeddedServerCommands serverCommandsMock;

    @Mock
    AutomatedInstallData idataMock;

    @Mock
    CredentialStore credentialStoreMock;

    @Mock
    LocaleDatabase localeDatabaseMock;

    @Captor
    ArgumentCaptor<Map<String, String>> attributesCaptor;

    @Captor
    ArgumentCaptor<CredentialSourceProtectionParameter> protectionParameterCaptor;

    @Captor
    ArgumentCaptor<String> stringCaptor;

    @Captor
    ArgumentCaptor<PasswordCredential> passwordCredentialCaptor;

    private static final List<ScopedMock> mockedSettings = new ArrayList<>();

    @BeforeClass
    public static void classSetup() {
        mockedSettings.add(mockStatic(AutomatedInstallData.class));
        mockedSettings.add(mockStatic(InstallationLogger.class));
        mockedSettings.add(mockStatic(CredentialStore.class));
    }

    @AfterClass
    public static void classTearDown() {
        mockedSettings.forEach(ScopedMock::close);
    }

    @Before
    public void setup() throws NoSuchAlgorithmException {
        System.setProperty("jboss.server.config.dir", JBOSS_SERVER_CONFIG_DIR);

        when(AutomatedInstallData.getInstance()).thenReturn(idataMock);
        when(InstallationLogger.getLogger()).thenReturn(mock(Logger.class));
        when(CredentialStore.getInstance(anyString(), any(Provider.class))).thenReturn(credentialStoreMock);

        when(modelNodeMock.get("outcome")).thenReturn(new ModelNode("success"));
        when(serverCommandsMock.submitCommand(anyString())).thenReturn(modelNodeMock);
        InstallCredentialStore.serverCommands = serverCommandsMock;

        when(localeDatabaseMock.getString(anyString())).thenReturn("");
        idataMock.langpack = localeDatabaseMock;

        InstallCredentialStore.mHandler = mock(AbstractUIProcessHandler.class);
    }

    @Test
    public void shouldCreateStoreInSpecifiedLocationAndProtectedByPassword() throws CredentialStoreException, IOException {
        InstallCredentialStore install = new InstallCredentialStore();
        String expectedPassword = "r4nd0mP4ssw0rd$";
        InstallCredentialStore.arguments = new String[] {
                "create=true",
                "name=test-credential-store",
                "location=my_test_credentials_store.cs",
                "relative-to=jboss.server.config.dir",
                "password=" + expectedPassword
        };
        install.performOperation();

        verify(credentialStoreMock).initialize(attributesCaptor.capture(), protectionParameterCaptor.capture(), any());

        Map<String, String> attributes = attributesCaptor.getValue();
        assertEquals("JCEKS", attributes.get("keyStoreType"));
        assertEquals("true", attributes.get("create"));
        assertEquals(JBOSS_SERVER_CONFIG_DIR + "/my_test_credentials_store.cs", attributes.get("location"));

        CredentialSourceProtectionParameter protectionParameter = protectionParameterCaptor.getValue();
        String password = passwordFrom(protectionParameterCaptor.getValue().getCredentialSource().getCredential(PasswordCredential.class));
        assertEquals(expectedPassword, password);
    }

    @Test
    public void shouldAddAliasesToStore() throws CredentialStoreException, IOException {
        InstallCredentialStore install = new InstallCredentialStore();
        InstallCredentialStore.arguments = new String[] {
                "create=true",
                "name=test-credential-store",
                "location=my_test_credentials_store.cs",
                "relative-to=jboss.server.config.dir",
                "password=12345",
                "alias=identity1:secret1",
                "alias=identity2:secret2"
        };
        install.performOperation();

        verify(credentialStoreMock, times(2)).store(stringCaptor.capture(), passwordCredentialCaptor.capture());
        List<String> identities = stringCaptor.getAllValues();
        List<PasswordCredential> secrets = passwordCredentialCaptor.getAllValues();
        for (int i = 0; i < identities.size(); i++) {
            assertEquals("identity" + (i + 1), identities.get(i));
            assertEquals("secret" + (i + 1), passwordFrom(secrets.get(i)));
        }
    }

    @Test
    public void shouldInstallStoreOnServer() {
        InstallCredentialStore install = new InstallCredentialStore();
        String storeName = "test-credential-store";
        String location = "my_test_credentials_store.cs";
        String relativeTo = "jboss.server.config.dir";
        InstallCredentialStore.arguments = new String[] {
                "name=" + storeName,
                "location=" + location,
                "relative-to=" + relativeTo,
                "password=12345"
        };
        install.performOperation();

        verify(serverCommandsMock).submitCommand(stringCaptor.capture());
        String expectedCommand = commandFor(storeName, "MASK-CWOBbCD7cmW;12345678;200", location, relativeTo, false);
        assertEquals(expectedCommand, stringCaptor.getValue());
    }

    private String passwordFrom(PasswordCredential passwordCredential) {
        return new String(passwordCredential.getPassword(ClearPassword.class).getPassword());
    }

    private String commandFor(String name, String maskedPassword, String location, String relativeTo, boolean create) {
        String command = "/subsystem=elytron/credential-store=" + name + ":add(location=" + location;
        if (!StringUtils.isEmpty(relativeTo)) {
            command += ", relative-to=" + relativeTo;
        }
        command += ", credential-reference={clear-text=\"" + maskedPassword + "\"}, create=" + create + ")";
        return command;
    }
}