/*
 * Copyright (c) 2007-2012, Stephen Colebourne & Michael Nascimento Santos
 *
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  * Neither the name of JSR-310 nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package java.time;

import java.io.Serializable;
import java.time.calendrical.ChronoField;
import java.time.calendrical.ChronoUnit;
import java.time.calendrical.DateTime;
import java.time.calendrical.DateTime.WithAdjuster;
import java.time.calendrical.DateTimeAccessor;
import java.time.calendrical.DateTimeAdjusters;
import java.time.calendrical.DateTimeField;
import java.time.calendrical.DateTimeValueRange;
import java.time.calendrical.PeriodUnit;
import java.time.chrono.ChronoLocalDateTime;
import java.time.chrono.ISOChrono;
import java.time.format.DateTimeParseException;
import java.time.jdk8.DefaultInterfaceChronoLocalDateTime;
import java.time.jdk8.Jdk7Methods;
import java.time.jdk8.Jdk8Methods;
import java.time.zone.ZoneRules;

import static java.time.LocalTime.HOURS_PER_DAY;
import static java.time.LocalTime.MICROS_PER_DAY;
import static java.time.LocalTime.MILLIS_PER_DAY;
import static java.time.LocalTime.MINUTES_PER_DAY;
import static java.time.LocalTime.NANOS_PER_DAY;
import static java.time.LocalTime.NANOS_PER_HOUR;
import static java.time.LocalTime.NANOS_PER_MINUTE;
import static java.time.LocalTime.NANOS_PER_SECOND;
import static java.time.LocalTime.SECONDS_PER_DAY;

/**
 * A date-time without a time-zone in the ISO-8601 calendar system, such as {@code 2007-12-03T10:15:30}.
 * <p>
 * {@code LocalDateTime} is an immutable date-time object that represents a date-time, often viewed as
 * year-month-day-hour-minute-second. This object can also access other fields such as day-of-year,
 * day-of-week and week-of-year.
 * <p>
 * This class stores all date and time fields, to a precision of nanoseconds. It does not store or represent a
 * time-zone. For example, the value "2nd October 2007 at 13:45.30.123456789" can be stored in an
 * {@code LocalDateTime}.
 * 
 * <h4>Implementation notes</h4> This class is immutable and thread-safe.
 */
public final class LocalDateTime extends DefaultInterfaceChronoLocalDateTime<ISOChrono> implements
    ChronoLocalDateTime<ISOChrono>, DateTime, WithAdjuster, Serializable {

  /**
   * Constant for the local date-time of midnight at the start of the minimum date. This combines
   * {@link LocalDate#MIN_DATE} and {@link LocalTime#MIN_TIME}. This could be used by an application as a
   * "far past" date-time.
   */
  public static final LocalDateTime MIN_DATE_TIME = LocalDateTime.of(LocalDate.MIN_DATE, LocalTime.MIN_TIME);

  /**
   * Constant for the local date-time just before midnight at the end of the maximum date. This combines
   * {@link LocalDate#MAX_DATE} and {@link LocalTime#MAX_TIME}. This could be used by an application as a
   * "far future" date-time.
   */
  public static final LocalDateTime MAX_DATE_TIME = LocalDateTime.of(LocalDate.MAX_DATE, LocalTime.MAX_TIME);

  /**
   * Serialization version.
   */
  private static final long serialVersionUID = 6207766400415563566L;

  /**
   * The date part.
   */
  private final LocalDate date;

  /**
   * The time part.
   */
  private final LocalTime time;

  // -----------------------------------------------------------------------
  /**
   * Obtains the current date-time from the system clock in the default time-zone.
   * <p>
   * This will query the {@link Clock#systemDefaultZone() system clock} in the default time-zone to obtain the
   * current date-time.
   * <p>
   * Using this method will prevent the ability to use an alternate clock for testing because the clock is
   * hard-coded.
   * 
   * @return the current date-time using the system clock and default time-zone, not null
   */
  public static LocalDateTime now() {

    return now(Clock.systemDefaultZone());
  }

  /**
   * Obtains the current date-time from the system clock in the specified time-zone.
   * <p>
   * This will query the {@link Clock#system(ZoneId) system clock} to obtain the current date-time. Specifying
   * the time-zone avoids dependence on the default time-zone.
   * <p>
   * Using this method will prevent the ability to use an alternate clock for testing because the clock is
   * hard-coded.
   * 
   * @return the current date-time using the system clock, not null
   */
  public static LocalDateTime now(ZoneId zone) {

    return now(Clock.system(zone));
  }

  /**
   * Obtains the current date-time from the specified clock.
   * <p>
   * This will query the specified clock to obtain the current date-time. Using this method allows the use of
   * an alternate clock for testing. The alternate clock may be introduced using {@link Clock dependency
   * injection}.
   * 
   * @param clock the clock to use, not null
   * @return the current date-time, not null
   */
  public static LocalDateTime now(Clock clock) {

    Jdk7Methods.Objects_requireNonNull(clock, "clock");
    final Instant now = clock.instant(); // called once
    ZoneOffset offset = clock.getZone().getRules().getOffset(now);
    return ofEpochSecond(now.getEpochSecond(), now.getNano(), offset);
  }

  // -----------------------------------------------------------------------
  /**
   * Obtains an instance of {@code LocalDateTime} from year, month, day, hour and minute, setting the second
   * and nanosecond to zero.
   * <p>
   * The day must be valid for the year and month, otherwise an exception will be thrown. The second and
   * nanosecond fields will be set to zero.
   * 
   * @param year the year to represent, from MIN_YEAR to MAX_YEAR
   * @param month the month-of-year to represent, not null
   * @param dayOfMonth the day-of-month to represent, from 1 to 31
   * @param hour the hour-of-day to represent, from 0 to 23
   * @param minute the minute-of-hour to represent, from 0 to 59
   * @return the local date-time, not null
   * @throws DateTimeException if the value of any field is out of range
   * @throws DateTimeException if the day-of-month is invalid for the month-year
   */
  public static LocalDateTime of(int year, Month month, int dayOfMonth, int hour, int minute) {

    LocalDate date = LocalDate.of(year, month, dayOfMonth);
    LocalTime time = LocalTime.of(hour, minute);
    return new LocalDateTime(date, time);
  }

  /**
   * Obtains an instance of {@code LocalDateTime} from year, month, day, hour, minute and second, setting the
   * nanosecond to zero.
   * <p>
   * The day must be valid for the year and month, otherwise an exception will be thrown. The nanosecond field
   * will be set to zero.
   * 
   * @param year the year to represent, from MIN_YEAR to MAX_YEAR
   * @param month the month-of-year to represent, not null
   * @param dayOfMonth the day-of-month to represent, from 1 to 31
   * @param hour the hour-of-day to represent, from 0 to 23
   * @param minute the minute-of-hour to represent, from 0 to 59
   * @param second the second-of-minute to represent, from 0 to 59
   * @return the local date-time, not null
   * @throws DateTimeException if the value of any field is out of range
   * @throws DateTimeException if the day-of-month is invalid for the month-year
   */
  public static LocalDateTime of(int year, Month month, int dayOfMonth, int hour, int minute, int second) {

    LocalDate date = LocalDate.of(year, month, dayOfMonth);
    LocalTime time = LocalTime.of(hour, minute, second);
    return new LocalDateTime(date, time);
  }

  /**
   * Obtains an instance of {@code LocalDateTime} from year, month, day, hour, minute, second and nanosecond.
   * <p>
   * The day must be valid for the year and month, otherwise an exception will be thrown.
   * 
   * @param year the year to represent, from MIN_YEAR to MAX_YEAR
   * @param month the month-of-year to represent, not null
   * @param dayOfMonth the day-of-month to represent, from 1 to 31
   * @param hour the hour-of-day to represent, from 0 to 23
   * @param minute the minute-of-hour to represent, from 0 to 59
   * @param second the second-of-minute to represent, from 0 to 59
   * @param nanoOfSecond the nano-of-second to represent, from 0 to 999,999,999
   * @return the local date-time, not null
   * @throws DateTimeException if the value of any field is out of range
   * @throws DateTimeException if the day-of-month is invalid for the month-year
   */
  public static LocalDateTime of(int year, Month month, int dayOfMonth, int hour, int minute, int second,
      int nanoOfSecond) {

    LocalDate date = LocalDate.of(year, month, dayOfMonth);
    LocalTime time = LocalTime.of(hour, minute, second, nanoOfSecond);
    return new LocalDateTime(date, time);
  }

  // -----------------------------------------------------------------------
  /**
   * Obtains an instance of {@code LocalDateTime} from year, month, day, hour and minute, setting the second
   * and nanosecond to zero.
   * <p>
   * The day must be valid for the year and month, otherwise an exception will be thrown. The second and
   * nanosecond fields will be set to zero.
   * 
   * @param year the year to represent, from MIN_YEAR to MAX_YEAR
   * @param month the month-of-year to represent, from 1 (January) to 12 (December)
   * @param dayOfMonth the day-of-month to represent, from 1 to 31
   * @param hour the hour-of-day to represent, from 0 to 23
   * @param minute the minute-of-hour to represent, from 0 to 59
   * @return the local date-time, not null
   * @throws DateTimeException if the value of any field is out of range
   * @throws DateTimeException if the day-of-month is invalid for the month-year
   */
  public static LocalDateTime of(int year, int month, int dayOfMonth, int hour, int minute) {

    LocalDate date = LocalDate.of(year, month, dayOfMonth);
    LocalTime time = LocalTime.of(hour, minute);
    return new LocalDateTime(date, time);
  }

  /**
   * Obtains an instance of {@code LocalDateTime} from year, month, day, hour, minute and second, setting the
   * nanosecond to zero.
   * <p>
   * The day must be valid for the year and month, otherwise an exception will be thrown. The nanosecond field
   * will be set to zero.
   * 
   * @param year the year to represent, from MIN_YEAR to MAX_YEAR
   * @param month the month-of-year to represent, from 1 (January) to 12 (December)
   * @param dayOfMonth the day-of-month to represent, from 1 to 31
   * @param hour the hour-of-day to represent, from 0 to 23
   * @param minute the minute-of-hour to represent, from 0 to 59
   * @param second the second-of-minute to represent, from 0 to 59
   * @return the local date-time, not null
   * @throws DateTimeException if the value of any field is out of range
   * @throws DateTimeException if the day-of-month is invalid for the month-year
   */
  public static LocalDateTime of(int year, int month, int dayOfMonth, int hour, int minute, int second) {

    LocalDate date = LocalDate.of(year, month, dayOfMonth);
    LocalTime time = LocalTime.of(hour, minute, second);
    return new LocalDateTime(date, time);
  }

  /**
   * Obtains an instance of {@code LocalDateTime} from year, month, day, hour, minute, second and nanosecond.
   * <p>
   * The day must be valid for the year and month, otherwise an exception will be thrown.
   * 
   * @param year the year to represent, from MIN_YEAR to MAX_YEAR
   * @param month the month-of-year to represent, from 1 (January) to 12 (December)
   * @param dayOfMonth the day-of-month to represent, from 1 to 31
   * @param hour the hour-of-day to represent, from 0 to 23
   * @param minute the minute-of-hour to represent, from 0 to 59
   * @param second the second-of-minute to represent, from 0 to 59
   * @param nanoOfSecond the nano-of-second to represent, from 0 to 999,999,999
   * @return the local date-time, not null
   * @throws DateTimeException if the value of any field is out of range
   * @throws DateTimeException if the day-of-month is invalid for the month-year
   */
  public static LocalDateTime of(int year, int month, int dayOfMonth, int hour, int minute, int second, int nanoOfSecond) {

    LocalDate date = LocalDate.of(year, month, dayOfMonth);
    LocalTime time = LocalTime.of(hour, minute, second, nanoOfSecond);
    return new LocalDateTime(date, time);
  }

  /**
   * Obtains an instance of {@code LocalDateTime} from a date and time.
   * 
   * @param date the local date, not null
   * @param time the local time, not null
   * @return the local date-time, not null
   */
  public static LocalDateTime of(LocalDate date, LocalTime time) {

    Jdk7Methods.Objects_requireNonNull(date, "date");
    Jdk7Methods.Objects_requireNonNull(time, "time");
    return new LocalDateTime(date, time);
  }

  // -------------------------------------------------------------------------
  /**
   * Obtains an instance of {@code LocalDateTime} from an {@code Instant} and zone ID.
   * <p>
   * This creates a local date-time based on the specified instant. First, the offset from UTC/Greenwich is
   * obtained using the zone ID and instant, which is simple as there is only one valid offset for each
   * instant. Then, the instant and offset are used to calculate the local date-time.
   * 
   * @param instant the instant to create the date-time from, not null
   * @param zone the time-zone, which may be an offset, not null
   * @return the local date-time, not null
   * @throws DateTimeException if the result exceeds the supported range
   */
  public static LocalDateTime ofInstant(Instant instant, ZoneId zone) {

    Jdk7Methods.Objects_requireNonNull(instant, "instant");
    Jdk7Methods.Objects_requireNonNull(zone, "zone");
    ZoneRules rules = zone.getRules();
    ZoneOffset offset = rules.getOffset(instant);
    return ofEpochSecond(instant.getEpochSecond(), instant.getNano(), offset);
  }

  /**
   * Obtains an instance of {@code LocalDateTime} using seconds from the epoch of 1970-01-01T00:00:00Z.
   * <p>
   * This allows the {@link ChronoField#INSTANT_SECONDS epoch-second} field to be converted to a local
   * date-time. This is primarily intended for low-level conversions rather than general application usage.
   * 
   * @param epochSecond the number of seconds from the epoch of 1970-01-01T00:00:00Z
   * @param nanoOfSecond the nanosecond within the second, from 0 to 999,999,999
   * @param offset the zone offset, not null
   * @return the local date-time, not null
   * @throws DateTimeException if the result exceeds the supported range
   */
  public static LocalDateTime ofEpochSecond(long epochSecond, int nanoOfSecond, ZoneOffset offset) {

    Jdk7Methods.Objects_requireNonNull(offset, "offset");
    long localSecond = epochSecond + offset.getTotalSeconds(); // overflow caught later
    long localEpochDay = Jdk8Methods.floorDiv(localSecond, SECONDS_PER_DAY);
    int secsOfDay = Jdk8Methods.floorMod(localSecond, SECONDS_PER_DAY);
    LocalDate date = LocalDate.ofEpochDay(localEpochDay);
    LocalTime time = LocalTime.ofSecondOfDay(secsOfDay, nanoOfSecond);
    return new LocalDateTime(date, time);
  }

  // -----------------------------------------------------------------------
  /**
   * Obtains an instance of {@code LocalDateTime} from a date-time object.
   * <p>
   * A {@code DateTimeAccessor} represents some form of date and time information. This factory converts the
   * arbitrary date-time object to an instance of {@code LocalDateTime}.
   * <p>
   * The conversion extracts and combines the {@code LocalDate} and {@code LocalTime}.
   * 
   * @param dateTime the date-time object to convert, not null
   * @return the local date-time, not null
   * @throws DateTimeException if unable to convert to a {@code LocalDateTime}
   */
  public static LocalDateTime from(DateTimeAccessor dateTime) {

    if (dateTime instanceof LocalDateTime) {
      return (LocalDateTime) dateTime;
    } else if (dateTime instanceof ZonedDateTime) {
      return ((ZonedDateTime) dateTime).getDateTime();
    }
    LocalDate date = LocalDate.from(dateTime);
    LocalTime time = LocalTime.from(dateTime);
    return new LocalDateTime(date, time);
  }

  // -----------------------------------------------------------------------
  /**
   * Obtains an instance of {@code LocalDateTime} from a text string such as {@code 2007-12-03T10:15:30}.
   * <p>
   * The string must represent a valid date-time and is parsed using
   * {@link java.time.format.DateTimeFormatters#isoLocalDateTime()}.
   * 
   * @param text the text to parse such as "2007-12-03T10:15:30", not null
   * @return the parsed local date-time, not null
   * @throws DateTimeParseException if the text cannot be parsed
   */
  public static LocalDateTime parse(CharSequence text) {

    // "YYYY-MM-ddTHH:mm".length() == 16, "YYYY-MM-ddTHH:mm:ss.SSSSSSSSS".length() == 29
    int length = text.length();
    if ((length >= 16) && (length <= 32)) {
      int timeStartIndex = 10;
      while (Character.toUpperCase(text.charAt(timeStartIndex)) != 'T') {
        timeStartIndex++;
        if (timeStartIndex >= length) {
          timeStartIndex = -1;
          break;
        }
      }
      if (timeStartIndex > 0) {
        LocalDate date = LocalDate.parse(text.subSequence(0, timeStartIndex));
        LocalTime time = LocalTime.parse(text.subSequence(timeStartIndex + 1, length));
        return new LocalDateTime(date, time);
      }
    }
    throw new DateTimeParseException("Expected format yyyy-MM-ddTHH:mm:ss", text, 0, null);
  }

  // -----------------------------------------------------------------------
  /**
   * Constructor.
   * 
   * @param date the date part of the date-time, validated not null
   * @param time the time part of the date-time, validated not null
   */
  private LocalDateTime(LocalDate date, LocalTime time) {

    this.date = date;
    this.time = time;
  }

  /**
   * Returns a copy of this date-time with the new date and time, checking to see if a new object is in fact
   * required.
   * 
   * @param newDate the date of the new date-time, not null
   * @param newTime the time of the new date-time, not null
   * @return the date-time, not null
   */
  private LocalDateTime with(LocalDate newDate, LocalTime newTime) {

    if (this.date == newDate && this.time == newTime) {
      return this;
    }
    return new LocalDateTime(newDate, newTime);
  }

  // -----------------------------------------------------------------------
  @Override
  public boolean isSupported(DateTimeField field) {

    if (field instanceof ChronoField) {
      ChronoField f = (ChronoField) field;
      return f.isDateField() || f.isTimeField();
    }
    return field != null && field.doIsSupported(this);
  }

  @Override
  public DateTimeValueRange range(DateTimeField field) {

    if (field instanceof ChronoField) {
      ChronoField f = (ChronoField) field;
      return (f.isTimeField() ? this.time.range(field) : this.date.range(field));
    }
    return field.doRange(this);
  }

  @Override
  public int get(DateTimeField field) {

    if (field instanceof ChronoField) {
      ChronoField f = (ChronoField) field;
      return (f.isTimeField() ? this.time.get(field) : this.date.get(field));
    }
    return super.get(field);
  }

  @Override
  public long getLong(DateTimeField field) {

    if (field instanceof ChronoField) {
      ChronoField f = (ChronoField) field;
      if (f.equals(ChronoField.INSTANT_SECONDS)) {
          return this.time.getSecond() + this.time.getMinute() * 60 + this.time.getHour() * 60 * 60 +
                  this.date.toEpochDay() * 60 * 60 * 24;
      }
      return (f.isTimeField() ? this.time.getLong(field) : this.date.getLong(field));
    }
    return field.doGet(this);
  }

  // -----------------------------------------------------------------------
  /**
   * Gets the {@code LocalDate} part of this date-time.
   * <p>
   * This returns a {@code LocalDate} with the same year, month and day as this date-time.
   * 
   * @return the date part of this date-time, not null
   */
  @Override
  public LocalDate getDate() {

    return this.date;
  }
  
  /**
   * Gets the {@code LocalDate} part of this date-time.
   * <p>
   * This returns a {@code LocalDate} with the same year, month and day as this date-time.
   * 
   * @return the date part of this date-time, not null
   */
  public LocalDate toLocalDate() {

    return this.date;
  }

  /**
   * Gets the year field.
   * <p>
   * This method returns the primitive {@code int} value for the year.
   * <p>
   * The year returned by this method is proleptic as per {@code get(YEAR)}. To obtain the year-of-era, use
   * {@code get(YEAR_OF_ERA}.
   * 
   * @return the year, from MIN_YEAR to MAX_YEAR
   */
  public int getYear() {

    return this.date.getYear();
  }

  /**
   * Gets the month-of-year field from 1 to 12.
   * <p>
   * This method returns the month as an {@code int} from 1 to 12. Application code is frequently clearer if
   * the enum {@link Month} is used by calling {@link #getMonth()}.
   * 
   * @return the month-of-year, from 1 to 12
   * @see #getMonth()
   */
  public int getMonthValue() {

    return this.date.getMonthValue();
  }

  /**
   * Gets the month-of-year field using the {@code Month} enum.
   * <p>
   * This method returns the enum {@link Month} for the month. This avoids confusion as to what {@code int}
   * values mean. If you need access to the primitive {@code int} value then the enum provides the
   * {@link Month#getValue() int value}.
   * 
   * @return the month-of-year, not null
   * @see #getMonthValue()
   */
  public Month getMonth() {

    return this.date.getMonth();
  }

  /**
   * Gets the day-of-month field.
   * <p>
   * This method returns the primitive {@code int} value for the day-of-month.
   * 
   * @return the day-of-month, from 1 to 31
   */
  public int getDayOfMonth() {

    return this.date.getDayOfMonth();
  }

  /**
   * Gets the day-of-year field.
   * <p>
   * This method returns the primitive {@code int} value for the day-of-year.
   * 
   * @return the day-of-year, from 1 to 365, or 366 in a leap year
   */
  public int getDayOfYear() {

    return this.date.getDayOfYear();
  }

  /**
   * Gets the day-of-week field, which is an enum {@code DayOfWeek}.
   * <p>
   * This method returns the enum {@link DayOfWeek} for the day-of-week. This avoids confusion as to what
   * {@code int} values mean. If you need access to the primitive {@code int} value then the enum provides the
   * {@link DayOfWeek#getValue() int value}.
   * <p>
   * Additional information can be obtained from the {@code DayOfWeek}. This includes textual names of the
   * values.
   * 
   * @return the day-of-week, not null
   */
  public DayOfWeek getDayOfWeek() {

    return this.date.getDayOfWeek();
  }

  // -----------------------------------------------------------------------
  /**
   * Gets the {@code LocalTime} part of this date-time.
   * <p>
   * This returns a {@code LocalTime} with the same hour, minute, second and nanosecond as this date-time.
   * 
   * @return the time part of this date-time, not null
   */
  @Override
  public LocalTime getTime() {

    return this.time;
  }

  /**
   * Gets the {@code LocalTime} part of this date-time.
   * <p>
   * This returns a {@code LocalTime} with the same hour, minute, second and nanosecond as this date-time.
   * 
   * @return the time part of this date-time, not null
   */
  public LocalTime toLocalTime() {

	    return this.time;
  }
  
  /**
   * Gets the hour-of-day field.
   * 
   * @return the hour-of-day, from 0 to 23
   */
  public int getHour() {

    return this.time.getHour();
  }

  /**
   * Gets the minute-of-hour field.
   * 
   * @return the minute-of-hour, from 0 to 59
   */
  public int getMinute() {

    return this.time.getMinute();
  }

  /**
   * Gets the second-of-minute field.
   * 
   * @return the second-of-minute, from 0 to 59
   */
  public int getSecond() {

    return this.time.getSecond();
  }

  /**
   * Gets the nano-of-second field.
   * 
   * @return the nano-of-second, from 0 to 999,999,999
   */
  public int getNano() {

    return this.time.getNano();
  }

  // -----------------------------------------------------------------------
  /**
   * Returns an adjusted date-time based on this date-time.
   * <p>
   * This adjusts the date-time according to the rules of the specified adjuster. A simple adjuster might
   * simply set the one of the fields, such as the year field. A more complex adjuster might set the date-time
   * to the last day of the month. A selection of common adjustments is provided in {@link DateTimeAdjusters}.
   * These include finding the "last day of the month" and "next Wednesday". The adjuster is responsible for
   * handling special cases, such as the varying lengths of month and leap years.
   * <p>
   * In addition, all principal classes implement the {@link WithAdjuster} interface, including this one. For
   * example, {@link LocalDate} implements the adjuster interface. As such, this code will compile and run:
   * 
   * <pre>
     *  dateTime.with(date);
     * </pre>
   * <p>
   * This instance is immutable and unaffected by this method call.
   * 
   * @param adjuster the adjuster to use, not null
   * @return a {@code LocalDateTime} based on this date-time with the adjustment made, not null
   * @throws DateTimeException if the adjustment cannot be made
   */
  @Override
  public LocalDateTime with(WithAdjuster adjuster) {

    if (adjuster instanceof LocalDate) {
      return with((LocalDate) adjuster, this.time);
    } else if (adjuster instanceof LocalTime) {
      return with(this.date, (LocalTime) adjuster);
    } else if (adjuster instanceof LocalDateTime) {
      return (LocalDateTime) adjuster;
    }
    return (LocalDateTime) adjuster.doWithAdjustment(this);
  }

  /**
   * Returns a copy of this date-time with the specified field altered.
   * <p>
   * This method returns a new date-time based on this date-time with a new value for the specified field.
   * This can be used to change any field, for example to set the year, month of day-of-month.
   * <p>
   * In some cases, changing the specified field can cause the resulting date-time to become invalid, such as
   * changing the month from January to February would make the day-of-month 31 invalid. In cases like this,
   * the field is responsible for resolving the date. Typically it will choose the previous valid date, which
   * would be the last valid day of February in this example.
   * <p>
   * This instance is immutable and unaffected by this method call.
   * 
   * @param field the field to set in the result, not null
   * @param newValue the new value of the field in the result
   * @return a {@code LocalDateTime} based on this date-time with the specified field set, not null
   * @throws DateTimeException if the value is invalid
   */
  @Override
  public LocalDateTime with(DateTimeField field, long newValue) {

    if (field instanceof ChronoField) {
      ChronoField f = (ChronoField) field;
      if (f.equals(ChronoField.INSTANT_SECONDS)) {
          return LocalDateTime.ofEpochSecond(newValue, 0, ZoneOffset.UTC);
      }
      if (f.isTimeField()) {
        return with(this.date, this.time.with(field, newValue));
      } else {
        return with(this.date.with(field, newValue), this.time);
      }
    }
    return field.doWith(this, newValue);
  }

  // -----------------------------------------------------------------------
  /**
   * Returns a copy of this {@code LocalDateTime} with the year altered. The time does not affect the
   * calculation and will be the same in the result. If the day-of-month is invalid for the year, it will be
   * changed to the last valid day of the month.
   * <p>
   * This instance is immutable and unaffected by this method call.
   * 
   * @param year the year to set in the result, from MIN_YEAR to MAX_YEAR
   * @return a {@code LocalDateTime} based on this date-time with the requested year, not null
   * @throws DateTimeException if the year value is invalid
   */
  public LocalDateTime withYear(int year) {

    return with(this.date.withYear(year), this.time);
  }

  /**
   * Returns a copy of this {@code LocalDateTime} with the month-of-year altered. The time does not affect the
   * calculation and will be the same in the result. If the day-of-month is invalid for the year, it will be
   * changed to the last valid day of the month.
   * <p>
   * This instance is immutable and unaffected by this method call.
   * 
   * @param month the month-of-year to set in the result, from 1 (January) to 12 (December)
   * @return a {@code LocalDateTime} based on this date-time with the requested month, not null
   * @throws DateTimeException if the month-of-year value is invalid
   */
  public LocalDateTime withMonth(int month) {

    return with(this.date.withMonth(month), this.time);
  }

  /**
   * Returns a copy of this {@code LocalDateTime} with the day-of-month altered. If the resulting
   * {@code LocalDateTime} is invalid, an exception is thrown. The time does not affect the calculation and
   * will be the same in the result.
   * <p>
   * This instance is immutable and unaffected by this method call.
   * 
   * @param dayOfMonth the day-of-month to set in the result, from 1 to 28-31
   * @return a {@code LocalDateTime} based on this date-time with the requested day, not null
   * @throws DateTimeException if the day-of-month value is invalid
   * @throws DateTimeException if the day-of-month is invalid for the month-year
   */
  public LocalDateTime withDayOfMonth(int dayOfMonth) {

    return with(this.date.withDayOfMonth(dayOfMonth), this.time);
  }

  /**
   * Returns a copy of this {@code LocalDateTime} with the day-of-year altered. If the resulting
   * {@code LocalDateTime} is invalid, an exception is thrown.
   * <p>
   * This instance is immutable and unaffected by this method call.
   * 
   * @param dayOfYear the day-of-year to set in the result, from 1 to 365-366
   * @return a {@code LocalDateTime} based on this date with the requested day, not null
   * @throws DateTimeException if the day-of-year value is invalid
   * @throws DateTimeException if the day-of-year is invalid for the year
   */
  public LocalDateTime withDayOfYear(int dayOfYear) {

    return with(this.date.withDayOfYear(dayOfYear), this.time);
  }

  // -----------------------------------------------------------------------
  /**
   * Returns a copy of this {@code LocalDateTime} with the hour-of-day value altered.
   * <p>
   * This instance is immutable and unaffected by this method call.
   * 
   * @param hour the hour-of-day to set in the result, from 0 to 23
   * @return a {@code LocalDateTime} based on this date-time with the requested hour, not null
   * @throws DateTimeException if the hour value is invalid
   */
  public LocalDateTime withHour(int hour) {

    LocalTime newTime = this.time.withHour(hour);
    return with(this.date, newTime);
  }

  /**
   * Returns a copy of this {@code LocalDateTime} with the minute-of-hour value altered.
   * <p>
   * This instance is immutable and unaffected by this method call.
   * 
   * @param minute the minute-of-hour to set in the result, from 0 to 59
   * @return a {@code LocalDateTime} based on this date-time with the requested minute, not null
   * @throws DateTimeException if the minute value is invalid
   */
  public LocalDateTime withMinute(int minute) {

    LocalTime newTime = this.time.withMinute(minute);
    return with(this.date, newTime);
  }

  /**
   * Returns a copy of this {@code LocalDateTime} with the second-of-minute value altered.
   * <p>
   * This instance is immutable and unaffected by this method call.
   * 
   * @param second the second-of-minute to set in the result, from 0 to 59
   * @return a {@code LocalDateTime} based on this date-time with the requested second, not null
   * @throws DateTimeException if the second value is invalid
   */
  public LocalDateTime withSecond(int second) {

    LocalTime newTime = this.time.withSecond(second);
    return with(this.date, newTime);
  }

  /**
   * Returns a copy of this {@code LocalDateTime} with the nano-of-second value altered.
   * <p>
   * This instance is immutable and unaffected by this method call.
   * 
   * @param nanoOfSecond the nano-of-second to set in the result, from 0 to 999,999,999
   * @return a {@code LocalDateTime} based on this date-time with the requested nanosecond, not null
   * @throws DateTimeException if the nano value is invalid
   */
  public LocalDateTime withNano(int nanoOfSecond) {

    LocalTime newTime = this.time.withNano(nanoOfSecond);
    return with(this.date, newTime);
  }

  // -----------------------------------------------------------------------
  /**
   * Returns a copy of this {@code LocalDateTime} with the time truncated.
   * <p>
   * Truncation returns a copy of the original date-time with fields smaller than the specified unit set to
   * zero. For example, truncating with the {@link ChronoUnit#MINUTES minutes} unit will set the
   * second-of-minute and nano-of-second field to zero.
   * <p>
   * Not all units are accepted. The {@link ChronoUnit#DAYS days} unit and time units with an exact duration
   * can be used, other units throw an exception.
   * <p>
   * This instance is immutable and unaffected by this method call.
   * 
   * @param unit the unit to truncate to, not null
   * @return a {@code LocalDateTime} based on this date-time with the time truncated, not null
   * @throws DateTimeException if unable to truncate
   */
  public LocalDateTime truncatedTo(PeriodUnit unit) {

    return with(this.date, this.time.truncatedTo(unit));
  }

  // -----------------------------------------------------------------------
  /**
   * Returns a copy of this date-time with the specified period added.
   * <p>
   * This method returns a new date-time based on this time with the specified period added. The adjuster is
   * typically {@link Period} but may be any other type implementing the
   * {@link java.time.calendrical.DateTime.PlusAdjuster} interface. The calculation is delegated to the
   * specified adjuster, which typically calls back to {@link #plus(long, PeriodUnit)}.
   * <p>
   * This instance is immutable and unaffected by this method call.
   * 
   * @param adjuster the adjuster to use, not null
   * @return a {@code LocalDateTime} based on this date-time with the addition made, not null
   * @throws DateTimeException if the addition cannot be made
   * @throws ArithmeticException if numeric overflow occurs
   */
  @Override
  public LocalDateTime plus(PlusAdjuster adjuster) {

    return (LocalDateTime) adjuster.doPlusAdjustment(this);
  }

  /**
   * Returns a copy of this date-time with the specified period added.
   * <p>
   * This method returns a new date-time based on this date-time with the specified period added. This can be
   * used to add any period that is defined by a unit, for example to add years, months or days. The unit is
   * responsible for the details of the calculation, including the resolution of any edge cases in the
   * calculation.
   * <p>
   * This instance is immutable and unaffected by this method call.
   * 
   * @param amountToAdd the amount of the unit to add to the result, may be negative
   * @param unit the unit of the period to add, not null
   * @return a {@code LocalDateTime} based on this date-time with the specified period added, not null
   * @throws DateTimeException if the unit cannot be added to this type
   */
  @Override
  public LocalDateTime plus(long amountToAdd, PeriodUnit unit) {

    if (unit instanceof ChronoUnit) {
      ChronoUnit f = (ChronoUnit) unit;
      switch (f) {
        case NANOS:
          return plusNanos(amountToAdd);
        case MICROS:
          return plusDays(amountToAdd / MICROS_PER_DAY).plusNanos((amountToAdd % MICROS_PER_DAY) * 1000);
        case MILLIS:
          return plusDays(amountToAdd / MILLIS_PER_DAY).plusNanos((amountToAdd % MILLIS_PER_DAY) * 1000000);
        case SECONDS:
          return plusSeconds(amountToAdd);
        case MINUTES:
          return plusMinutes(amountToAdd);
        case HOURS:
          return plusHours(amountToAdd);
        case HALF_DAYS:
          return plusDays(amountToAdd / 256).plusHours((amountToAdd % 256) * 12); // no overflow (256 is
                                                                                  // multiple of 2)
      }
      return with(this.date.plus(amountToAdd, unit), this.time);
    }
    return unit.doPlus(this, amountToAdd);
  }

  // -----------------------------------------------------------------------
  /**
   * Returns a copy of this {@code LocalDateTime} with the specified period in years added.
   * <p>
   * This method adds the specified amount to the years field in three steps:
   * <ol>
   * <li>Add the input years to the year field</li>
   * <li>Check if the resulting date would be invalid</li>
   * <li>Adjust the day-of-month to the last valid day if necessary</li>
   * </ol>
   * <p>
   * For example, 2008-02-29 (leap year) plus one year would result in the invalid date 2009-02-29 (standard
   * year). Instead of returning an invalid result, the last valid day of the month, 2009-02-28, is selected
   * instead.
   * <p>
   * This instance is immutable and unaffected by this method call.
   * 
   * @param years the years to add, may be negative
   * @return a {@code LocalDateTime} based on this date-time with the years added, not null
   * @throws DateTimeException if the result exceeds the supported date range
   */
  public LocalDateTime plusYears(long years) {

    LocalDate newDate = this.date.plusYears(years);
    return with(newDate, this.time);
  }

  /**
   * Returns a copy of this {@code LocalDateTime} with the specified period in months added.
   * <p>
   * This method adds the specified amount to the months field in three steps:
   * <ol>
   * <li>Add the input months to the month-of-year field</li>
   * <li>Check if the resulting date would be invalid</li>
   * <li>Adjust the day-of-month to the last valid day if necessary</li>
   * </ol>
   * <p>
   * For example, 2007-03-31 plus one month would result in the invalid date 2007-04-31. Instead of returning
   * an invalid result, the last valid day of the month, 2007-04-30, is selected instead.
   * <p>
   * This instance is immutable and unaffected by this method call.
   * 
   * @param months the months to add, may be negative
   * @return a {@code LocalDateTime} based on this date-time with the months added, not null
   * @throws DateTimeException if the result exceeds the supported date range
   */
  public LocalDateTime plusMonths(long months) {

    LocalDate newDate = this.date.plusMonths(months);
    return with(newDate, this.time);
  }

  /**
   * Returns a copy of this {@code LocalDateTime} with the specified period in weeks added.
   * <p>
   * This method adds the specified amount in weeks to the days field incrementing the month and year fields
   * as necessary to ensure the result remains valid. The result is only invalid if the maximum/minimum year
   * is exceeded.
   * <p>
   * For example, 2008-12-31 plus one week would result in 2009-01-07.
   * <p>
   * This instance is immutable and unaffected by this method call.
   * 
   * @param weeks the weeks to add, may be negative
   * @return a {@code LocalDateTime} based on this date-time with the weeks added, not null
   * @throws DateTimeException if the result exceeds the supported date range
   */
  public LocalDateTime plusWeeks(long weeks) {

    LocalDate newDate = this.date.plusWeeks(weeks);
    return with(newDate, this.time);
  }

  /**
   * Returns a copy of this {@code LocalDateTime} with the specified period in days added.
   * <p>
   * This method adds the specified amount to the days field incrementing the month and year fields as
   * necessary to ensure the result remains valid. The result is only invalid if the maximum/minimum year is
   * exceeded.
   * <p>
   * For example, 2008-12-31 plus one day would result in 2009-01-01.
   * <p>
   * This instance is immutable and unaffected by this method call.
   * 
   * @param days the days to add, may be negative
   * @return a {@code LocalDateTime} based on this date-time with the days added, not null
   * @throws DateTimeException if the result exceeds the supported date range
   */
  public LocalDateTime plusDays(long days) {

    LocalDate newDate = this.date.plusDays(days);
    return with(newDate, this.time);
  }

  // -----------------------------------------------------------------------
  /**
   * Returns a copy of this {@code LocalDateTime} with the specified period in hours added.
   * <p>
   * This instance is immutable and unaffected by this method call.
   * 
   * @param hours the hours to add, may be negative
   * @return a {@code LocalDateTime} based on this date-time with the hours added, not null
   * @throws DateTimeException if the result exceeds the supported date range
   */
  public LocalDateTime plusHours(long hours) {

    return plusWithOverflow(this.date, hours, 0, 0, 0, 1);
  }

  /**
   * Returns a copy of this {@code LocalDateTime} with the specified period in minutes added.
   * <p>
   * This instance is immutable and unaffected by this method call.
   * 
   * @param minutes the minutes to add, may be negative
   * @return a {@code LocalDateTime} based on this date-time with the minutes added, not null
   * @throws DateTimeException if the result exceeds the supported date range
   */
  public LocalDateTime plusMinutes(long minutes) {

    return plusWithOverflow(this.date, 0, minutes, 0, 0, 1);
  }

  /**
   * Returns a copy of this {@code LocalDateTime} with the specified period in seconds added.
   * <p>
   * This instance is immutable and unaffected by this method call.
   * 
   * @param seconds the seconds to add, may be negative
   * @return a {@code LocalDateTime} based on this date-time with the seconds added, not null
   * @throws DateTimeException if the result exceeds the supported date range
   */
  public LocalDateTime plusSeconds(long seconds) {

    return plusWithOverflow(this.date, 0, 0, seconds, 0, 1);
  }

  /**
   * Returns a copy of this {@code LocalDateTime} with the specified period in nanoseconds added.
   * <p>
   * This instance is immutable and unaffected by this method call.
   * 
   * @param nanos the nanos to add, may be negative
   * @return a {@code LocalDateTime} based on this date-time with the nanoseconds added, not null
   * @throws DateTimeException if the result exceeds the supported date range
   */
  public LocalDateTime plusNanos(long nanos) {

    return plusWithOverflow(this.date, 0, 0, 0, nanos, 1);
  }

  // -----------------------------------------------------------------------
  /**
   * Returns a copy of this date-time with the specified period subtracted.
   * <p>
   * This method returns a new date-time based on this time with the specified period subtracted. The adjuster
   * is typically {@link Period} but may be any other type implementing the
   * {@link java.time.calendrical.DateTime.MinusAdjuster} interface. The calculation is delegated to the
   * specified adjuster, which typically calls back to {@link #minus(long, PeriodUnit)}.
   * <p>
   * This instance is immutable and unaffected by this method call.
   * 
   * @param adjuster the adjuster to use, not null
   * @return a {@code LocalDateTime} based on this date-time with the subtraction made, not null
   * @throws DateTimeException if the subtraction cannot be made
   * @throws ArithmeticException if numeric overflow occurs
   */
  @Override
  public LocalDateTime minus(MinusAdjuster adjuster) {

    return (LocalDateTime) adjuster.doMinusAdjustment(this);
  }

  /**
   * Returns a copy of this date-time with the specified period subtracted.
   * <p>
   * This method returns a new date-time based on this date-time with the specified period subtracted. This
   * can be used to subtract any period that is defined by a unit, for example to subtract years, months or
   * days. The unit is responsible for the details of the calculation, including the resolution of any edge
   * cases in the calculation.
   * <p>
   * This instance is immutable and unaffected by this method call.
   * 
   * @param amountToSubtract the amount of the unit to subtract from the result, may be negative
   * @param unit the unit of the period to subtract, not null
   * @return a {@code LocalDateTime} based on this date-time with the specified period subtracted, not null
   * @throws DateTimeException if the unit cannot be added to this type
   */
  @Override
  public LocalDateTime minus(long amountToSubtract, PeriodUnit unit) {

    return (amountToSubtract == Long.MIN_VALUE ? plus(Long.MAX_VALUE, unit).plus(1, unit) : plus(-amountToSubtract,
        unit));
  }

  // -----------------------------------------------------------------------
  /**
   * Returns a copy of this {@code LocalDateTime} with the specified period in years subtracted.
   * <p>
   * This method subtracts the specified amount from the years field in three steps:
   * <ol>
   * <li>Subtract the input years from the year field</li>
   * <li>Check if the resulting date would be invalid</li>
   * <li>Adjust the day-of-month to the last valid day if necessary</li>
   * </ol>
   * <p>
   * For example, 2008-02-29 (leap year) minus one year would result in the invalid date 2009-02-29 (standard
   * year). Instead of returning an invalid result, the last valid day of the month, 2009-02-28, is selected
   * instead.
   * <p>
   * This instance is immutable and unaffected by this method call.
   * 
   * @param years the years to subtract, may be negative
   * @return a {@code LocalDateTime} based on this date-time with the years subtracted, not null
   * @throws DateTimeException if the result exceeds the supported date range
   */
  public LocalDateTime minusYears(long years) {

    return (years == Long.MIN_VALUE ? plusYears(Long.MAX_VALUE).plusYears(1) : plusYears(-years));
  }

  /**
   * Returns a copy of this {@code LocalDateTime} with the specified period in months subtracted.
   * <p>
   * This method subtracts the specified amount from the months field in three steps:
   * <ol>
   * <li>Subtract the input months from the month-of-year field</li>
   * <li>Check if the resulting date would be invalid</li>
   * <li>Adjust the day-of-month to the last valid day if necessary</li>
   * </ol>
   * <p>
   * For example, 2007-03-31 minus one month would result in the invalid date 2007-04-31. Instead of returning
   * an invalid result, the last valid day of the month, 2007-04-30, is selected instead.
   * <p>
   * This instance is immutable and unaffected by this method call.
   * 
   * @param months the months to subtract, may be negative
   * @return a {@code LocalDateTime} based on this date-time with the months subtracted, not null
   * @throws DateTimeException if the result exceeds the supported date range
   */
  public LocalDateTime minusMonths(long months) {

    return (months == Long.MIN_VALUE ? plusMonths(Long.MAX_VALUE).plusMonths(1) : plusMonths(-months));
  }

  /**
   * Returns a copy of this {@code LocalDateTime} with the specified period in weeks subtracted.
   * <p>
   * This method subtracts the specified amount in weeks from the days field decrementing the month and year
   * fields as necessary to ensure the result remains valid. The result is only invalid if the maximum/minimum
   * year is exceeded.
   * <p>
   * For example, 2009-01-07 minus one week would result in 2008-12-31.
   * <p>
   * This instance is immutable and unaffected by this method call.
   * 
   * @param weeks the weeks to subtract, may be negative
   * @return a {@code LocalDateTime} based on this date-time with the weeks subtracted, not null
   * @throws DateTimeException if the result exceeds the supported date range
   */
  public LocalDateTime minusWeeks(long weeks) {

    return (weeks == Long.MIN_VALUE ? plusWeeks(Long.MAX_VALUE).plusWeeks(1) : plusWeeks(-weeks));
  }

  /**
   * Returns a copy of this {@code LocalDateTime} with the specified period in days subtracted.
   * <p>
   * This method subtracts the specified amount from the days field incrementing the month and year fields as
   * necessary to ensure the result remains valid. The result is only invalid if the maximum/minimum year is
   * exceeded.
   * <p>
   * For example, 2009-01-01 minus one day would result in 2008-12-31.
   * <p>
   * This instance is immutable and unaffected by this method call.
   * 
   * @param days the days to subtract, may be negative
   * @return a {@code LocalDateTime} based on this date-time with the days subtracted, not null
   * @throws DateTimeException if the result exceeds the supported date range
   */
  public LocalDateTime minusDays(long days) {

    return (days == Long.MIN_VALUE ? plusDays(Long.MAX_VALUE).plusDays(1) : plusDays(-days));
  }

  // -----------------------------------------------------------------------
  /**
   * Returns a copy of this {@code LocalDateTime} with the specified period in hours subtracted.
   * <p>
   * This instance is immutable and unaffected by this method call.
   * 
   * @param hours the hours to subtract, may be negative
   * @return a {@code LocalDateTime} based on this date-time with the hours subtracted, not null
   * @throws DateTimeException if the result exceeds the supported date range
   */
  public LocalDateTime minusHours(long hours) {

    return plusWithOverflow(this.date, hours, 0, 0, 0, -1);
  }

  /**
   * Returns a copy of this {@code LocalDateTime} with the specified period in minutes subtracted.
   * <p>
   * This instance is immutable and unaffected by this method call.
   * 
   * @param minutes the minutes to subtract, may be negative
   * @return a {@code LocalDateTime} based on this date-time with the minutes subtracted, not null
   * @throws DateTimeException if the result exceeds the supported date range
   */
  public LocalDateTime minusMinutes(long minutes) {

    return plusWithOverflow(this.date, 0, minutes, 0, 0, -1);
  }

  /**
   * Returns a copy of this {@code LocalDateTime} with the specified period in seconds subtracted.
   * <p>
   * This instance is immutable and unaffected by this method call.
   * 
   * @param seconds the seconds to subtract, may be negative
   * @return a {@code LocalDateTime} based on this date-time with the seconds subtracted, not null
   * @throws DateTimeException if the result exceeds the supported date range
   */
  public LocalDateTime minusSeconds(long seconds) {

    return plusWithOverflow(this.date, 0, 0, seconds, 0, -1);
  }

  /**
   * Returns a copy of this {@code LocalDateTime} with the specified period in nanoseconds subtracted.
   * <p>
   * This instance is immutable and unaffected by this method call.
   * 
   * @param nanos the nanos to subtract, may be negative
   * @return a {@code LocalDateTime} based on this date-time with the nanoseconds subtracted, not null
   * @throws DateTimeException if the result exceeds the supported date range
   */
  public LocalDateTime minusNanos(long nanos) {

    return plusWithOverflow(this.date, 0, 0, 0, nanos, -1);
  }

  // -----------------------------------------------------------------------
  /**
   * Returns a copy of this {@code LocalDateTime} with the specified period added.
   * <p>
   * This instance is immutable and unaffected by this method call.
   * 
   * @param newDate the new date to base the calculation on, not null
   * @param hours the hours to add, may be negative
   * @param minutes the minutes to add, may be negative
   * @param seconds the seconds to add, may be negative
   * @param nanos the nanos to add, may be negative
   * @param sign the sign to determine add or subtract
   * @return the combined result, not null
   */
  private LocalDateTime plusWithOverflow(LocalDate newDate, long hours, long minutes, long seconds, long nanos, int sign) {

    // 9223372036854775808 long, 2147483648 int
    if ((hours | minutes | seconds | nanos) == 0) {
      return with(newDate, this.time);
    }
    long totDays = nanos / NANOS_PER_DAY + // max/24*60*60*1B
        seconds / SECONDS_PER_DAY + // max/24*60*60
        minutes / MINUTES_PER_DAY + // max/24*60
        hours / HOURS_PER_DAY; // max/24
    totDays *= sign; // total max*0.4237...
    long totNanos = nanos % NANOS_PER_DAY + // max 86400000000000
        (seconds % SECONDS_PER_DAY) * NANOS_PER_SECOND + // max 86400000000000
        (minutes % MINUTES_PER_DAY) * NANOS_PER_MINUTE + // max 86400000000000
        (hours % HOURS_PER_DAY) * NANOS_PER_HOUR; // max 86400000000000
    long curNoD = this.time.toNanoOfDay(); // max 86400000000000
    totNanos = totNanos * sign + curNoD; // total 432000000000000
    totDays += Jdk8Methods.floorDiv(totNanos, NANOS_PER_DAY);
    long newNoD = Jdk8Methods.floorMod(totNanos, NANOS_PER_DAY);
    LocalTime newTime = (newNoD == curNoD ? this.time : LocalTime.ofNanoOfDay(newNoD));
    return with(newDate.plusDays(totDays), newTime);
  }

  // -----------------------------------------------------------------------
  /**
   * Returns a zoned date-time formed from this date-time and the specified time-zone.
   * <p>
   * This creates a zoned date-time matching the input date-time as closely as possible. Time-zone rules, such
   * as daylight savings, mean that not every local date-time is valid for the specified zone, thus the local
   * date-time may be adjusted.
   * <p>
   * The local date-time is resolved to a single instant on the time-line. This is achieved by finding a valid
   * offset from UTC/Greenwich for the local date-time as defined by the {@link ZoneRules rules} of the zone
   * ID.
   * <p>
   * In most cases, there is only one valid offset for a local date-time. In the case of an overlap, where
   * clocks are set back, there are two valid offsets. This method uses the earlier offset typically
   * corresponding to "summer".
   * <p>
   * In the case of a gap, where clocks jump forward, there is no valid offset. Instead, the local date-time
   * is adjusted to be later by the length of the gap. For a typical one hour daylight savings change, the
   * local date-time will be moved one hour later into the offset typically corresponding to "summer".
   * <p>
   * To obtain the later offset during an overlap, call {@link ZonedDateTime#withLaterOffsetAtOverlap()} on
   * the result of this method. To throw an exception when there is a gap or overlap, use
   * {@link ZonedDateTime#ofStrict(LocalDateTime, ZoneOffset, ZoneId)}.
   * <p>
   * This instance is immutable and unaffected by this method call.
   * 
   * @param zone the time-zone to use, not null
   * @return the zoned date-time formed from this date-time, not null
   */
  @Override
  public ZonedDateTime atZone(ZoneId zone) {

    return ZonedDateTime.of(this, zone);
  }

  // -----------------------------------------------------------------------
  @Override
  public long periodUntil(DateTime endDateTime, PeriodUnit unit) {

    if (endDateTime instanceof LocalDateTime == false) {
      throw new DateTimeException("Unable to calculate period between objects of two different types");
    }
    LocalDateTime end = (LocalDateTime) endDateTime;
    if (unit instanceof ChronoUnit) {
      ChronoUnit f = (ChronoUnit) unit;
      if (f.isTimeUnit()) {
        long amount = this.date.daysUntil(end.date);
        switch (f) {
          case NANOS:
            amount = Jdk8Methods.safeMultiply(amount, NANOS_PER_DAY);
            break;
          case MICROS:
            amount = Jdk8Methods.safeMultiply(amount, MICROS_PER_DAY);
            break;
          case MILLIS:
            amount = Jdk8Methods.safeMultiply(amount, MILLIS_PER_DAY);
            break;
          case SECONDS:
            amount = Jdk8Methods.safeMultiply(amount, SECONDS_PER_DAY);
            break;
          case MINUTES:
            amount = Jdk8Methods.safeMultiply(amount, MINUTES_PER_DAY);
            break;
          case HOURS:
            amount = Jdk8Methods.safeMultiply(amount, HOURS_PER_DAY);
            break;
          case HALF_DAYS:
            amount = Jdk8Methods.safeMultiply(amount, 2);
            break;
        }
        return Jdk8Methods.safeAdd(amount, this.time.periodUntil(end.time, unit));
      }
      LocalDate endDate = end.date;
      if (end.time.isBefore(this.time)) {
        endDate = endDate.minusDays(1);
      }
      return this.date.periodUntil(end.date, unit);
    }
    return unit.between(this, endDateTime).getAmount();
  }

  // -----------------------------------------------------------------------
  /**
   * Compares this date-time to another date-time.
   * <p>
   * The comparison is primarily based on the date-time, from earliest to latest. It is
   * "consistent with equals", as defined by {@link Comparable}.
   * <p>
   * If all the date-times being compared are instances of {@code LocalDateTime}, then the comparison will be
   * entirely based on the date-time. If some dates being compared are in different chronologies, then the
   * chronology is also considered, see {@link ChronoLocalDateTime#compareTo}.
   * 
   * @param other the other date-time to compare to, not null
   * @return the comparator value, negative if less, positive if greater
   */
  @Override
  // override for Javadoc and performance
  public int compareTo(ChronoLocalDateTime<?> other) {

    if (other instanceof LocalDateTime) {
      return compareTo0((LocalDateTime) other);
    }
    return super.compareTo(other);
  }

  private int compareTo0(LocalDateTime other) {

    int cmp = this.date.compareTo0(other.getDate());
    if (cmp == 0) {
      cmp = this.time.compareTo(other.getTime());
    }
    return cmp;
  }

  /**
   * Checks if this date-time is after the specified date-time ignoring the chronology.
   * <p>
   * This method differs from the comparison in {@link #compareTo} in that it only compares the underlying
   * date-time and not the chronology. This allows dates in different calendar systems to be compared based on
   * the time-line position.
   * 
   * @param other the other date-time to compare to, not null
   * @return true if this is after the specified date-time
   */
  @Override
  // override for Javadoc and performance
  public boolean isAfter(ChronoLocalDateTime<?> other) {

    if (other instanceof LocalDateTime) {
      return compareTo0((LocalDateTime) other) > 0;
    }
    return super.isAfter(other);
  }

  /**
   * Checks if this date-time is before the specified date-time ignoring the chronology.
   * <p>
   * This method differs from the comparison in {@link #compareTo} in that it only compares the underlying
   * date-time and not the chronology. This allows dates in different calendar systems to be compared based on
   * the time-line position.
   * 
   * @param other the other date-time to compare to, not null
   * @return true if this is before the specified date-time
   */
  @Override
  // override for Javadoc and performance
  public boolean isBefore(ChronoLocalDateTime<?> other) {

    if (other instanceof LocalDateTime) {
      return compareTo0((LocalDateTime) other) < 0;
    }
    return super.isBefore(other);
  }

  // -----------------------------------------------------------------------
  /**
   * Checks if this date-time is equal to another date-time.
   * <p>
   * Compares this {@code LocalDateTime} with another ensuring that the date-time is the same. Only objects of
   * type {@code LocalDateTime} are compared, other types return false.
   * 
   * @param obj the object to check, null returns false
   * @return true if this is equal to the other date-time
   */
  @Override
  public boolean equals(Object obj) {

    if (this == obj) {
      return true;
    }
    if (obj instanceof LocalDateTime) {
      LocalDateTime other = (LocalDateTime) obj;
      return this.date.equals(other.date) && this.time.equals(other.time);
    }
    return false;
  }

  /**
   * A hash code for this date-time.
   * 
   * @return a suitable hash code
   */
  @Override
  public int hashCode() {

    return this.date.hashCode() ^ this.time.hashCode();
  }

  // -----------------------------------------------------------------------
  /**
   * Outputs this date-time as a {@code String}, such as {@code 2007-12-03T10:15:30}.
   * <p>
   * The output will be one of the following ISO-8601 formats:
   * <p>
   * <ul>
   * <li>{@code yyyy-MM-dd'T'HH:mm}</li>
   * <li>{@code yyyy-MM-dd'T'HH:mm:ss}</li>
   * <li>{@code yyyy-MM-dd'T'HH:mm:ss.SSS}</li>
   * <li>{@code yyyy-MM-dd'T'HH:mm:ss.SSSSSS}</li>
   * <li>{@code yyyy-MM-dd'T'HH:mm:ss.SSSSSSSSS}</li>
   * </ul>
   * <p>
   * The format used will be the shortest that outputs the full value of the time where the omitted parts are
   * implied to be zero.
   * 
   * @return a string representation of this date-time, not null
   */
  @Override
  public String toString() {

    return this.date.toString() + 'T' + this.time.toString();
  }

}
