package org.jboss.brmsbpmsuite.patching.client;

import org.osgi.framework.Version;

import java.io.IOException;
import java.util.Properties;

public class VersionCheckerProvider implements VersionChecker {
    private final Version finalRelease;
    private final Version finalPatch;

    private VersionCheckerProvider(Version finalRelease, Version finalPatch) {
        validate(finalRelease, finalPatch);
        this.finalRelease = finalRelease;
        this.finalPatch = finalPatch;
    }

    private void validate(Version finalRelease, Version finalPatch) {
        if (!isFinal(finalRelease)) {
            throw new InvalidVersionException("version-range-lower-limit is not a valid version!");
        }
        if (!isFinal(finalPatch)) {
            throw new InvalidVersionException("version-range-upper-limit is not a valid community version!");
        }
    }

    private boolean isFinal(Version version) {
        return version.getQualifier().toLowerCase().startsWith("final");
    }

    public boolean isValidForVersion(Version version) {
        return isFinal(version) &&
                version.compareTo(finalRelease) >= 0
                && version.compareTo(finalPatch) < 0;
    }

    public static VersionChecker getInstance() {
        Properties versionProperties = new Properties();
        try {
            versionProperties.load(ClassLoader.getSystemResourceAsStream("version.properties"));
        } catch (IOException e) {
            throw new InvalidVersionException("Couldn't find the applicable version definitions!");
        }

        return createInstance(versionProperties);
    }

    static VersionChecker createInstance(Properties versionProperties) {
        Version finalRelease = versionFromProperties(versionProperties, "version-range-lower-limit");
        Version finalPatch = versionFromProperties(versionProperties, "version-range-upper-limit");
        return new VersionCheckerProvider(finalRelease, finalPatch);
    }

    static Version versionFromProperties(Properties versionProperties, String versionName) {
        if (versionProperties.containsKey(versionName)) {
            return Version.parseVersion(versionProperties.getProperty(versionName));
        }
        throw new InvalidVersionException("The property to create the '" + versionName + "' version doesn't exist!");
    }

}
