/*
Copyright (C) 2016 Red Hat, Inc.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package registration

import (
	"fmt"

	"github.com/docker/machine/libmachine/host"
	"github.com/docker/machine/libmachine/log"
	"github.com/docker/machine/libmachine/provision"
)

type RegistrationParameters struct {
	Username               string
	Password               string
	GetUsernameInteractive func(message string) string
	GetPasswordInteractive func(message string) string
}

// Detect supported Registrator
func NeedRegistration(host *host.Host) (bool, error) {
	commander := provision.GenericSSHCommander{Driver: host.Driver}
	_, supportRegistration, err := DetectRegistrator(commander)
	return supportRegistration, err
}

// Register host VM
func RegisterHostVM(host *host.Host, param *RegistrationParameters) (bool, error) {
	commander := provision.GenericSSHCommander{Driver: host.Driver}
	registrator, supportRegistration, err := DetectRegistrator(commander)
	if !supportRegistration {
		log.Debug("Distribution doesn't support registration")
	}

	if err != nil && err != ErrDetectionFailed {
		return supportRegistration, err
	}

	if registrator != nil {
		fmt.Println("Registering machine using subscription-manager")
		if err := registrator.Register(param); err != nil {
			return supportRegistration, err
		}
	}
	return supportRegistration, nil
}

// Unregister host VM
func UnregisterHostVM(host *host.Host, param *RegistrationParameters) error {
	commander := provision.GenericSSHCommander{Driver: host.Driver}
	registrator, supportUnregistration, err := DetectRegistrator(commander)

	if !supportUnregistration {
		log.Debug("Distribution doesn't support unregistration")
	}

	if err != nil && err != ErrDetectionFailed {
		return err
	}

	if registrator != nil {
		fmt.Println("Unregistering machine")
		if err := registrator.Unregister(param); err != nil {
			return err
		}
	}
	return nil
}
