/************************************************************************
**
**	idt_except.s - functions to initialize R3000 and
**		       prepare for Stand alone exception handling
**
**	Copyright 1991 Integrated Device Technology, Inc.
**	All Rights Reserved
**
**************************************************************************/


#include "iregdef.h"
#include "idtcpu.h"  
#include "idtmon.h"  
#include "excepthdr.h"  
#include "saunder.h"
#include "setjmp.h"


	.data
	.globl	fp_int_flag
fp_int_flag:
	.word	0
		.text

#define EXCP_STACK_SIZE (90*4)  	/* BIG stack frame for environ save */


/**************************************************************************
**
** enable_int(mask) - enables interrupts - mask is positioned so it only 
**			needs to be or'ed into the status reg. This
**			also does some other things !!!! caution should
**			be used if invoking this while in the middle
**			of a debugging session where the client may have
**			nested interrupts.
**
****************************************************************************/
FRAME(enable_int,sp,0,ra)
	.set	noreorder
	mfc0	t0,C0_SR
	or	a0,1
	or	t0,a0
	mtc0	t0,C0_SR
	j	ra
	nop
	.set	reorder
ENDFRAME(enable_int)





/***************************************************************************  
**
**	disable_int(mask) - disable the interrupt - mask is the complement
**			    of the bits to be cleared - i.e. to clear ext int
**			    5 the mask would be - 0xffff7fff
**
****************************************************************************/
FRAME(disable_int,sp,0,ra)
	.set	noreorder
	mfc0	t0,C0_SR
	nop
	and	t0,a0
	mtc0	t0,C0_SR
	j	ra
	nop
	.set	reorder
ENDFRAME(disable_int)

/**************************************************************************
**
**	init_exc_vecs() - moves the exception code into the addresses
**			  reserved for exception vectors
**
**	UTLB Miss exception vector at address 0x80000000
**
**	General exception vector at address 0x80000080
**
**	RESET exception vector is at address 0xbfc00000
**
***************************************************************************/

#define INITEXCFRM ((2*4)+4)		/* ra + 2 arguments */
FRAME(init_exc_vecs,sp,0,ra)
	subu	sp, INITEXCFRM		/* make local stack frame */
	sw	ra, INITEXCFRM-4(sp)	/* save return */

	.set	noreorder
/* - - move UTLB MISS & external exception handlers to correct vectors - - */
	la	t1,exc_utlb_code
	la	t2,exc_norm_code
	li	t3,UT_VEC
	li	t4,E_VEC
	li	t5,VEC_CODE_LENGTH
1:
	lw	t6,0(t1)
	lw	t7,0(t2)
	sw	t6,0(t3)
	sw	t7,0(t4)
	addiu	t1,4
	addiu	t3,4
	addiu	t4,4
	subu	t5,4
	bne	t5,zero,1b
	addiu	t2,4
	li	a0,UT_VEC
	jal	clear_Icache
	li	a1,VEC_CODE_LENGTH
	li	a0,E_VEC
	jal	clear_Icache
	li	a1,VEC_CODE_LENGTH

	lw	ra, INITEXCFRM-4(sp)	/* restore ra */
	addu	sp, INITEXCFRM		/* reset stack */
	j	ra
	nop
	.set	reorder
ENDFRAME(init_exc_vecs)



/************************************************************************
**
** the following sections of code are copied to the vector area
**	at location 0x80000000 (utlb miss) and location 0x80000080
**	(general exception) by init_exc_vecs 
**
*************************************************************************/

	.set	noreorder

FRAME(exc_norm_code,sp,0,ra)
	la	k0, exception			/* generic external int hndlr */
	j	k0
	subu	sp, EXCP_STACK_SIZE		/* set up local stack frame */
ENDFRAME(exc_norm_code)
	.set	reorder

/***********************************************************************
**
**	UTLB MISS 
**
**  if except_ptr != 0 
**		   use longjmp() to user  exception handling routine 
**  else 
**	jump into IDT/sim exception handling routine
**
**************************************************************************/

FRAME(exc_utlb_code,sp,0,ra)

	jal	get_except_ptr;		/* user UTLB exception hndlr? */
	beqz	v0, 1f			/* NO, go to monitor */
	jal	mem_exc_hdlr 		/* handle it; use longjmp out */
1:
	la	k0, (R_VEC+((32)*8))	/* let Monitor handle it */
	j	k0

ENDFRAME(exc_utlb_code)

/*
**  set_fp_flag()
*/
FRAME(set_fp_flag,sp,0,ra)
	li	t1,1
	sw	t1,fp_int_flag
	j	ra	
ENDFRAME(set_fp_flag)


/***********************************************************************
**
** common exception handling code
**
**  This routine receives control for all external interrupts except
**  UTLB miss which is trapped separately.  When entered, an exception
**  machine state frame has been reserved on the current stack; reg k0
**  has already been saved in the frame.
**
**  save environment
**  call the C language interrupt handler    
**
**  NOTE: s0-s7, f20-f30 not saved per ANSI C save standards
**
*************************************************************************/

FRAME(exception,sp, 0, ra)

/* - - 	save machine environment on exception stack - - */

	.set	noreorder
	sw	v0, R_V0*4(sp)
	sw	v1, R_V1*4(sp)
	mfc0	k0, C0_EPC		/* exception program counter */
	.set	noat
	sw	AT, R_AT*4(sp)
	.set	at
	sw	k0, R_EPC*4(sp)
	sw	a0, R_A0*4(sp)
	sw	a1, R_A1*4(sp)
	sw	a2, R_A2*4(sp)
	sw	a3, R_A3*4(sp)
	sw	t0, R_T0*4(sp)
	sw	t1, R_T1*4(sp)
	mfc0	a0, C0_CAUSE		/* Cause Register from CP0 */
	li	t0, CAUSE_EXCMASK	/* is it external interrupt? */
	and	t0, a0
	sw	t2, R_T2*4(sp)
	sw	t3, R_T3*4(sp)
	sw	t4, R_T4*4(sp)
	sw	t5, R_T5*4(sp)
	sw	t6, R_T6*4(sp)
	sw	t7, R_T7*4(sp)
	mflo	k0
	sw	t8, R_T8*4(sp)
	sw	k0, R_MDLO*4(sp)
	sw	t9, R_T9*4(sp)
	mfhi	k0
	sw	gp, R_GP*4(sp)
	sw	fp, R_FP*4(sp)
	sw	ra, R_RA*4(sp)

/***********************************************************************
** 
**  If floating point coprocessor present (cp1)
** 
**	DO NOT save Floating Point Control/Status Register; 
**		the FP Interrupt Handler will change and set it
**
**   SAVE:  only those floating point registers used by your interrupt
**	    handler - do NOT waste instructions (time) on unnecessary
**	    saves.  Sample code is provided. 
**
**	swc1	fp0, R_F0*4(sp)
**	swc1	fp1, R_F1*4(sp)
**		...
**	swc1	fp19,R_F19*4(sp)	 
**
***********************************************************************/

	sw	zero,fp_int_flag	/* MUST clear before int hndlr */

	bnez	t0, other_excp		/* everything other than external */
	sw	k0, R_MDHI*4(sp)

/***********************************************************************
**
**    EXTERNAL INTERRUPT:
**
**		a0 =>  CAUSE Register 
**		a1 =>  STATUS  Register
**
**		 save any of the registers you use s0..s7 & f20..f30
**
************************************************************************/
	jal	extern_int		/* process external interrupts */
	mfc0	a1, C0_SR		/* get STATUS Register	*/ 

/* - - return from C Interrupt Handler - - restore machine environment - - */

/***********************************************************************
** 
**  If floating point coprocessor present (cp1)
**
**  DO NOT restore the Floating Point Control/Status Register;
**	   the FP Interrupt Handler will changed it
**
**   RESTORE:only those floating point registers used by your interrupt
**	    handler - do NOT waste instructions (time) on unnecessary
**	    restores.  Sample code:
**
**	swc1	fp0, R_F0*4(sp)
**	swc1	fp1, R_F1*4(sp)
**		...
**	swc1	fp19,R_F19*4(sp)	
**
***********************************************************************/

	lw	k0,fp_int_flag		/* Floating Point Int serviced? */
	nop
	beq	k0,zero,2f		/* NOT FP Int, don't bump PC   */
	nop

	mfc0	v0, C0_CAUSE		/* occurred in Branch Delay Slot? */	
	nop
	and	v0, v0,CAUSE_BD 
	beq	v0, zero, 3f			
	nop			
	jal	sae_errmsg 		/* yes, fatal error, msg & abort */ 
3:   
	lw	k0, R_EPC*4(sp)		/* no, Ok to go on bump PC to */
	nop
	addu	k0,4 			/* jump past fp instruction */
	sw	k0, R_EPC*4(sp)         


/* - - 	restore machine environment from exception stack - - 
**
**	C0_EPC and R_FEIR are read only don't bother restoring 
**
*/
2:
	lw	k0, R_MDLO*4(sp)
	.set	noat
	lw	AT, R_AT*4(sp)
	.set	at
	lw	v0, R_V0*4(sp)
	mtlo	k0
	lw	v1, R_V1*4(sp)
	lw	a0, R_A0*4(sp)
	lw	k0, R_MDHI*4(sp)	
	lw	a1, R_A1*4(sp)
	lw	a2, R_A2*4(sp)
	mthi	k0
	lw	a3, R_A3*4(sp)
	lw	t0, R_T0*4(sp)
	lw	t1, R_T1*4(sp)
	lw	t2, R_T2*4(sp)
	lw	t3, R_T3*4(sp)
	lw	t4, R_T4*4(sp)
	lw	t5, R_T5*4(sp)
	lw	t6, R_T6*4(sp)
	lw	t7, R_T7*4(sp)
	lw	t8, R_T8*4(sp)
	lw	t9, R_T9*4(sp)
	lw	gp, R_GP*4(sp)
	lw	fp, R_FP*4(sp)
	lw	ra, R_RA*4(sp)

	lw	k0, R_EPC*4(sp)
	addu	sp, EXCP_STACK_SIZE	/* reset exception stack frame */
	j	k0
	rfe
	.set	reorder


/***********************************************************************
**
**	Any exception other than UTLB MISS or External
**	If except_ptr not NULL AND cause ExcCode not Breakpoint
**	  handle memory probe exception (longjmp out)
**
**	  Restore state at time of interrupt
**	  Cause still in a0
**	  Jump into PROM Monitor at hard offset for handling
**	     or into user's exception handler if no IDT/sim
**
************************************************************************/

other_excp:

	jal	get_except_ptr;		/* any mem exceptions? */
	beqz	v0, 1f
	and	t1, a0, EXCMASK		/* could it be breakpoint  */
	beq	t1, EXC_BREAK, 1f	/* and nothing else?	   */
	jal	mem_exc_hdlr 		/* handle it; use longjmp out */

/* - - 	restore machine environment from exception stack - - 
**
**	C0_EPC and R_FEIR are read only don't bother restoring 
**
*/
1:
	.set	noreorder
	.set	noat
	lw	AT, R_AT*4(sp)
	.set	at
	lw	v0, R_V0*4(sp)
	lw	v1, R_V1*4(sp)
	lw	a0, R_A0*4(sp)
	lw	a1, R_A1*4(sp)
	lw	a2, R_A2*4(sp)
	lw	a3, R_A3*4(sp)
	lw	t0, R_T0*4(sp)
	lw	t1, R_T1*4(sp)
	lw	t2, R_T2*4(sp)
	lw	t3, R_T3*4(sp)
	lw	t4, R_T4*4(sp)
	lw	t5, R_T5*4(sp)
	lw	t6, R_T6*4(sp)
	lw	t7, R_T7*4(sp)
	lw	t8, R_T8*4(sp)
	lw	t9, R_T9*4(sp)
	lw	gp, R_GP*4(sp)
	lw	fp, R_FP*4(sp)
	lw	ra, R_RA*4(sp)

/*************************************************************************
**
**  if no IDT/sim
**		   la k0, address of exception handling routine 
**  else 
**	jump into IDT/sim exception handling routine
**
**		   machine state has been restored 
**
**************************************************************************/

	la	k0, (R_VEC+((48)*8))
	j	k0
	addu	sp, EXCP_STACK_SIZE	/* reset stack pointer */
	.set	reorder
ENDFRAME(exception)


/***********************************************************************
**
** setjmp(jmp_buf) -- save current context for non-local goto's
** return 0
**
************************************************************************/
FRAME(setjmp,sp,0,ra)
	sw	ra,JB_PC*4(a0)
	sw	sp,JB_SP*4(a0)
	sw	fp,JB_FP*4(a0)
	sw	s0,JB_S0*4(a0)
	sw	s1,JB_S1*4(a0)
	sw	s2,JB_S2*4(a0)
	sw	s3,JB_S3*4(a0)
	sw	s4,JB_S4*4(a0)
	sw	s5,JB_S5*4(a0)
	sw	s6,JB_S6*4(a0)
	sw	s7,JB_S7*4(a0)
	move	v0,zero
	j	ra
	ENDFRAME(setjmp)
 

/***********************************************************************
**
** longjmp(jmp_buf, rval)
**
************************************************************************/
FRAME(longjmp,sp,0,ra)
	lw	ra,JB_PC*4(a0)
	lw	sp,JB_SP*4(a0)
	lw	fp,JB_FP*4(a0)
	lw	s0,JB_S0*4(a0)
	lw	s1,JB_S1*4(a0)
	lw	s2,JB_S2*4(a0)
	lw	s3,JB_S3*4(a0)
	lw	s4,JB_S4*4(a0)
	lw	s5,JB_S5*4(a0)
	lw	s6,JB_S6*4(a0)
	lw	s7,JB_S7*4(a0)
	move	v0,a1
	j	ra
	ENDFRAME(longjmp)

