/* 
 *  Unix SMB/CIFS implementation.
 *  error mapping functions
 *  Copyright (C) Andrew Tridgell 2001
 *  Copyright (C) Andrew Bartlett 2001
 *  Copyright (C) Tim Potter 2000
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "includes.h"

/* This map was extracted by the ERRMAPEXTRACT smbtorture command. 
   The setup was a Samba HEAD (2002-01-03) PDC and an Win2k member 
   workstation.  The PDC was modified (by using the 'name_to_nt_status'
   authentication module) to convert the username (in hex) into the
   corresponding NTSTATUS error return. 

   By opening two nbt sessions to the Win2k workstation, one negotiating
   DOS and one negotiating NT errors it was possible to extract the
   error mapping.  (Because the server only supplies NT errors, the 
   NT4 workstation had to use its own error tables to convert these
   to dos errors). 

   Some errors show up as 'squashed' because the NT error connection
   got back a different error to the one it sent, so a mapping could
   not be determined (a guess has been made in this case, to map the
   error as squashed).  This is done mainly to prevent users from getting
   NT_STATUS_WRONG_PASSWORD and NT_STATUS_NO_SUCH_USER errors (they get
   NT_STATUS_LOGON_FAILURE instead.

   -- abartlet (2002-01-03)
*/

/* NT status -> dos error map */
static const struct {
	uint8 dos_class;
	uint32 dos_code;
	NTSTATUS ntstatus;
} ntstatus_to_dos_map[] = {
	{ERRDOS,	ERRgeneral,	NT_STATUS_UNSUCCESSFUL},
	{ERRDOS,	ERRbadfunc,	NT_STATUS_NOT_IMPLEMENTED},
	{ERRDOS,	87,	NT_STATUS_INVALID_INFO_CLASS},
	{ERRDOS,	24,	NT_STATUS_INFO_LENGTH_MISMATCH},
	{ERRHRD,	ERRgeneral,	NT_STATUS_ACCESS_VIOLATION},
	{ERRHRD,	ERRgeneral,	NT_STATUS_IN_PAGE_ERROR},
	{ERRHRD,	ERRgeneral,	NT_STATUS_PAGEFILE_QUOTA},
	{ERRDOS,	ERRbadfid,	NT_STATUS_INVALID_HANDLE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_BAD_INITIAL_STACK},
	{ERRDOS,	193,	NT_STATUS_BAD_INITIAL_PC},
	{ERRDOS,	87,	NT_STATUS_INVALID_CID},
	{ERRHRD,	ERRgeneral,	NT_STATUS_TIMER_NOT_CANCELED},
	{ERRDOS,	87,	NT_STATUS_INVALID_PARAMETER},
	{ERRDOS,	ERRbadfile,	NT_STATUS_NO_SUCH_DEVICE},
	{ERRDOS,	ERRbadfile,	NT_STATUS_NO_SUCH_FILE},
	{ERRDOS,	ERRbadfunc,	NT_STATUS_INVALID_DEVICE_REQUEST},
	{ERRDOS,	38,	NT_STATUS_END_OF_FILE},
	{ERRDOS,	34,	NT_STATUS_WRONG_VOLUME},
	{ERRDOS,	21,	NT_STATUS_NO_MEDIA_IN_DEVICE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_UNRECOGNIZED_MEDIA},
	{ERRDOS,	27,	NT_STATUS_NONEXISTENT_SECTOR},
/** Session setup succeeded.  This shouldn't happen...*/
/** Session setup succeeded.  This shouldn't happen...*/
/** NT error on DOS connection! (NT_STATUS_OK) */
/*	{ This NT error code was 'sqashed'
	 from NT_STATUS_MORE_PROCESSING_REQUIRED to NT_STATUS_OK 
	 during the session setup }
*/
#if 0
	{SUCCESS,	0,	NT_STATUS_OK},
#endif
	{ERRDOS,	ERRnomem,	NT_STATUS_NO_MEMORY},
	{ERRDOS,	487,	NT_STATUS_CONFLICTING_ADDRESSES},
	{ERRDOS,	487,	NT_STATUS_NOT_MAPPED_VIEW},
	{ERRDOS,	87,	NT_STATUS_UNABLE_TO_FREE_VM},
	{ERRDOS,	87,	NT_STATUS_UNABLE_TO_DELETE_SECTION},
	{ERRDOS,	2142,	NT_STATUS_INVALID_SYSTEM_SERVICE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_ILLEGAL_INSTRUCTION},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_INVALID_LOCK_SEQUENCE},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_INVALID_VIEW_SIZE},
	{ERRDOS,	193,	NT_STATUS_INVALID_FILE_FOR_SECTION},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_ALREADY_COMMITTED},
/*	{ This NT error code was 'sqashed'
	 from NT_STATUS_ACCESS_DENIED to NT_STATUS_TRUSTED_RELATIONSHIP_FAILURE 
	 during the session setup }
*/
	{ERRDOS,	ERRnoaccess,	NT_STATUS_ACCESS_DENIED},
	{ERRDOS,	111,	NT_STATUS_BUFFER_TOO_SMALL},
/*
 * Not an official error, as only bit 0x80000000, not bits 0xC0000000 are set.
 */
	{ERRDOS,	ERRmoredata,	STATUS_BUFFER_OVERFLOW},
	{ERRDOS,	ERRbadfid,	NT_STATUS_OBJECT_TYPE_MISMATCH},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NONCONTINUABLE_EXCEPTION},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_DISPOSITION},
	{ERRHRD,	ERRgeneral,	NT_STATUS_UNWIND},
	{ERRHRD,	ERRgeneral,	NT_STATUS_BAD_STACK},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_UNWIND_TARGET},
	{ERRDOS,	158,	NT_STATUS_NOT_LOCKED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_PARITY_ERROR},
	{ERRDOS,	487,	NT_STATUS_UNABLE_TO_DECOMMIT_VM},
	{ERRDOS,	487,	NT_STATUS_NOT_COMMITTED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_PORT_ATTRIBUTES},
	{ERRHRD,	ERRgeneral,	NT_STATUS_PORT_MESSAGE_TOO_LONG},
	{ERRDOS,	87,	NT_STATUS_INVALID_PARAMETER_MIX},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_QUOTA_LOWER},
	{ERRHRD,	ERRgeneral,	NT_STATUS_DISK_CORRUPT_ERROR},
	{ERRDOS,	ERRinvalidname,	NT_STATUS_OBJECT_NAME_INVALID},
	{ERRDOS,	ERRbadfile,	NT_STATUS_OBJECT_NAME_NOT_FOUND},
	{ERRDOS,	183,	NT_STATUS_OBJECT_NAME_COLLISION},
	{ERRHRD,	ERRgeneral,	NT_STATUS_HANDLE_NOT_WAITABLE},
	{ERRDOS,	ERRbadfid,	NT_STATUS_PORT_DISCONNECTED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_DEVICE_ALREADY_ATTACHED},
	{ERRDOS,	161,	NT_STATUS_OBJECT_PATH_INVALID},
	{ERRDOS,	ERRbadpath,	NT_STATUS_OBJECT_PATH_NOT_FOUND},
	{ERRDOS,	161,	NT_STATUS_OBJECT_PATH_SYNTAX_BAD},
	{ERRHRD,	ERRgeneral,	NT_STATUS_DATA_OVERRUN},
	{ERRHRD,	ERRgeneral,	NT_STATUS_DATA_LATE_ERROR},
	{ERRDOS,	23,	NT_STATUS_DATA_ERROR},
	{ERRDOS,	23,	NT_STATUS_CRC_ERROR},
	{ERRDOS,	ERRnomem,	NT_STATUS_SECTION_TOO_BIG},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_PORT_CONNECTION_REFUSED},
	{ERRDOS,	ERRbadfid,	NT_STATUS_INVALID_PORT_HANDLE},
	{ERRDOS,	ERRbadshare,	NT_STATUS_SHARING_VIOLATION},
	{ERRHRD,	ERRgeneral,	NT_STATUS_QUOTA_EXCEEDED},
	{ERRDOS,	87,	NT_STATUS_INVALID_PAGE_PROTECTION},
	{ERRDOS,	288,	NT_STATUS_MUTANT_NOT_OWNED},
	{ERRDOS,	298,	NT_STATUS_SEMAPHORE_LIMIT_EXCEEDED},
	{ERRDOS,	87,	NT_STATUS_PORT_ALREADY_SET},
	{ERRDOS,	87,	NT_STATUS_SECTION_NOT_IMAGE},
	{ERRDOS,	156,	NT_STATUS_SUSPEND_COUNT_EXCEEDED},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_THREAD_IS_TERMINATING},
	{ERRDOS,	87,	NT_STATUS_BAD_WORKING_SET_LIMIT},
	{ERRDOS,	87,	NT_STATUS_INCOMPATIBLE_FILE_MAP},
	{ERRDOS,	87,	NT_STATUS_SECTION_PROTECTION},
	{ERRDOS,	282,	NT_STATUS_EAS_NOT_SUPPORTED},
	{ERRDOS,	255,	NT_STATUS_EA_TOO_LARGE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NONEXISTENT_EA_ENTRY},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NO_EAS_ON_FILE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_EA_CORRUPT_ERROR},
	{ERRDOS,	ERRlock,	NT_STATUS_FILE_LOCK_CONFLICT},
	{ERRDOS,	ERRlock,	NT_STATUS_LOCK_NOT_GRANTED},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_DELETE_PENDING},
	{ERRDOS,	ERRunsup,	NT_STATUS_CTL_FILE_NOT_SUPPORTED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_UNKNOWN_REVISION},
	{ERRHRD,	ERRgeneral,	NT_STATUS_REVISION_MISMATCH},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_OWNER},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_PRIMARY_GROUP},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NO_IMPERSONATION_TOKEN},
	{ERRHRD,	ERRgeneral,	NT_STATUS_CANT_DISABLE_MANDATORY},
	{ERRDOS,	2215,	NT_STATUS_NO_LOGON_SERVERS},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NO_SUCH_LOGON_SESSION},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NO_SUCH_PRIVILEGE},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_PRIVILEGE_NOT_HELD},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_ACCOUNT_NAME},
	{ERRHRD,	ERRgeneral,	NT_STATUS_USER_EXISTS},
/*	{ This NT error code was 'sqashed'
	 from NT_STATUS_NO_SUCH_USER to NT_STATUS_LOGON_FAILURE 
	 during the session setup }
*/
	{ERRDOS,	ERRnoaccess,	NT_STATUS_NO_SUCH_USER},
	{ERRHRD,	ERRgeneral,	NT_STATUS_GROUP_EXISTS},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NO_SUCH_GROUP},
	{ERRHRD,	ERRgeneral,	NT_STATUS_MEMBER_IN_GROUP},
	{ERRHRD,	ERRgeneral,	NT_STATUS_MEMBER_NOT_IN_GROUP},
	{ERRHRD,	ERRgeneral,	NT_STATUS_LAST_ADMIN},
/*	{ This NT error code was 'sqashed'
	 from NT_STATUS_WRONG_PASSWORD to NT_STATUS_LOGON_FAILURE 
	 during the session setup }
*/
	{ERRSRV,	ERRbadpw,	NT_STATUS_WRONG_PASSWORD},
	{ERRHRD,	ERRgeneral,	NT_STATUS_ILL_FORMED_PASSWORD},
	{ERRHRD,	ERRgeneral,	NT_STATUS_PASSWORD_RESTRICTION},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_LOGON_FAILURE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_ACCOUNT_RESTRICTION},
	{ERRSRV,	2241,	NT_STATUS_INVALID_LOGON_HOURS},
	{ERRSRV,	2240,	NT_STATUS_INVALID_WORKSTATION},
	{ERRSRV,	2242,	NT_STATUS_PASSWORD_EXPIRED},
	{ERRSRV,	2239,	NT_STATUS_ACCOUNT_DISABLED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NONE_MAPPED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_TOO_MANY_LUIDS_REQUESTED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_LUIDS_EXHAUSTED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_SUB_AUTHORITY},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_ACL},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_SID},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_SECURITY_DESCR},
	{ERRDOS,	127,	NT_STATUS_PROCEDURE_NOT_FOUND},
	{ERRDOS,	193,	NT_STATUS_INVALID_IMAGE_FORMAT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NO_TOKEN},
	{ERRHRD,	ERRgeneral,	NT_STATUS_BAD_INHERITANCE_ACL},
	{ERRDOS,	158,	NT_STATUS_RANGE_NOT_LOCKED},
	{ERRDOS,	112,	NT_STATUS_DISK_FULL},
	{ERRHRD,	ERRgeneral,	NT_STATUS_SERVER_DISABLED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_SERVER_NOT_DISABLED},
	{ERRDOS,	68,	NT_STATUS_TOO_MANY_GUIDS_REQUESTED},
	{ERRDOS,	259,	NT_STATUS_GUIDS_EXHAUSTED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_ID_AUTHORITY},
	{ERRDOS,	259,	NT_STATUS_AGENTS_EXHAUSTED},
	{ERRDOS,	154,	NT_STATUS_INVALID_VOLUME_LABEL},
	{ERRDOS,	ERRres,	NT_STATUS_SECTION_NOT_EXTENDED},
	{ERRDOS,	487,	NT_STATUS_NOT_MAPPED_DATA},
	{ERRHRD,	ERRgeneral,	NT_STATUS_RESOURCE_DATA_NOT_FOUND},
	{ERRHRD,	ERRgeneral,	NT_STATUS_RESOURCE_TYPE_NOT_FOUND},
	{ERRHRD,	ERRgeneral,	NT_STATUS_RESOURCE_NAME_NOT_FOUND},
	{ERRHRD,	ERRgeneral,	NT_STATUS_ARRAY_BOUNDS_EXCEEDED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_FLOAT_DENORMAL_OPERAND},
	{ERRHRD,	ERRgeneral,	NT_STATUS_FLOAT_DIVIDE_BY_ZERO},
	{ERRHRD,	ERRgeneral,	NT_STATUS_FLOAT_INEXACT_RESULT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_FLOAT_INVALID_OPERATION},
	{ERRHRD,	ERRgeneral,	NT_STATUS_FLOAT_OVERFLOW},
	{ERRHRD,	ERRgeneral,	NT_STATUS_FLOAT_STACK_CHECK},
	{ERRHRD,	ERRgeneral,	NT_STATUS_FLOAT_UNDERFLOW},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INTEGER_DIVIDE_BY_ZERO},
	{ERRDOS,	534,	NT_STATUS_INTEGER_OVERFLOW},
	{ERRHRD,	ERRgeneral,	NT_STATUS_PRIVILEGED_INSTRUCTION},
	{ERRDOS,	ERRnomem,	NT_STATUS_TOO_MANY_PAGING_FILES},
	{ERRHRD,	ERRgeneral,	NT_STATUS_FILE_INVALID},
	{ERRHRD,	ERRgeneral,	NT_STATUS_ALLOTTED_SPACE_EXCEEDED},
/*	{ This NT error code was 'sqashed'
	 from NT_STATUS_INSUFFICIENT_RESOURCES to NT_STATUS_INSUFF_SERVER_RESOURCES 
	 during the session setup }
*/
	{ERRDOS,	ERRnomem,	NT_STATUS_INSUFFICIENT_RESOURCES},
	{ERRDOS,	ERRbadpath,	NT_STATUS_DFS_EXIT_PATH_FOUND},
	{ERRDOS,	23,	NT_STATUS_DEVICE_DATA_ERROR},
	{ERRHRD,	ERRgeneral,	NT_STATUS_DEVICE_NOT_CONNECTED},
	{ERRDOS,	21,	NT_STATUS_DEVICE_POWER_FAILURE},
	{ERRDOS,	487,	NT_STATUS_FREE_VM_NOT_AT_BASE},
	{ERRDOS,	487,	NT_STATUS_MEMORY_NOT_ALLOCATED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_WORKING_SET_QUOTA},
	{ERRDOS,	19,	NT_STATUS_MEDIA_WRITE_PROTECTED},
	{ERRDOS,	21,	NT_STATUS_DEVICE_NOT_READY},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_GROUP_ATTRIBUTES},
	{ERRHRD,	ERRgeneral,	NT_STATUS_BAD_IMPERSONATION_LEVEL},
	{ERRHRD,	ERRgeneral,	NT_STATUS_CANT_OPEN_ANONYMOUS},
	{ERRHRD,	ERRgeneral,	NT_STATUS_BAD_VALIDATION_CLASS},
	{ERRHRD,	ERRgeneral,	NT_STATUS_BAD_TOKEN_TYPE},
	{ERRDOS,	87,	NT_STATUS_BAD_MASTER_BOOT_RECORD},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INSTRUCTION_MISALIGNMENT},
	{ERRDOS,	ERRpipebusy,	NT_STATUS_INSTANCE_NOT_AVAILABLE},
	{ERRDOS,	ERRpipebusy,	NT_STATUS_PIPE_NOT_AVAILABLE},
	{ERRDOS,	ERRbadpipe,	NT_STATUS_INVALID_PIPE_STATE},
	{ERRDOS,	ERRpipebusy,	NT_STATUS_PIPE_BUSY},
	{ERRDOS,	ERRbadfunc,	NT_STATUS_ILLEGAL_FUNCTION},
	{ERRDOS,	ERRnotconnected,	NT_STATUS_PIPE_DISCONNECTED},
	{ERRDOS,	ERRpipeclosing,	NT_STATUS_PIPE_CLOSING},
	{ERRHRD,	ERRgeneral,	NT_STATUS_PIPE_CONNECTED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_PIPE_LISTENING},
	{ERRDOS,	ERRbadpipe,	NT_STATUS_INVALID_READ_MODE},
	{ERRDOS,	121,	NT_STATUS_IO_TIMEOUT},
	{ERRDOS,	38,	NT_STATUS_FILE_FORCED_CLOSED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_PROFILING_NOT_STARTED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_PROFILING_NOT_STOPPED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_COULD_NOT_INTERPRET},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_FILE_IS_A_DIRECTORY},
	{ERRDOS,	ERRunsup,	NT_STATUS_NOT_SUPPORTED},
	{ERRDOS,	51,	NT_STATUS_REMOTE_NOT_LISTENING},
	{ERRDOS,	52,	NT_STATUS_DUPLICATE_NAME},
	{ERRDOS,	53,	NT_STATUS_BAD_NETWORK_PATH},
	{ERRDOS,	54,	NT_STATUS_NETWORK_BUSY},
	{ERRDOS,	55,	NT_STATUS_DEVICE_DOES_NOT_EXIST},
	{ERRDOS,	56,	NT_STATUS_TOO_MANY_COMMANDS},
	{ERRDOS,	57,	NT_STATUS_ADAPTER_HARDWARE_ERROR},
	{ERRDOS,	58,	NT_STATUS_INVALID_NETWORK_RESPONSE},
	{ERRDOS,	59,	NT_STATUS_UNEXPECTED_NETWORK_ERROR},
	{ERRDOS,	60,	NT_STATUS_BAD_REMOTE_ADAPTER},
	{ERRDOS,	61,	NT_STATUS_PRINT_QUEUE_FULL},
	{ERRDOS,	62,	NT_STATUS_NO_SPOOL_SPACE},
	{ERRDOS,	63,	NT_STATUS_PRINT_CANCELLED},
	{ERRDOS,	64,	NT_STATUS_NETWORK_NAME_DELETED},
	{ERRDOS,	65,	NT_STATUS_NETWORK_ACCESS_DENIED},
	{ERRDOS,	66,	NT_STATUS_BAD_DEVICE_TYPE},
	{ERRDOS,	ERRnosuchshare,	NT_STATUS_BAD_NETWORK_NAME},
	{ERRDOS,	68,	NT_STATUS_TOO_MANY_NAMES},
	{ERRDOS,	69,	NT_STATUS_TOO_MANY_SESSIONS},
	{ERRDOS,	70,	NT_STATUS_SHARING_PAUSED},
	{ERRDOS,	71,	NT_STATUS_REQUEST_NOT_ACCEPTED},
	{ERRDOS,	72,	NT_STATUS_REDIRECTOR_PAUSED},
	{ERRDOS,	88,	NT_STATUS_NET_WRITE_FAULT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_PROFILING_AT_LIMIT},
	{ERRDOS,	ERRdiffdevice,	NT_STATUS_NOT_SAME_DEVICE},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_FILE_RENAMED},
	{ERRDOS,	240,	NT_STATUS_VIRTUAL_CIRCUIT_CLOSED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NO_SECURITY_ON_OBJECT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_CANT_WAIT},
	{ERRDOS,	ERRpipeclosing,	NT_STATUS_PIPE_EMPTY},
	{ERRHRD,	ERRgeneral,	NT_STATUS_CANT_ACCESS_DOMAIN_INFO},
	{ERRHRD,	ERRgeneral,	NT_STATUS_CANT_TERMINATE_SELF},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_SERVER_STATE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_DOMAIN_STATE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_DOMAIN_ROLE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NO_SUCH_DOMAIN},
	{ERRHRD,	ERRgeneral,	NT_STATUS_DOMAIN_EXISTS},
	{ERRHRD,	ERRgeneral,	NT_STATUS_DOMAIN_LIMIT_EXCEEDED},
	{ERRDOS,	300,	NT_STATUS_OPLOCK_NOT_GRANTED},
	{ERRDOS,	301,	NT_STATUS_INVALID_OPLOCK_PROTOCOL},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INTERNAL_DB_CORRUPTION},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INTERNAL_ERROR},
	{ERRHRD,	ERRgeneral,	NT_STATUS_GENERIC_NOT_MAPPED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_BAD_DESCRIPTOR_FORMAT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_USER_BUFFER},
	{ERRHRD,	ERRgeneral,	NT_STATUS_UNEXPECTED_IO_ERROR},
	{ERRHRD,	ERRgeneral,	NT_STATUS_UNEXPECTED_MM_CREATE_ERR},
	{ERRHRD,	ERRgeneral,	NT_STATUS_UNEXPECTED_MM_MAP_ERROR},
	{ERRHRD,	ERRgeneral,	NT_STATUS_UNEXPECTED_MM_EXTEND_ERR},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NOT_LOGON_PROCESS},
	{ERRHRD,	ERRgeneral,	NT_STATUS_LOGON_SESSION_EXISTS},
	{ERRDOS,	87,	NT_STATUS_INVALID_PARAMETER_1},
	{ERRDOS,	87,	NT_STATUS_INVALID_PARAMETER_2},
	{ERRDOS,	87,	NT_STATUS_INVALID_PARAMETER_3},
	{ERRDOS,	87,	NT_STATUS_INVALID_PARAMETER_4},
	{ERRDOS,	87,	NT_STATUS_INVALID_PARAMETER_5},
	{ERRDOS,	87,	NT_STATUS_INVALID_PARAMETER_6},
	{ERRDOS,	87,	NT_STATUS_INVALID_PARAMETER_7},
	{ERRDOS,	87,	NT_STATUS_INVALID_PARAMETER_8},
	{ERRDOS,	87,	NT_STATUS_INVALID_PARAMETER_9},
	{ERRDOS,	87,	NT_STATUS_INVALID_PARAMETER_10},
	{ERRDOS,	87,	NT_STATUS_INVALID_PARAMETER_11},
	{ERRDOS,	87,	NT_STATUS_INVALID_PARAMETER_12},
	{ERRDOS,	ERRbadpath,	NT_STATUS_REDIRECTOR_NOT_STARTED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_REDIRECTOR_STARTED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_STACK_OVERFLOW},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NO_SUCH_PACKAGE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_BAD_FUNCTION_TABLE},
	{ERRDOS,	203,	NT_STATUS(0xc0000100)},
	{ERRDOS,	145,	NT_STATUS_DIRECTORY_NOT_EMPTY},
	{ERRHRD,	ERRgeneral,	NT_STATUS_FILE_CORRUPT_ERROR},
	{ERRDOS,	267,	NT_STATUS_NOT_A_DIRECTORY},
	{ERRHRD,	ERRgeneral,	NT_STATUS_BAD_LOGON_SESSION_STATE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_LOGON_SESSION_COLLISION},
	{ERRDOS,	206,	NT_STATUS_NAME_TOO_LONG},
	{ERRDOS,	2401,	NT_STATUS_FILES_OPEN},
	{ERRDOS,	2404,	NT_STATUS_CONNECTION_IN_USE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_MESSAGE_NOT_FOUND},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_PROCESS_IS_TERMINATING},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_LOGON_TYPE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NO_GUID_TRANSLATION},
	{ERRHRD,	ERRgeneral,	NT_STATUS_CANNOT_IMPERSONATE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_IMAGE_ALREADY_LOADED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_ABIOS_NOT_PRESENT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_ABIOS_LID_NOT_EXIST},
	{ERRHRD,	ERRgeneral,	NT_STATUS_ABIOS_LID_ALREADY_OWNED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_ABIOS_NOT_LID_OWNER},
	{ERRHRD,	ERRgeneral,	NT_STATUS_ABIOS_INVALID_COMMAND},
	{ERRHRD,	ERRgeneral,	NT_STATUS_ABIOS_INVALID_LID},
	{ERRHRD,	ERRgeneral,	NT_STATUS_ABIOS_SELECTOR_NOT_AVAILABLE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_ABIOS_INVALID_SELECTOR},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NO_LDT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_LDT_SIZE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_LDT_OFFSET},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_LDT_DESCRIPTOR},
	{ERRDOS,	193,	NT_STATUS_INVALID_IMAGE_NE_FORMAT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_RXACT_INVALID_STATE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_RXACT_COMMIT_FAILURE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_MAPPED_FILE_SIZE_ZERO},
	{ERRDOS,	ERRnofids,	NT_STATUS_TOO_MANY_OPENED_FILES},
	{ERRHRD,	ERRgeneral,	NT_STATUS_CANCELLED},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_CANNOT_DELETE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_COMPUTER_NAME},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_FILE_DELETED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_SPECIAL_ACCOUNT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_SPECIAL_GROUP},
	{ERRHRD,	ERRgeneral,	NT_STATUS_SPECIAL_USER},
	{ERRHRD,	ERRgeneral,	NT_STATUS_MEMBERS_PRIMARY_GROUP},
	{ERRDOS,	ERRbadfid,	NT_STATUS_FILE_CLOSED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_TOO_MANY_THREADS},
	{ERRHRD,	ERRgeneral,	NT_STATUS_THREAD_NOT_IN_PROCESS},
	{ERRHRD,	ERRgeneral,	NT_STATUS_TOKEN_ALREADY_IN_USE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_PAGEFILE_QUOTA_EXCEEDED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_COMMITMENT_LIMIT},
	{ERRDOS,	193,	NT_STATUS_INVALID_IMAGE_LE_FORMAT},
	{ERRDOS,	193,	NT_STATUS_INVALID_IMAGE_NOT_MZ},
	{ERRDOS,	193,	NT_STATUS_INVALID_IMAGE_PROTECT},
	{ERRDOS,	193,	NT_STATUS_INVALID_IMAGE_WIN_16},
	{ERRHRD,	ERRgeneral,	NT_STATUS_LOGON_SERVER_CONFLICT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_TIME_DIFFERENCE_AT_DC},
	{ERRHRD,	ERRgeneral,	NT_STATUS_SYNCHRONIZATION_REQUIRED},
	{ERRDOS,	126,	NT_STATUS_DLL_NOT_FOUND},
	{ERRHRD,	ERRgeneral,	NT_STATUS_OPEN_FAILED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_IO_PRIVILEGE_FAILED},
	{ERRDOS,	182,	NT_STATUS_ORDINAL_NOT_FOUND},
	{ERRDOS,	127,	NT_STATUS_ENTRYPOINT_NOT_FOUND},
	{ERRHRD,	ERRgeneral,	NT_STATUS_CONTROL_C_EXIT},
	{ERRDOS,	64,	NT_STATUS_LOCAL_DISCONNECT},
	{ERRDOS,	64,	NT_STATUS_REMOTE_DISCONNECT},
	{ERRDOS,	51,	NT_STATUS_REMOTE_RESOURCES},
	{ERRDOS,	59,	NT_STATUS_LINK_FAILED},
	{ERRDOS,	59,	NT_STATUS_LINK_TIMEOUT},
	{ERRDOS,	59,	NT_STATUS_INVALID_CONNECTION},
	{ERRDOS,	59,	NT_STATUS_INVALID_ADDRESS},
	{ERRHRD,	ERRgeneral,	NT_STATUS_DLL_INIT_FAILED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_MISSING_SYSTEMFILE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_UNHANDLED_EXCEPTION},
	{ERRHRD,	ERRgeneral,	NT_STATUS_APP_INIT_FAILURE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_PAGEFILE_CREATE_FAILED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NO_PAGEFILE},
	{ERRDOS,	124,	NT_STATUS_INVALID_LEVEL},
	{ERRDOS,	86,	NT_STATUS_WRONG_PASSWORD_CORE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_ILLEGAL_FLOAT_CONTEXT},
	{ERRDOS,	109,	NT_STATUS_PIPE_BROKEN},
	{ERRHRD,	ERRgeneral,	NT_STATUS_REGISTRY_CORRUPT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_REGISTRY_IO_FAILED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NO_EVENT_PAIR},
	{ERRHRD,	ERRgeneral,	NT_STATUS_UNRECOGNIZED_VOLUME},
	{ERRHRD,	ERRgeneral,	NT_STATUS_SERIAL_NO_DEVICE_INITED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NO_SUCH_ALIAS},
	{ERRHRD,	ERRgeneral,	NT_STATUS_MEMBER_NOT_IN_ALIAS},
	{ERRHRD,	ERRgeneral,	NT_STATUS_MEMBER_IN_ALIAS},
	{ERRHRD,	ERRgeneral,	NT_STATUS_ALIAS_EXISTS},
	{ERRHRD,	ERRgeneral,	NT_STATUS_LOGON_NOT_GRANTED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_TOO_MANY_SECRETS},
	{ERRHRD,	ERRgeneral,	NT_STATUS_SECRET_TOO_LONG},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INTERNAL_DB_ERROR},
	{ERRHRD,	ERRgeneral,	NT_STATUS_FULLSCREEN_MODE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_TOO_MANY_CONTEXT_IDS},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_LOGON_TYPE_NOT_GRANTED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NOT_REGISTRY_FILE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NT_CROSS_ENCRYPTION_REQUIRED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_DOMAIN_CTRLR_CONFIG_ERROR},
	{ERRHRD,	ERRgeneral,	NT_STATUS_FT_MISSING_MEMBER},
	{ERRHRD,	ERRgeneral,	NT_STATUS_ILL_FORMED_SERVICE_ENTRY},
	{ERRHRD,	ERRgeneral,	NT_STATUS_ILLEGAL_CHARACTER},
	{ERRHRD,	ERRgeneral,	NT_STATUS_UNMAPPABLE_CHARACTER},
	{ERRHRD,	ERRgeneral,	NT_STATUS_UNDEFINED_CHARACTER},
	{ERRHRD,	ERRgeneral,	NT_STATUS_FLOPPY_VOLUME},
	{ERRHRD,	ERRgeneral,	NT_STATUS_FLOPPY_ID_MARK_NOT_FOUND},
	{ERRHRD,	ERRgeneral,	NT_STATUS_FLOPPY_WRONG_CYLINDER},
	{ERRHRD,	ERRgeneral,	NT_STATUS_FLOPPY_UNKNOWN_ERROR},
	{ERRHRD,	ERRgeneral,	NT_STATUS_FLOPPY_BAD_REGISTERS},
	{ERRHRD,	ERRgeneral,	NT_STATUS_DISK_RECALIBRATE_FAILED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_DISK_OPERATION_FAILED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_DISK_RESET_FAILED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_SHARED_IRQ_BUSY},
	{ERRHRD,	ERRgeneral,	NT_STATUS_FT_ORPHANING},
	{ERRHRD,	ERRgeneral,	NT_STATUS(0xc000016e)},
	{ERRHRD,	ERRgeneral,	NT_STATUS(0xc000016f)},
	{ERRHRD,	ERRgeneral,	NT_STATUS(0xc0000170)},
	{ERRHRD,	ERRgeneral,	NT_STATUS(0xc0000171)},
	{ERRHRD,	ERRgeneral,	NT_STATUS_PARTITION_FAILURE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_BLOCK_LENGTH},
	{ERRHRD,	ERRgeneral,	NT_STATUS_DEVICE_NOT_PARTITIONED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_UNABLE_TO_LOCK_MEDIA},
	{ERRHRD,	ERRgeneral,	NT_STATUS_UNABLE_TO_UNLOAD_MEDIA},
	{ERRHRD,	ERRgeneral,	NT_STATUS_EOM_OVERFLOW},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NO_MEDIA},
	{ERRHRD,	ERRgeneral,	NT_STATUS(0xc0000179)},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NO_SUCH_MEMBER},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_MEMBER},
	{ERRHRD,	ERRgeneral,	NT_STATUS_KEY_DELETED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NO_LOG_SPACE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_TOO_MANY_SIDS},
	{ERRHRD,	ERRgeneral,	NT_STATUS_LM_CROSS_ENCRYPTION_REQUIRED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_KEY_HAS_CHILDREN},
	{ERRHRD,	ERRgeneral,	NT_STATUS_CHILD_MUST_BE_VOLATILE},
	{ERRDOS,	87,	NT_STATUS_DEVICE_CONFIGURATION_ERROR},
	{ERRHRD,	ERRgeneral,	NT_STATUS_DRIVER_INTERNAL_ERROR},
	{ERRDOS,	22,	NT_STATUS_INVALID_DEVICE_STATE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_IO_DEVICE_ERROR},
	{ERRHRD,	ERRgeneral,	NT_STATUS_DEVICE_PROTOCOL_ERROR},
	{ERRHRD,	ERRgeneral,	NT_STATUS_BACKUP_CONTROLLER},
	{ERRHRD,	ERRgeneral,	NT_STATUS_LOG_FILE_FULL},
	{ERRDOS,	19,	NT_STATUS_TOO_LATE},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_NO_TRUST_LSA_SECRET},
/*	{ This NT error code was 'sqashed'
	 from NT_STATUS_NO_TRUST_SAM_ACCOUNT to NT_STATUS_TRUSTED_RELATIONSHIP_FAILURE 
	 during the session setup }
*/
	{ERRDOS,	ERRnoaccess,	NT_STATUS_NO_TRUST_SAM_ACCOUNT},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_TRUSTED_DOMAIN_FAILURE},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_TRUSTED_RELATIONSHIP_FAILURE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_EVENTLOG_FILE_CORRUPT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_EVENTLOG_CANT_START},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_TRUST_FAILURE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_MUTANT_LIMIT_EXCEEDED},
	{ERRDOS,	ERRinvgroup,	NT_STATUS_NETLOGON_NOT_STARTED},
	{ERRSRV,	2239,	NT_STATUS_ACCOUNT_EXPIRED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_POSSIBLE_DEADLOCK},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NETWORK_CREDENTIAL_CONFLICT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_REMOTE_SESSION_LIMIT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_EVENTLOG_FILE_CHANGED},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_NOLOGON_INTERDOMAIN_TRUST_ACCOUNT},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_NOLOGON_WORKSTATION_TRUST_ACCOUNT},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_NOLOGON_SERVER_TRUST_ACCOUNT},
/*	{ This NT error code was 'sqashed'
	 from NT_STATUS_DOMAIN_TRUST_INCONSISTENT to NT_STATUS_LOGON_FAILURE 
	 during the session setup }
*/
	{ERRDOS,	ERRnoaccess,	NT_STATUS_DOMAIN_TRUST_INCONSISTENT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_FS_DRIVER_REQUIRED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NO_USER_SESSION_KEY},
	{ERRDOS,	59,	NT_STATUS_USER_SESSION_DELETED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_RESOURCE_LANG_NOT_FOUND},
	{ERRDOS,	ERRnomem,	NT_STATUS_INSUFF_SERVER_RESOURCES},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_BUFFER_SIZE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_ADDRESS_COMPONENT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_ADDRESS_WILDCARD},
	{ERRDOS,	68,	NT_STATUS_TOO_MANY_ADDRESSES},
	{ERRDOS,	52,	NT_STATUS_ADDRESS_ALREADY_EXISTS},
	{ERRDOS,	64,	NT_STATUS_ADDRESS_CLOSED},
	{ERRDOS,	64,	NT_STATUS_CONNECTION_DISCONNECTED},
	{ERRDOS,	64,	NT_STATUS_CONNECTION_RESET},
	{ERRDOS,	68,	NT_STATUS_TOO_MANY_NODES},
	{ERRDOS,	59,	NT_STATUS_TRANSACTION_ABORTED},
	{ERRDOS,	59,	NT_STATUS_TRANSACTION_TIMED_OUT},
	{ERRDOS,	59,	NT_STATUS_TRANSACTION_NO_RELEASE},
	{ERRDOS,	59,	NT_STATUS_TRANSACTION_NO_MATCH},
	{ERRDOS,	59,	NT_STATUS_TRANSACTION_RESPONDED},
	{ERRDOS,	59,	NT_STATUS_TRANSACTION_INVALID_ID},
	{ERRDOS,	59,	NT_STATUS_TRANSACTION_INVALID_TYPE},
	{ERRDOS,	ERRunsup,	NT_STATUS_NOT_SERVER_SESSION},
	{ERRDOS,	ERRunsup,	NT_STATUS_NOT_CLIENT_SESSION},
	{ERRHRD,	ERRgeneral,	NT_STATUS_CANNOT_LOAD_REGISTRY_FILE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_DEBUG_ATTACH_FAILED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_SYSTEM_PROCESS_TERMINATED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_DATA_NOT_ACCEPTED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NO_BROWSER_SERVERS_FOUND},
	{ERRHRD,	ERRgeneral,	NT_STATUS_VDM_HARD_ERROR},
	{ERRHRD,	ERRgeneral,	NT_STATUS_DRIVER_CANCEL_TIMEOUT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_REPLY_MESSAGE_MISMATCH},
	{ERRHRD,	ERRgeneral,	NT_STATUS_MAPPED_ALIGNMENT},
	{ERRDOS,	193,	NT_STATUS_IMAGE_CHECKSUM_MISMATCH},
	{ERRHRD,	ERRgeneral,	NT_STATUS_LOST_WRITEBEHIND_DATA},
	{ERRHRD,	ERRgeneral,	NT_STATUS_CLIENT_SERVER_PARAMETERS_INVALID},
	{ERRSRV,	2242,	NT_STATUS_PASSWORD_MUST_CHANGE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NOT_FOUND},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NOT_TINY_STREAM},
	{ERRHRD,	ERRgeneral,	NT_STATUS_RECOVERY_FAILURE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_STACK_OVERFLOW_READ},
	{ERRHRD,	ERRgeneral,	NT_STATUS_FAIL_CHECK},
	{ERRHRD,	ERRgeneral,	NT_STATUS_DUPLICATE_OBJECTID},
	{ERRHRD,	ERRgeneral,	NT_STATUS_OBJECTID_EXISTS},
	{ERRHRD,	ERRgeneral,	NT_STATUS_CONVERT_TO_LARGE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_RETRY},
	{ERRHRD,	ERRgeneral,	NT_STATUS_FOUND_OUT_OF_SCOPE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_ALLOCATE_BUCKET},
	{ERRHRD,	ERRgeneral,	NT_STATUS_PROPSET_NOT_FOUND},
	{ERRHRD,	ERRgeneral,	NT_STATUS_MARSHALL_OVERFLOW},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_VARIANT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_DOMAIN_CONTROLLER_NOT_FOUND},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_ACCOUNT_LOCKED_OUT},
	{ERRDOS,	ERRbadfid,	NT_STATUS_HANDLE_NOT_CLOSABLE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_CONNECTION_REFUSED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_GRACEFUL_DISCONNECT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_ADDRESS_ALREADY_ASSOCIATED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_ADDRESS_NOT_ASSOCIATED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_CONNECTION_INVALID},
	{ERRHRD,	ERRgeneral,	NT_STATUS_CONNECTION_ACTIVE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NETWORK_UNREACHABLE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_HOST_UNREACHABLE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_PROTOCOL_UNREACHABLE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_PORT_UNREACHABLE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_REQUEST_ABORTED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_CONNECTION_ABORTED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_BAD_COMPRESSION_BUFFER},
	{ERRHRD,	ERRgeneral,	NT_STATUS_USER_MAPPED_FILE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_AUDIT_FAILED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_TIMER_RESOLUTION_NOT_SET},
	{ERRHRD,	ERRgeneral,	NT_STATUS_CONNECTION_COUNT_LIMIT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_LOGIN_TIME_RESTRICTION},
	{ERRHRD,	ERRgeneral,	NT_STATUS_LOGIN_WKSTA_RESTRICTION},
	{ERRDOS,	193,	NT_STATUS_IMAGE_MP_UP_MISMATCH},
	{ERRHRD,	ERRgeneral,	NT_STATUS(0xc000024a)},
	{ERRHRD,	ERRgeneral,	NT_STATUS(0xc000024b)},
	{ERRHRD,	ERRgeneral,	NT_STATUS(0xc000024c)},
	{ERRHRD,	ERRgeneral,	NT_STATUS(0xc000024d)},
	{ERRHRD,	ERRgeneral,	NT_STATUS(0xc000024e)},
	{ERRHRD,	ERRgeneral,	NT_STATUS(0xc000024f)},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INSUFFICIENT_LOGON_INFO},
	{ERRHRD,	ERRgeneral,	NT_STATUS_BAD_DLL_ENTRYPOINT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_BAD_SERVICE_ENTRYPOINT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_LPC_REPLY_LOST},
	{ERRHRD,	ERRgeneral,	NT_STATUS_IP_ADDRESS_CONFLICT1},
	{ERRHRD,	ERRgeneral,	NT_STATUS_IP_ADDRESS_CONFLICT2},
	{ERRHRD,	ERRgeneral,	NT_STATUS_REGISTRY_QUOTA_LIMIT},
	{ERRSRV,	ERRbadtype,	NT_STATUS_PATH_NOT_COVERED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NO_CALLBACK_ACTIVE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_LICENSE_QUOTA_EXCEEDED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_PWD_TOO_SHORT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_PWD_TOO_RECENT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_PWD_HISTORY_CONFLICT},
	{ERRHRD,	ERRgeneral,	NT_STATUS(0xc000025d)},
	{ERRHRD,	ERRgeneral,	NT_STATUS_PLUGPLAY_NO_DEVICE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_UNSUPPORTED_COMPRESSION},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_HW_PROFILE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_PLUGPLAY_DEVICE_PATH},
	{ERRDOS,	182,	NT_STATUS_DRIVER_ORDINAL_NOT_FOUND},
	{ERRDOS,	127,	NT_STATUS_DRIVER_ENTRYPOINT_NOT_FOUND},
	{ERRDOS,	288,	NT_STATUS_RESOURCE_NOT_OWNED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_TOO_MANY_LINKS},
	{ERRHRD,	ERRgeneral,	NT_STATUS_QUOTA_LIST_INCONSISTENT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_FILE_IS_OFFLINE},
	{ERRDOS,	21,	NT_STATUS(0xc000026e)},
	{ERRDOS,	161,	NT_STATUS(0xc0000281)},
	{ERRDOS,	ERRnoaccess,	NT_STATUS(0xc000028a)},
	{ERRDOS,	ERRnoaccess,	NT_STATUS(0xc000028b)},
	{ERRHRD,	ERRgeneral,	NT_STATUS(0xc000028c)},
	{ERRDOS,	ERRnoaccess,	NT_STATUS(0xc000028d)},
	{ERRDOS,	ERRnoaccess,	NT_STATUS(0xc000028e)},
	{ERRDOS,	ERRnoaccess,	NT_STATUS(0xc000028f)},
	{ERRDOS,	ERRnoaccess,	NT_STATUS(0xc0000290)},
	{ERRDOS,	ERRbadfunc,	NT_STATUS(0xc000029c)},
};


/* dos -> nt status error map */
static const struct {
	uint8 dos_class;
	uint32 dos_code;
	NTSTATUS ntstatus;
} dos_to_ntstatus_map[] = {
	{ERRDOS,	ERRbadfunc,	NT_STATUS_NOT_IMPLEMENTED},
	{ERRDOS,	ERRbadfile,	NT_STATUS_NO_SUCH_FILE},
	{ERRDOS,	ERRbadpath,	NT_STATUS_OBJECT_PATH_NOT_FOUND},
	{ERRDOS,	ERRnofids,	NT_STATUS_TOO_MANY_OPENED_FILES},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_ACCESS_DENIED},
	{ERRDOS,	ERRbadfid,	NT_STATUS_INVALID_HANDLE},
	{ERRDOS,	ERRnomem,	NT_STATUS_INSUFFICIENT_RESOURCES},
	{ERRDOS,	ERRbadaccess,	NT_STATUS_INVALID_LOCK_SEQUENCE},
	{ERRDOS,	ERRbaddata,	NT_STATUS_DATA_ERROR},
	{ERRDOS,	14,	NT_STATUS_SECTION_NOT_EXTENDED},
	{ERRDOS,	ERRremcd,	NT_STATUS_DIRECTORY_NOT_EMPTY},
	{ERRDOS,	ERRdiffdevice,	NT_STATUS_NOT_SAME_DEVICE},
	{ERRDOS,	ERRnofiles,	NT_STATUS(0x80000006)},
	{ERRDOS,	19,	NT_STATUS_MEDIA_WRITE_PROTECTED},
	{ERRDOS,	21,	NT_STATUS_NO_MEDIA_IN_DEVICE},
	{ERRDOS,	22,	NT_STATUS_INVALID_DEVICE_STATE},
	{ERRDOS,	23,	NT_STATUS_DATA_ERROR},
	{ERRDOS,	24,	NT_STATUS_DATA_ERROR},
	{ERRDOS,	26,	NT_STATUS_DISK_CORRUPT_ERROR},
	{ERRDOS,	27,	NT_STATUS_NONEXISTENT_SECTOR},
	{ERRDOS,	28,	NT_STATUS(0x8000000e)},
	{ERRDOS,	31,	NT_STATUS_UNSUCCESSFUL},
	{ERRDOS,	ERRbadshare,	NT_STATUS_SHARING_VIOLATION},
	{ERRDOS,	ERRlock,	NT_STATUS_FILE_LOCK_CONFLICT},
	{ERRDOS,	34,	NT_STATUS_WRONG_VOLUME},
	{ERRDOS,	38,	NT_STATUS_END_OF_FILE},
	{ERRDOS,	ERRunsup,	NT_STATUS_CTL_FILE_NOT_SUPPORTED},
	{ERRDOS,	51,	NT_STATUS_REMOTE_NOT_LISTENING},
	{ERRDOS,	52,	NT_STATUS_DUPLICATE_NAME},
	{ERRDOS,	53,	NT_STATUS_BAD_NETWORK_PATH},
	{ERRDOS,	54,	NT_STATUS_NETWORK_BUSY},
	{ERRDOS,	55,	NT_STATUS_DEVICE_DOES_NOT_EXIST},
	{ERRDOS,	56,	NT_STATUS_TOO_MANY_COMMANDS},
	{ERRDOS,	57,	NT_STATUS_ADAPTER_HARDWARE_ERROR},
	{ERRDOS,	58,	NT_STATUS_INVALID_NETWORK_RESPONSE},
	{ERRDOS,	59,	NT_STATUS_UNEXPECTED_NETWORK_ERROR},
	{ERRDOS,	60,	NT_STATUS_BAD_REMOTE_ADAPTER},
	{ERRDOS,	61,	NT_STATUS_PRINT_QUEUE_FULL},
	{ERRDOS,	62,	NT_STATUS_NO_SPOOL_SPACE},
	{ERRDOS,	63,	NT_STATUS_PRINT_CANCELLED},
	{ERRDOS,	64,	NT_STATUS_NETWORK_NAME_DELETED},
	{ERRDOS,	65,	NT_STATUS_NETWORK_ACCESS_DENIED},
	{ERRDOS,	66,	NT_STATUS_BAD_DEVICE_TYPE},
	{ERRDOS,	ERRnosuchshare,	NT_STATUS_BAD_NETWORK_NAME},
	{ERRDOS,	68,	NT_STATUS_TOO_MANY_GUIDS_REQUESTED},
	{ERRDOS,	69,	NT_STATUS_TOO_MANY_SESSIONS},
	{ERRDOS,	70,	NT_STATUS_SHARING_PAUSED},
	{ERRDOS,	71,	NT_STATUS_REQUEST_NOT_ACCEPTED},
	{ERRDOS,	72,	NT_STATUS_REDIRECTOR_PAUSED},
	{ERRDOS,	ERRfilexists,	NT_STATUS_OBJECT_NAME_COLLISION},
	{ERRDOS,	86,	NT_STATUS_WRONG_PASSWORD},
	{ERRDOS,	87,	NT_STATUS_INVALID_INFO_CLASS},
	{ERRDOS,	88,	NT_STATUS_NET_WRITE_FAULT},
	{ERRDOS,	109,	NT_STATUS_PIPE_BROKEN},
	{ERRDOS,	111,	STATUS_MORE_ENTRIES},
	{ERRDOS,	112,	NT_STATUS_DISK_FULL},
	{ERRDOS,	121,	NT_STATUS_IO_TIMEOUT},
	{ERRDOS,	122,	NT_STATUS_BUFFER_TOO_SMALL},
	{ERRDOS,	ERRinvalidname,	NT_STATUS_OBJECT_NAME_INVALID},
	{ERRDOS,	124,	NT_STATUS_INVALID_LEVEL},
	{ERRDOS,	126,	NT_STATUS_DLL_NOT_FOUND},
	{ERRDOS,	127,	NT_STATUS_PROCEDURE_NOT_FOUND},
	{ERRDOS,	145,	NT_STATUS_DIRECTORY_NOT_EMPTY},
	{ERRDOS,	154,	NT_STATUS_INVALID_VOLUME_LABEL},
	{ERRDOS,	156,	NT_STATUS_SUSPEND_COUNT_EXCEEDED},
	{ERRDOS,	158,	NT_STATUS_NOT_LOCKED},
	{ERRDOS,	161,	NT_STATUS_OBJECT_PATH_INVALID},
	{ERRDOS,	170,	NT_STATUS(0x80000011)},
	{ERRDOS,	182,	NT_STATUS_ORDINAL_NOT_FOUND},
	{ERRDOS,	183,	NT_STATUS_OBJECT_NAME_COLLISION},
	{ERRDOS,	193,	NT_STATUS_BAD_INITIAL_PC},
	{ERRDOS,	203,	NT_STATUS(0xc0000100)},
	{ERRDOS,	206,	NT_STATUS_NAME_TOO_LONG},
	{ERRDOS,	ERRbadpipe,	NT_STATUS_INVALID_INFO_CLASS},
	{ERRDOS,	ERRpipebusy,	NT_STATUS_INSTANCE_NOT_AVAILABLE},
	{ERRDOS,	ERRpipeclosing,	NT_STATUS_PIPE_CLOSING},
	{ERRDOS,	ERRnotconnected,	NT_STATUS_PIPE_DISCONNECTED},
	{ERRDOS,	ERRmoredata,	NT_STATUS_MORE_PROCESSING_REQUIRED},
	{ERRDOS,	240,	NT_STATUS_VIRTUAL_CIRCUIT_CLOSED},
	{ERRDOS,	254,	NT_STATUS(0x80000013)},
	{ERRDOS,	255,	NT_STATUS_EA_TOO_LARGE},
	{ERRDOS,	259,	NT_STATUS_GUIDS_EXHAUSTED},
	{ERRDOS,	267,	NT_STATUS_NOT_A_DIRECTORY},
	{ERRDOS,	275,	NT_STATUS_EA_TOO_LARGE},
	{ERRDOS,	276,	NT_STATUS_NONEXISTENT_EA_ENTRY},
	{ERRDOS,	277,	NT_STATUS_NONEXISTENT_EA_ENTRY},
	{ERRDOS,	278,	NT_STATUS_NONEXISTENT_EA_ENTRY},
	{ERRDOS,	282,	NT_STATUS_EAS_NOT_SUPPORTED},
	{ERRDOS,	288,	NT_STATUS_MUTANT_NOT_OWNED},
	{ERRDOS,	298,	NT_STATUS_SEMAPHORE_LIMIT_EXCEEDED},
	{ERRDOS,	299,	NT_STATUS(0x8000000d)},
	{ERRDOS,	300,	NT_STATUS_OPLOCK_NOT_GRANTED},
	{ERRDOS,	301,	NT_STATUS_INVALID_OPLOCK_PROTOCOL},
	{ERRDOS,	487,	NT_STATUS_CONFLICTING_ADDRESSES},
	{ERRDOS,	534,	NT_STATUS_INTEGER_OVERFLOW},
	{ERRDOS,	535,	NT_STATUS_PIPE_CONNECTED},
	{ERRDOS,	536,	NT_STATUS_PIPE_LISTENING},
	{ERRDOS,	995,	NT_STATUS_CANCELLED},
	{ERRDOS,	997,	NT_STATUS(0x00000103)},
	{ERRDOS,	998,	NT_STATUS_ACCESS_VIOLATION},
	{ERRDOS,	999,	NT_STATUS_IN_PAGE_ERROR},
	{ERRDOS,	1001,	NT_STATUS_BAD_INITIAL_STACK},
	{ERRDOS,	1005,	NT_STATUS_UNRECOGNIZED_VOLUME},
	{ERRDOS,	1006,	NT_STATUS_FILE_INVALID},
	{ERRDOS,	1007,	NT_STATUS_FULLSCREEN_MODE},
	{ERRDOS,	1008,	NT_STATUS_NO_TOKEN},
	{ERRDOS,	1009,	NT_STATUS_REGISTRY_CORRUPT},
	{ERRDOS,	1016,	NT_STATUS_REGISTRY_IO_FAILED},
	{ERRDOS,	1017,	NT_STATUS_NOT_REGISTRY_FILE},
	{ERRDOS,	1018,	NT_STATUS_KEY_DELETED},
	{ERRDOS,	1019,	NT_STATUS_NO_LOG_SPACE},
	{ERRDOS,	1020,	NT_STATUS_KEY_HAS_CHILDREN},
	{ERRDOS,	1021,	NT_STATUS_CHILD_MUST_BE_VOLATILE},
	{ERRDOS,	1022,	NT_STATUS(0x0000010c)},
	{ERRSRV,	ERRbadpw,	NT_STATUS_WRONG_PASSWORD},
	{ERRSRV,	ERRbadtype,	NT_STATUS_BAD_DEVICE_TYPE},
	{ERRSRV,	ERRaccess,	NT_STATUS_NETWORK_ACCESS_DENIED},
	{ERRSRV,	ERRinvnid,	NT_STATUS_NETWORK_NAME_DELETED},
	{ERRSRV,	ERRinvnetname,	NT_STATUS_BAD_NETWORK_NAME},
	{ERRSRV,	ERRinvdevice,	NT_STATUS_BAD_DEVICE_TYPE},
	{ERRSRV,	ERRqfull,	NT_STATUS_PRINT_QUEUE_FULL},
	{ERRSRV,	ERRqtoobig,	NT_STATUS_NO_SPOOL_SPACE},
	{ERRSRV,	ERRinvpfid,	NT_STATUS_PRINT_CANCELLED},
	{ERRSRV,	ERRsmbcmd,	NT_STATUS_NOT_IMPLEMENTED},
	{ERRSRV,	ERRbadpermits,	NT_STATUS_NETWORK_ACCESS_DENIED},
	{ERRSRV,	ERRpaused,	NT_STATUS_SHARING_PAUSED},
	{ERRSRV,	ERRmsgoff,	NT_STATUS_REQUEST_NOT_ACCEPTED},
	{ERRSRV,	ERRnoroom,	NT_STATUS_DISK_FULL},
	{ERRSRV,	ERRnoresource,	NT_STATUS_REQUEST_NOT_ACCEPTED},
	{ERRSRV,	ERRtoomanyuids,	NT_STATUS_TOO_MANY_SESSIONS},
	{ERRSRV,	123,	NT_STATUS_OBJECT_NAME_INVALID},
	{ERRSRV,	206,	NT_STATUS_OBJECT_NAME_INVALID},
	{ERRHRD,	1,	NT_STATUS_NOT_IMPLEMENTED},
	{ERRHRD,	2,	NT_STATUS_NO_SUCH_DEVICE},
	{ERRHRD,	3,	NT_STATUS_OBJECT_PATH_NOT_FOUND},
	{ERRHRD,	4,	NT_STATUS_TOO_MANY_OPENED_FILES},
	{ERRHRD,	5,	NT_STATUS_INVALID_LOCK_SEQUENCE},
	{ERRHRD,	6,	NT_STATUS_INVALID_HANDLE},
	{ERRHRD,	8,	NT_STATUS_INSUFFICIENT_RESOURCES},
	{ERRHRD,	12,	NT_STATUS_INVALID_LOCK_SEQUENCE},
	{ERRHRD,	13,	NT_STATUS_DATA_ERROR},
	{ERRHRD,	14,	NT_STATUS_SECTION_NOT_EXTENDED},
	{ERRHRD,	16,	NT_STATUS_DIRECTORY_NOT_EMPTY},
	{ERRHRD,	17,	NT_STATUS_NOT_SAME_DEVICE},
	{ERRHRD,	18,	NT_STATUS(0x80000006)},
	{ERRHRD,	ERRnowrite,	NT_STATUS_MEDIA_WRITE_PROTECTED},
	{ERRHRD,	ERRnotready,	NT_STATUS_NO_MEDIA_IN_DEVICE},
	{ERRHRD,	ERRbadcmd,	NT_STATUS_INVALID_DEVICE_STATE},
	{ERRHRD,	ERRdata,	NT_STATUS_DATA_ERROR},
	{ERRHRD,	ERRbadreq,	NT_STATUS_DATA_ERROR},
	{ERRHRD,	ERRbadmedia,	NT_STATUS_DISK_CORRUPT_ERROR},
	{ERRHRD,	ERRbadsector,	NT_STATUS_NONEXISTENT_SECTOR},
	{ERRHRD,	ERRnopaper,	NT_STATUS(0x8000000e)},
	{ERRHRD,	ERRgeneral,	NT_STATUS_UNSUCCESSFUL},
	{ERRHRD,	ERRbadshare,	NT_STATUS_SHARING_VIOLATION},
	{ERRHRD,	ERRlock,	NT_STATUS_FILE_LOCK_CONFLICT},
	{ERRHRD,	ERRwrongdisk,	NT_STATUS_WRONG_VOLUME},
	{ERRHRD,	38,	NT_STATUS_END_OF_FILE},
	{ERRHRD,	ERRdiskfull,	NT_STATUS_DISK_FULL},
	{ERRHRD,	50,	NT_STATUS_CTL_FILE_NOT_SUPPORTED},
	{ERRHRD,	51,	NT_STATUS_REMOTE_NOT_LISTENING},
	{ERRHRD,	52,	NT_STATUS_DUPLICATE_NAME},
	{ERRHRD,	53,	NT_STATUS_BAD_NETWORK_PATH},
	{ERRHRD,	54,	NT_STATUS_NETWORK_BUSY},
	{ERRHRD,	55,	NT_STATUS_DEVICE_DOES_NOT_EXIST},
	{ERRHRD,	56,	NT_STATUS_TOO_MANY_COMMANDS},
	{ERRHRD,	57,	NT_STATUS_ADAPTER_HARDWARE_ERROR},
	{ERRHRD,	58,	NT_STATUS_INVALID_NETWORK_RESPONSE},
	{ERRHRD,	59,	NT_STATUS_UNEXPECTED_NETWORK_ERROR},
	{ERRHRD,	60,	NT_STATUS_BAD_REMOTE_ADAPTER},
	{ERRHRD,	61,	NT_STATUS_PRINT_QUEUE_FULL},
	{ERRHRD,	62,	NT_STATUS_NO_SPOOL_SPACE},
	{ERRHRD,	63,	NT_STATUS_PRINT_CANCELLED},
	{ERRHRD,	64,	NT_STATUS_NETWORK_NAME_DELETED},
	{ERRHRD,	65,	NT_STATUS_NETWORK_ACCESS_DENIED},
	{ERRHRD,	66,	NT_STATUS_BAD_DEVICE_TYPE},
	{ERRHRD,	67,	NT_STATUS_BAD_NETWORK_NAME},
	{ERRHRD,	68,	NT_STATUS_TOO_MANY_GUIDS_REQUESTED},
	{ERRHRD,	69,	NT_STATUS_TOO_MANY_SESSIONS},
	{ERRHRD,	70,	NT_STATUS_SHARING_PAUSED},
	{ERRHRD,	71,	NT_STATUS_REQUEST_NOT_ACCEPTED},
	{ERRHRD,	72,	NT_STATUS_REDIRECTOR_PAUSED},
	{ERRHRD,	80,	NT_STATUS_OBJECT_NAME_COLLISION},
	{ERRHRD,	86,	NT_STATUS_WRONG_PASSWORD},
	{ERRHRD,	87,	NT_STATUS_INVALID_INFO_CLASS},
	{ERRHRD,	88,	NT_STATUS_NET_WRITE_FAULT},
	{ERRHRD,	109,	NT_STATUS_PIPE_BROKEN},
	{ERRHRD,	111,	STATUS_MORE_ENTRIES},
	{ERRHRD,	112,	NT_STATUS_DISK_FULL},
	{ERRHRD,	121,	NT_STATUS_IO_TIMEOUT},
	{ERRHRD,	122,	NT_STATUS_BUFFER_TOO_SMALL},
	{ERRHRD,	123,	NT_STATUS_OBJECT_NAME_INVALID},
	{ERRHRD,	124,	NT_STATUS_INVALID_LEVEL},
	{ERRHRD,	126,	NT_STATUS_DLL_NOT_FOUND},
	{ERRHRD,	127,	NT_STATUS_PROCEDURE_NOT_FOUND},
	{ERRHRD,	145,	NT_STATUS_DIRECTORY_NOT_EMPTY},
	{ERRHRD,	154,	NT_STATUS_INVALID_VOLUME_LABEL},
	{ERRHRD,	156,	NT_STATUS_SUSPEND_COUNT_EXCEEDED},
	{ERRHRD,	158,	NT_STATUS_NOT_LOCKED},
	{ERRHRD,	161,	NT_STATUS_OBJECT_PATH_INVALID},
	{ERRHRD,	170,	NT_STATUS(0x80000011)},
	{ERRHRD,	182,	NT_STATUS_ORDINAL_NOT_FOUND},
	{ERRHRD,	183,	NT_STATUS_OBJECT_NAME_COLLISION},
	{ERRHRD,	193,	NT_STATUS_BAD_INITIAL_PC},
	{ERRHRD,	203,	NT_STATUS(0xc0000100)},
	{ERRHRD,	206,	NT_STATUS_NAME_TOO_LONG},
	{ERRHRD,	230,	NT_STATUS_INVALID_INFO_CLASS},
	{ERRHRD,	231,	NT_STATUS_INSTANCE_NOT_AVAILABLE},
	{ERRHRD,	232,	NT_STATUS_PIPE_CLOSING},
	{ERRHRD,	233,	NT_STATUS_PIPE_DISCONNECTED},
	{ERRHRD,	234,	STATUS_MORE_ENTRIES},
	{ERRHRD,	240,	NT_STATUS_VIRTUAL_CIRCUIT_CLOSED},
	{ERRHRD,	254,	NT_STATUS(0x80000013)},
	{ERRHRD,	255,	NT_STATUS_EA_TOO_LARGE},
	{ERRHRD,	259,	NT_STATUS_GUIDS_EXHAUSTED},
	{ERRHRD,	267,	NT_STATUS_NOT_A_DIRECTORY},
	{ERRHRD,	275,	NT_STATUS_EA_TOO_LARGE},
	{ERRHRD,	276,	NT_STATUS_NONEXISTENT_EA_ENTRY},
	{ERRHRD,	277,	NT_STATUS_NONEXISTENT_EA_ENTRY},
	{ERRHRD,	278,	NT_STATUS_NONEXISTENT_EA_ENTRY},
	{ERRHRD,	282,	NT_STATUS_EAS_NOT_SUPPORTED},
	{ERRHRD,	288,	NT_STATUS_MUTANT_NOT_OWNED},
	{ERRHRD,	298,	NT_STATUS_SEMAPHORE_LIMIT_EXCEEDED},
	{ERRHRD,	299,	NT_STATUS(0x8000000d)},
	{ERRHRD,	300,	NT_STATUS_OPLOCK_NOT_GRANTED},
	{ERRHRD,	301,	NT_STATUS_INVALID_OPLOCK_PROTOCOL},
	{ERRHRD,	487,	NT_STATUS_CONFLICTING_ADDRESSES},
	{ERRHRD,	534,	NT_STATUS_INTEGER_OVERFLOW},
	{ERRHRD,	535,	NT_STATUS_PIPE_CONNECTED},
	{ERRHRD,	536,	NT_STATUS_PIPE_LISTENING},
	{ERRHRD,	995,	NT_STATUS_CANCELLED},
	{ERRHRD,	997,	NT_STATUS(0x00000103)},
	{ERRHRD,	998,	NT_STATUS_ACCESS_VIOLATION},
	{ERRHRD,	999,	NT_STATUS_IN_PAGE_ERROR},
	{ERRHRD,	1001,	NT_STATUS_BAD_INITIAL_STACK},
	{ERRHRD,	1005,	NT_STATUS_UNRECOGNIZED_VOLUME},
	{ERRHRD,	1006,	NT_STATUS_FILE_INVALID},
	{ERRHRD,	1007,	NT_STATUS_FULLSCREEN_MODE},
	{ERRHRD,	1008,	NT_STATUS_NO_TOKEN},
	{ERRHRD,	1009,	NT_STATUS_REGISTRY_CORRUPT},
	{ERRHRD,	1016,	NT_STATUS_REGISTRY_IO_FAILED},
	{ERRHRD,	1017,	NT_STATUS_NOT_REGISTRY_FILE},
	{ERRHRD,	1018,	NT_STATUS_KEY_DELETED},
	{ERRHRD,	1019,	NT_STATUS_NO_LOG_SPACE},
	{ERRHRD,	1020,	NT_STATUS_KEY_HAS_CHILDREN},
	{ERRHRD,	1021,	NT_STATUS_CHILD_MUST_BE_VOLATILE},
	{ERRHRD,	1022,	NT_STATUS(0x0000010c)},
};

/* errmap NTSTATUS->Win32 */
static const struct {
	NTSTATUS ntstatus;
	WERROR werror;
} ntstatus_to_werror_map[] = {
	{NT_STATUS(0x103), W_ERROR(0x3e5)},
	{NT_STATUS(0x105), W_ERROR(0xea)},
	{NT_STATUS(0x106), W_ERROR(0x514)},
	{NT_STATUS(0x107), W_ERROR(0x515)},
	{NT_STATUS(0x10c), W_ERROR(0x3fe)},
	{NT_STATUS(0x10d), W_ERROR(0x516)},
	{NT_STATUS(0x121), W_ERROR(0x2009)},
	{NT_STATUS(0xc0000001), W_ERROR(0x1f)},
	{NT_STATUS(0xc0000002), W_ERROR(0x1)},
	{NT_STATUS(0xc0000003), W_ERROR(0x57)},
	{NT_STATUS(0xc0000004), W_ERROR(0x18)},
	{NT_STATUS(0xc0000005), W_ERROR(0x3e6)},
	{NT_STATUS(0xc0000006), W_ERROR(0x3e7)},
	{NT_STATUS(0xc0000007), W_ERROR(0x5ae)},
	{NT_STATUS(0xc0000008), W_ERROR(0x6)},
	{NT_STATUS(0xc0000009), W_ERROR(0x3e9)},
	{NT_STATUS(0xc000000a), W_ERROR(0xc1)},
	{NT_STATUS(0xc000000b), W_ERROR(0x57)},
	{NT_STATUS(0xc000000d), W_ERROR(0x57)},
	{NT_STATUS(0xc000000e), W_ERROR(0x2)},
	{NT_STATUS(0xc000000f), W_ERROR(0x2)},
	{NT_STATUS(0xc0000010), W_ERROR(0x1)},
	{NT_STATUS(0xc0000011), W_ERROR(0x26)},
	{NT_STATUS(0xc0000012), W_ERROR(0x22)},
	{NT_STATUS(0xc0000013), W_ERROR(0x15)},
	{NT_STATUS(0xc0000014), W_ERROR(0x6f9)},
	{NT_STATUS(0xc0000015), W_ERROR(0x1b)},
	{NT_STATUS(0xc0000016), W_ERROR(0xea)},
	{NT_STATUS(0xc0000017), W_ERROR(0x8)},
	{NT_STATUS(0xc0000018), W_ERROR(0x1e7)},
	{NT_STATUS(0xc0000019), W_ERROR(0x1e7)},
	{NT_STATUS(0xc000001a), W_ERROR(0x57)},
	{NT_STATUS(0xc000001b), W_ERROR(0x57)},
	{NT_STATUS(0xc000001c), W_ERROR(0x1)},
	{NT_STATUS(0xc000001d), W_ERROR(0xc000001d)},
	{NT_STATUS(0xc000001e), W_ERROR(0x5)},
	{NT_STATUS(0xc000001f), W_ERROR(0x5)},
	{NT_STATUS(0xc0000020), W_ERROR(0xc1)},
	{NT_STATUS(0xc0000021), W_ERROR(0x5)},
	{NT_STATUS(0xc0000022), W_ERROR(0x5)},
	{NT_STATUS(0xc0000023), W_ERROR(0x7a)},
	{NT_STATUS(0xc0000024), W_ERROR(0x6)},
	{NT_STATUS(0xc0000025), W_ERROR(0xc0000025)},
	{NT_STATUS(0xc0000026), W_ERROR(0xc0000026)},
	{NT_STATUS(0xc000002a), W_ERROR(0x9e)},
	{NT_STATUS(0xc000002b), W_ERROR(0xc000002b)},
	{NT_STATUS(0xc000002c), W_ERROR(0x1e7)},
	{NT_STATUS(0xc000002d), W_ERROR(0x1e7)},
	{NT_STATUS(0xc0000030), W_ERROR(0x57)},
	{NT_STATUS(0xc0000032), W_ERROR(0x571)},
	{NT_STATUS(0xc0000033), W_ERROR(0x7b)},
	{NT_STATUS(0xc0000034), W_ERROR(0x2)},
	{NT_STATUS(0xc0000035), W_ERROR(0xb7)},
	{NT_STATUS(0xc0000037), W_ERROR(0x6)},
	{NT_STATUS(0xc0000039), W_ERROR(0xa1)},
	{NT_STATUS(0xc000003a), W_ERROR(0x3)},
	{NT_STATUS(0xc000003b), W_ERROR(0xa1)},
	{NT_STATUS(0xc000003c), W_ERROR(0x45d)},
	{NT_STATUS(0xc000003d), W_ERROR(0x45d)},
	{NT_STATUS(0xc000003e), W_ERROR(0x17)},
	{NT_STATUS(0xc000003f), W_ERROR(0x17)},
	{NT_STATUS(0xc0000040), W_ERROR(0x8)},
	{NT_STATUS(0xc0000041), W_ERROR(0x5)},
	{NT_STATUS(0xc0000042), W_ERROR(0x6)},
	{NT_STATUS(0xc0000043), W_ERROR(0x20)},
	{NT_STATUS(0xc0000044), W_ERROR(0x718)},
	{NT_STATUS(0xc0000045), W_ERROR(0x57)},
	{NT_STATUS(0xc0000046), W_ERROR(0x120)},
	{NT_STATUS(0xc0000047), W_ERROR(0x12a)},
	{NT_STATUS(0xc0000048), W_ERROR(0x57)},
	{NT_STATUS(0xc0000049), W_ERROR(0x57)},
	{NT_STATUS(0xc000004a), W_ERROR(0x9c)},
	{NT_STATUS(0xc000004b), W_ERROR(0x5)},
	{NT_STATUS(0xc000004c), W_ERROR(0x57)},
	{NT_STATUS(0xc000004d), W_ERROR(0x57)},
	{NT_STATUS(0xc000004e), W_ERROR(0x57)},
	{NT_STATUS(0xc000004f), W_ERROR(0x11a)},
	{NT_STATUS(0xc0000050), W_ERROR(0xff)},
	{NT_STATUS(0xc0000051), W_ERROR(0x570)},
	{NT_STATUS(0xc0000052), W_ERROR(0x570)},
	{NT_STATUS(0xc0000053), W_ERROR(0x570)},
	{NT_STATUS(0xc0000054), W_ERROR(0x21)},
	{NT_STATUS(0xc0000055), W_ERROR(0x21)},
	{NT_STATUS(0xc0000056), W_ERROR(0x5)},
	{NT_STATUS(0xc0000057), W_ERROR(0x32)},
	{NT_STATUS(0xc0000058), W_ERROR(0x519)},
	{NT_STATUS(0xc0000059), W_ERROR(0x51a)},
	{NT_STATUS(0xc000005a), W_ERROR(0x51b)},
	{NT_STATUS(0xc000005b), W_ERROR(0x51c)},
	{NT_STATUS(0xc000005c), W_ERROR(0x51d)},
	{NT_STATUS(0xc000005d), W_ERROR(0x51e)},
	{NT_STATUS(0xc000005e), W_ERROR(0x51f)},
	{NT_STATUS(0xc000005f), W_ERROR(0x520)},
	{NT_STATUS(0xc0000060), W_ERROR(0x521)},
	{NT_STATUS(0xc0000061), W_ERROR(0x522)},
	{NT_STATUS(0xc0000062), W_ERROR(0x523)},
	{NT_STATUS(0xc0000063), W_ERROR(0x524)},
	{NT_STATUS(0xc0000064), W_ERROR(0x525)},
	{NT_STATUS(0xc0000065), W_ERROR(0x526)},
	{NT_STATUS(0xc0000066), W_ERROR(0x527)},
	{NT_STATUS(0xc0000067), W_ERROR(0x528)},
	{NT_STATUS(0xc0000068), W_ERROR(0x529)},
	{NT_STATUS(0xc0000069), W_ERROR(0x52a)},
	{NT_STATUS(0xc000006a), W_ERROR(0x56)},
	{NT_STATUS(0xc000006b), W_ERROR(0x52c)},
	{NT_STATUS(0xc000006c), W_ERROR(0x52d)},
	{NT_STATUS(0xc000006d), W_ERROR(0x52e)},
	{NT_STATUS(0xc000006e), W_ERROR(0x52f)},
	{NT_STATUS(0xc000006f), W_ERROR(0x530)},
	{NT_STATUS(0xc0000070), W_ERROR(0x531)},
	{NT_STATUS(0xc0000071), W_ERROR(0x532)},
	{NT_STATUS(0xc0000072), W_ERROR(0x533)},
	{NT_STATUS(0xc0000073), W_ERROR(0x534)},
	{NT_STATUS(0xc0000074), W_ERROR(0x535)},
	{NT_STATUS(0xc0000075), W_ERROR(0x536)},
	{NT_STATUS(0xc0000076), W_ERROR(0x537)},
	{NT_STATUS(0xc0000077), W_ERROR(0x538)},
	{NT_STATUS(0xc0000078), W_ERROR(0x539)},
	{NT_STATUS(0xc0000079), W_ERROR(0x53a)},
	{NT_STATUS(0xc000007a), W_ERROR(0x7f)},
	{NT_STATUS(0xc000007b), W_ERROR(0xc1)},
	{NT_STATUS(0xc000007c), W_ERROR(0x3f0)},
	{NT_STATUS(0xc000007d), W_ERROR(0x53c)},
	{NT_STATUS(0xc000007e), W_ERROR(0x9e)},
	{NT_STATUS(0xc000007f), W_ERROR(0x70)},
	{NT_STATUS(0xc0000080), W_ERROR(0x53d)},
	{NT_STATUS(0xc0000081), W_ERROR(0x53e)},
	{NT_STATUS(0xc0000082), W_ERROR(0x44)},
	{NT_STATUS(0xc0000083), W_ERROR(0x103)},
	{NT_STATUS(0xc0000084), W_ERROR(0x53f)},
	{NT_STATUS(0xc0000085), W_ERROR(0x103)},
	{NT_STATUS(0xc0000086), W_ERROR(0x9a)},
	{NT_STATUS(0xc0000087), W_ERROR(0xe)},
	{NT_STATUS(0xc0000088), W_ERROR(0x1e7)},
	{NT_STATUS(0xc0000089), W_ERROR(0x714)},
	{NT_STATUS(0xc000008a), W_ERROR(0x715)},
	{NT_STATUS(0xc000008b), W_ERROR(0x716)},
	{NT_STATUS(0xc000008c), W_ERROR(0xc000008c)},
	{NT_STATUS(0xc000008d), W_ERROR(0xc000008d)},
	{NT_STATUS(0xc000008e), W_ERROR(0xc000008e)},
	{NT_STATUS(0xc000008f), W_ERROR(0xc000008f)},
	{NT_STATUS(0xc0000090), W_ERROR(0xc0000090)},
	{NT_STATUS(0xc0000091), W_ERROR(0xc0000091)},
	{NT_STATUS(0xc0000092), W_ERROR(0xc0000092)},
	{NT_STATUS(0xc0000093), W_ERROR(0xc0000093)},
	{NT_STATUS(0xc0000094), W_ERROR(0xc0000094)},
	{NT_STATUS(0xc0000095), W_ERROR(0x216)},
	{NT_STATUS(0xc0000096), W_ERROR(0xc0000096)},
	{NT_STATUS(0xc0000097), W_ERROR(0x8)},
	{NT_STATUS(0xc0000098), W_ERROR(0x3ee)},
	{NT_STATUS(0xc0000099), W_ERROR(0x540)},
	{NT_STATUS(0xc000009a), W_ERROR(0x5aa)},
	{NT_STATUS(0xc000009b), W_ERROR(0x3)},
	{NT_STATUS(0xc000009c), W_ERROR(0x17)},
	{NT_STATUS(0xc000009d), W_ERROR(0x48f)},
	{NT_STATUS(0xc000009e), W_ERROR(0x15)},
	{NT_STATUS(0xc000009f), W_ERROR(0x1e7)},
	{NT_STATUS(0xc00000a0), W_ERROR(0x1e7)},
	{NT_STATUS(0xc00000a1), W_ERROR(0x5ad)},
	{NT_STATUS(0xc00000a2), W_ERROR(0x13)},
	{NT_STATUS(0xc00000a3), W_ERROR(0x15)},
	{NT_STATUS(0xc00000a4), W_ERROR(0x541)},
	{NT_STATUS(0xc00000a5), W_ERROR(0x542)},
	{NT_STATUS(0xc00000a6), W_ERROR(0x543)},
	{NT_STATUS(0xc00000a7), W_ERROR(0x544)},
	{NT_STATUS(0xc00000a8), W_ERROR(0x545)},
	{NT_STATUS(0xc00000a9), W_ERROR(0x57)},
	{NT_STATUS(0xc00000ab), W_ERROR(0xe7)},
	{NT_STATUS(0xc00000ac), W_ERROR(0xe7)},
	{NT_STATUS(0xc00000ad), W_ERROR(0xe6)},
	{NT_STATUS(0xc00000ae), W_ERROR(0xe7)},
	{NT_STATUS(0xc00000af), W_ERROR(0x1)},
	{NT_STATUS(0xc00000b0), W_ERROR(0xe9)},
	{NT_STATUS(0xc00000b1), W_ERROR(0xe8)},
	{NT_STATUS(0xc00000b2), W_ERROR(0x217)},
	{NT_STATUS(0xc00000b3), W_ERROR(0x218)},
	{NT_STATUS(0xc00000b4), W_ERROR(0xe6)},
	{NT_STATUS(0xc00000b5), W_ERROR(0x79)},
	{NT_STATUS(0xc00000b6), W_ERROR(0x26)},
	{NT_STATUS(0xc00000ba), W_ERROR(0x5)},
	{NT_STATUS(0xc00000bb), W_ERROR(0x32)},
	{NT_STATUS(0xc00000bc), W_ERROR(0x33)},
	{NT_STATUS(0xc00000bd), W_ERROR(0x34)},
	{NT_STATUS(0xc00000be), W_ERROR(0x35)},
	{NT_STATUS(0xc00000bf), W_ERROR(0x36)},
	{NT_STATUS(0xc00000c0), W_ERROR(0x37)},
	{NT_STATUS(0xc00000c1), W_ERROR(0x38)},
	{NT_STATUS(0xc00000c2), W_ERROR(0x39)},
	{NT_STATUS(0xc00000c3), W_ERROR(0x3a)},
	{NT_STATUS(0xc00000c4), W_ERROR(0x3b)},
	{NT_STATUS(0xc00000c5), W_ERROR(0x3c)},
	{NT_STATUS(0xc00000c6), W_ERROR(0x3d)},
	{NT_STATUS(0xc00000c7), W_ERROR(0x3e)},
	{NT_STATUS(0xc00000c8), W_ERROR(0x3f)},
	{NT_STATUS(0xc00000c9), W_ERROR(0x40)},
	{NT_STATUS(0xc00000ca), W_ERROR(0x41)},
	{NT_STATUS(0xc00000cb), W_ERROR(0x42)},
	{NT_STATUS(0xc00000cc), W_ERROR(0x43)},
	{NT_STATUS(0xc00000cd), W_ERROR(0x44)},
	{NT_STATUS(0xc00000ce), W_ERROR(0x45)},
	{NT_STATUS(0xc00000cf), W_ERROR(0x46)},
	{NT_STATUS(0xc00000d0), W_ERROR(0x47)},
	{NT_STATUS(0xc00000d1), W_ERROR(0x48)},
	{NT_STATUS(0xc00000d2), W_ERROR(0x58)},
	{NT_STATUS(0xc00000d4), W_ERROR(0x11)},
	{NT_STATUS(0xc00000d5), W_ERROR(0x5)},
	{NT_STATUS(0xc00000d6), W_ERROR(0xf0)},
	{NT_STATUS(0xc00000d7), W_ERROR(0x546)},
	{NT_STATUS(0xc00000d9), W_ERROR(0xe8)},
	{NT_STATUS(0xc00000da), W_ERROR(0x547)},
	{NT_STATUS(0xc00000dc), W_ERROR(0x548)},
	{NT_STATUS(0xc00000dd), W_ERROR(0x549)},
	{NT_STATUS(0xc00000de), W_ERROR(0x54a)},
	{NT_STATUS(0xc00000df), W_ERROR(0x54b)},
	{NT_STATUS(0xc00000e0), W_ERROR(0x54c)},
	{NT_STATUS(0xc00000e1), W_ERROR(0x54d)},
	{NT_STATUS(0xc00000e2), W_ERROR(0x12c)},
	{NT_STATUS(0xc00000e3), W_ERROR(0x12d)},
	{NT_STATUS(0xc00000e4), W_ERROR(0x54e)},
	{NT_STATUS(0xc00000e5), W_ERROR(0x54f)},
	{NT_STATUS(0xc00000e6), W_ERROR(0x550)},
	{NT_STATUS(0xc00000e7), W_ERROR(0x551)},
	{NT_STATUS(0xc00000e8), W_ERROR(0x6f8)},
	{NT_STATUS(0xc00000ed), W_ERROR(0x552)},
	{NT_STATUS(0xc00000ee), W_ERROR(0x553)},
	{NT_STATUS(0xc00000ef), W_ERROR(0x57)},
	{NT_STATUS(0xc00000f0), W_ERROR(0x57)},
	{NT_STATUS(0xc00000f1), W_ERROR(0x57)},
	{NT_STATUS(0xc00000f2), W_ERROR(0x57)},
	{NT_STATUS(0xc00000f3), W_ERROR(0x57)},
	{NT_STATUS(0xc00000f4), W_ERROR(0x57)},
	{NT_STATUS(0xc00000f5), W_ERROR(0x57)},
	{NT_STATUS(0xc00000f6), W_ERROR(0x57)},
	{NT_STATUS(0xc00000f7), W_ERROR(0x57)},
	{NT_STATUS(0xc00000f8), W_ERROR(0x57)},
	{NT_STATUS(0xc00000f9), W_ERROR(0x57)},
	{NT_STATUS(0xc00000fa), W_ERROR(0x57)},
	{NT_STATUS(0xc00000fb), W_ERROR(0x3)},
	{NT_STATUS(0xc00000fd), W_ERROR(0x3e9)},
	{NT_STATUS(0xc00000fe), W_ERROR(0x554)},
	{NT_STATUS(0xc0000100), W_ERROR(0xcb)},
	{NT_STATUS(0xc0000101), W_ERROR(0x91)},
	{NT_STATUS(0xc0000102), W_ERROR(0x570)},
	{NT_STATUS(0xc0000103), W_ERROR(0x10b)},
	{NT_STATUS(0xc0000104), W_ERROR(0x555)},
	{NT_STATUS(0xc0000105), W_ERROR(0x556)},
	{NT_STATUS(0xc0000106), W_ERROR(0xce)},
	{NT_STATUS(0xc0000107), W_ERROR(0x961)},
	{NT_STATUS(0xc0000108), W_ERROR(0x964)},
	{NT_STATUS(0xc000010a), W_ERROR(0x5)},
	{NT_STATUS(0xc000010b), W_ERROR(0x557)},
	{NT_STATUS(0xc000010d), W_ERROR(0x558)},
	{NT_STATUS(0xc000010e), W_ERROR(0x420)},
	{NT_STATUS(0xc0000117), W_ERROR(0x5a4)},
	{NT_STATUS(0xc000011b), W_ERROR(0xc1)},
	{NT_STATUS(0xc000011c), W_ERROR(0x559)},
	{NT_STATUS(0xc000011d), W_ERROR(0x55a)},
	{NT_STATUS(0xc000011e), W_ERROR(0x3ee)},
	{NT_STATUS(0xc000011f), W_ERROR(0x4)},
	{NT_STATUS(0xc0000120), W_ERROR(0x3e3)},
	{NT_STATUS(0xc0000121), W_ERROR(0x5)},
	{NT_STATUS(0xc0000122), W_ERROR(0x4ba)},
	{NT_STATUS(0xc0000123), W_ERROR(0x5)},
	{NT_STATUS(0xc0000124), W_ERROR(0x55b)},
	{NT_STATUS(0xc0000125), W_ERROR(0x55c)},
	{NT_STATUS(0xc0000126), W_ERROR(0x55d)},
	{NT_STATUS(0xc0000127), W_ERROR(0x55e)},
	{NT_STATUS(0xc0000128), W_ERROR(0x6)},
	{NT_STATUS(0xc000012b), W_ERROR(0x55f)},
	{NT_STATUS(0xc000012d), W_ERROR(0x5af)},
	{NT_STATUS(0xc000012e), W_ERROR(0xc1)},
	{NT_STATUS(0xc000012f), W_ERROR(0xc1)},
	{NT_STATUS(0xc0000130), W_ERROR(0xc1)},
	{NT_STATUS(0xc0000131), W_ERROR(0xc1)},
	{NT_STATUS(0xc0000133), W_ERROR(0x576)},
	{NT_STATUS(0xc0000135), W_ERROR(0x7e)},
	{NT_STATUS(0xc0000138), W_ERROR(0xb6)},
	{NT_STATUS(0xc0000139), W_ERROR(0x7f)},
	{NT_STATUS(0xc000013b), W_ERROR(0x40)},
	{NT_STATUS(0xc000013c), W_ERROR(0x40)},
	{NT_STATUS(0xc000013d), W_ERROR(0x33)},
	{NT_STATUS(0xc000013e), W_ERROR(0x3b)},
	{NT_STATUS(0xc000013f), W_ERROR(0x3b)},
	{NT_STATUS(0xc0000140), W_ERROR(0x3b)},
	{NT_STATUS(0xc0000141), W_ERROR(0x3b)},
	{NT_STATUS(0xc0000142), W_ERROR(0x45a)},
	{NT_STATUS(0xc0000148), W_ERROR(0x7c)},
	{NT_STATUS(0xc0000149), W_ERROR(0x56)},
	{NT_STATUS(0xc000014b), W_ERROR(0x6d)},
	{NT_STATUS(0xc000014c), W_ERROR(0x3f1)},
	{NT_STATUS(0xc000014d), W_ERROR(0x3f8)},
	{NT_STATUS(0xc000014f), W_ERROR(0x3ed)},
	{NT_STATUS(0xc0000150), W_ERROR(0x45e)},
	{NT_STATUS(0xc0000151), W_ERROR(0x560)},
	{NT_STATUS(0xc0000152), W_ERROR(0x561)},
	{NT_STATUS(0xc0000153), W_ERROR(0x562)},
	{NT_STATUS(0xc0000154), W_ERROR(0x563)},
	{NT_STATUS(0xc0000155), W_ERROR(0x564)},
	{NT_STATUS(0xc0000156), W_ERROR(0x565)},
	{NT_STATUS(0xc0000157), W_ERROR(0x566)},
	{NT_STATUS(0xc0000158), W_ERROR(0x567)},
	{NT_STATUS(0xc0000159), W_ERROR(0x3ef)},
	{NT_STATUS(0xc000015a), W_ERROR(0x568)},
	{NT_STATUS(0xc000015b), W_ERROR(0x569)},
	{NT_STATUS(0xc000015c), W_ERROR(0x3f9)},
	{NT_STATUS(0xc000015d), W_ERROR(0x56a)},
	{NT_STATUS(0xc000015f), W_ERROR(0x45d)},
	{NT_STATUS(0xc0000162), W_ERROR(0x459)},
	{NT_STATUS(0xc0000165), W_ERROR(0x462)},
	{NT_STATUS(0xc0000166), W_ERROR(0x463)},
	{NT_STATUS(0xc0000167), W_ERROR(0x464)},
	{NT_STATUS(0xc0000168), W_ERROR(0x465)},
	{NT_STATUS(0xc0000169), W_ERROR(0x466)},
	{NT_STATUS(0xc000016a), W_ERROR(0x467)},
	{NT_STATUS(0xc000016b), W_ERROR(0x468)},
	{NT_STATUS(0xc000016c), W_ERROR(0x45f)},
	{NT_STATUS(0xc000016d), W_ERROR(0x45d)},
	{NT_STATUS(0xc0000172), W_ERROR(0x451)},
	{NT_STATUS(0xc0000173), W_ERROR(0x452)},
	{NT_STATUS(0xc0000174), W_ERROR(0x453)},
	{NT_STATUS(0xc0000175), W_ERROR(0x454)},
	{NT_STATUS(0xc0000176), W_ERROR(0x455)},
	{NT_STATUS(0xc0000177), W_ERROR(0x469)},
	{NT_STATUS(0xc0000178), W_ERROR(0x458)},
	{NT_STATUS(0xc000017a), W_ERROR(0x56b)},
	{NT_STATUS(0xc000017b), W_ERROR(0x56c)},
	{NT_STATUS(0xc000017c), W_ERROR(0x3fa)},
	{NT_STATUS(0xc000017d), W_ERROR(0x3fb)},
	{NT_STATUS(0xc000017e), W_ERROR(0x56d)},
	{NT_STATUS(0xc000017f), W_ERROR(0x56e)},
	{NT_STATUS(0xc0000180), W_ERROR(0x3fc)},
	{NT_STATUS(0xc0000181), W_ERROR(0x3fd)},
	{NT_STATUS(0xc0000182), W_ERROR(0x57)},
	{NT_STATUS(0xc0000183), W_ERROR(0x45d)},
	{NT_STATUS(0xc0000184), W_ERROR(0x16)},
	{NT_STATUS(0xc0000185), W_ERROR(0x45d)},
	{NT_STATUS(0xc0000186), W_ERROR(0x45d)},
	{NT_STATUS(0xc0000188), W_ERROR(0x5de)},
	{NT_STATUS(0xc0000189), W_ERROR(0x13)},
	{NT_STATUS(0xc000018a), W_ERROR(0x6fa)},
	{NT_STATUS(0xc000018b), W_ERROR(0x6fb)},
	{NT_STATUS(0xc000018c), W_ERROR(0x6fc)},
	{NT_STATUS(0xc000018d), W_ERROR(0x6fd)},
	{NT_STATUS(0xc000018e), W_ERROR(0x5dc)},
	{NT_STATUS(0xc000018f), W_ERROR(0x5dd)},
	{NT_STATUS(0xc0000190), W_ERROR(0x6fe)},
	{NT_STATUS(0xc0000192), W_ERROR(0x700)},
	{NT_STATUS(0xc0000193), W_ERROR(0x701)},
	{NT_STATUS(0xc0000194), W_ERROR(0x46b)},
	{NT_STATUS(0xc0000195), W_ERROR(0x4c3)},
	{NT_STATUS(0xc0000196), W_ERROR(0x4c4)},
	{NT_STATUS(0xc0000197), W_ERROR(0x5df)},
	{NT_STATUS(0xc0000198), W_ERROR(0x70f)},
	{NT_STATUS(0xc0000199), W_ERROR(0x710)},
	{NT_STATUS(0xc000019a), W_ERROR(0x711)},
	{NT_STATUS(0xc000019b), W_ERROR(0x712)},
	{NT_STATUS(0xc0000202), W_ERROR(0x572)},
	{NT_STATUS(0xc0000203), W_ERROR(0x3b)},
	{NT_STATUS(0xc0000204), W_ERROR(0x717)},
	{NT_STATUS(0xc0000205), W_ERROR(0x46a)},
	{NT_STATUS(0xc0000206), W_ERROR(0x6f8)},
	{NT_STATUS(0xc0000207), W_ERROR(0x4be)},
	{NT_STATUS(0xc0000208), W_ERROR(0x4be)},
	{NT_STATUS(0xc0000209), W_ERROR(0x44)},
	{NT_STATUS(0xc000020a), W_ERROR(0x34)},
	{NT_STATUS(0xc000020b), W_ERROR(0x40)},
	{NT_STATUS(0xc000020c), W_ERROR(0x40)},
	{NT_STATUS(0xc000020d), W_ERROR(0x40)},
	{NT_STATUS(0xc000020e), W_ERROR(0x44)},
	{NT_STATUS(0xc000020f), W_ERROR(0x3b)},
	{NT_STATUS(0xc0000210), W_ERROR(0x3b)},
	{NT_STATUS(0xc0000211), W_ERROR(0x3b)},
	{NT_STATUS(0xc0000212), W_ERROR(0x3b)},
	{NT_STATUS(0xc0000213), W_ERROR(0x3b)},
	{NT_STATUS(0xc0000214), W_ERROR(0x3b)},
	{NT_STATUS(0xc0000215), W_ERROR(0x3b)},
	{NT_STATUS(0xc0000216), W_ERROR(0x32)},
	{NT_STATUS(0xc0000217), W_ERROR(0x32)},
	{NT_STATUS(0xc000021c), W_ERROR(0x17e6)},
	{NT_STATUS(0xc0000220), W_ERROR(0x46c)},
	{NT_STATUS(0xc0000221), W_ERROR(0xc1)},
	{NT_STATUS(0xc0000224), W_ERROR(0x773)},
	{NT_STATUS(0xc0000225), W_ERROR(0x490)},
	{NT_STATUS(0xc000022a), W_ERROR(0xc000022a)},
	{NT_STATUS(0xc000022b), W_ERROR(0xc000022b)},
	{NT_STATUS(0xc000022d), W_ERROR(0x4d5)},
	{NT_STATUS(0xc0000230), W_ERROR(0x492)},
	{NT_STATUS(0xc0000233), W_ERROR(0x774)},
	{NT_STATUS(0xc0000234), W_ERROR(0x775)},
	{NT_STATUS(0xc0000235), W_ERROR(0x6)},
	{NT_STATUS(0xc0000236), W_ERROR(0x4c9)},
	{NT_STATUS(0xc0000237), W_ERROR(0x4ca)},
	{NT_STATUS(0xc0000238), W_ERROR(0x4cb)},
	{NT_STATUS(0xc0000239), W_ERROR(0x4cc)},
	{NT_STATUS(0xc000023a), W_ERROR(0x4cd)},
	{NT_STATUS(0xc000023b), W_ERROR(0x4ce)},
	{NT_STATUS(0xc000023c), W_ERROR(0x4cf)},
	{NT_STATUS(0xc000023d), W_ERROR(0x4d0)},
	{NT_STATUS(0xc000023e), W_ERROR(0x4d1)},
	{NT_STATUS(0xc000023f), W_ERROR(0x4d2)},
	{NT_STATUS(0xc0000240), W_ERROR(0x4d3)},
	{NT_STATUS(0xc0000241), W_ERROR(0x4d4)},
	{NT_STATUS(0xc0000243), W_ERROR(0x4c8)},
	{NT_STATUS(0xc0000246), W_ERROR(0x4d6)},
	{NT_STATUS(0xc0000247), W_ERROR(0x4d7)},
	{NT_STATUS(0xc0000248), W_ERROR(0x4d8)},
	{NT_STATUS(0xc0000249), W_ERROR(0xc1)},
	{NT_STATUS(0xc0000253), W_ERROR(0x54f)},
	{NT_STATUS(0xc0000257), W_ERROR(0x4d0)},
	{NT_STATUS(0xc0000259), W_ERROR(0x573)},
	{NT_STATUS(0xc000025e), W_ERROR(0x422)},
	{NT_STATUS(0xc0000262), W_ERROR(0xb6)},
	{NT_STATUS(0xc0000263), W_ERROR(0x7f)},
	{NT_STATUS(0xc0000264), W_ERROR(0x120)},
	{NT_STATUS(0xc0000265), W_ERROR(0x476)},
	{NT_STATUS(0xc0000267), W_ERROR(0x10fe)},
	{NT_STATUS(0xc000026c), W_ERROR(0x7d1)},
	{NT_STATUS(0xc000026d), W_ERROR(0x4b1)},
	{NT_STATUS(0xc000026e), W_ERROR(0x15)},
	{NT_STATUS(0xc0000272), W_ERROR(0x491)},
	{NT_STATUS(0xc0000275), W_ERROR(0x1126)},
	{NT_STATUS(0xc0000276), W_ERROR(0x1129)},
	{NT_STATUS(0xc0000277), W_ERROR(0x112a)},
	{NT_STATUS(0xc0000278), W_ERROR(0x1128)},
	{NT_STATUS(0xc0000279), W_ERROR(0x780)},
	{NT_STATUS(0xc0000280), W_ERROR(0x781)},
	{NT_STATUS(0xc0000281), W_ERROR(0xa1)},
	{NT_STATUS(0xc0000283), W_ERROR(0x488)},
	{NT_STATUS(0xc0000284), W_ERROR(0x489)},
	{NT_STATUS(0xc0000285), W_ERROR(0x48a)},
	{NT_STATUS(0xc0000286), W_ERROR(0x48b)},
	{NT_STATUS(0xc0000287), W_ERROR(0x48c)},
	{NT_STATUS(0xc000028a), W_ERROR(0x5)},
	{NT_STATUS(0xc000028b), W_ERROR(0x5)},
	{NT_STATUS(0xc000028d), W_ERROR(0x5)},
	{NT_STATUS(0xc000028e), W_ERROR(0x5)},
	{NT_STATUS(0xc000028f), W_ERROR(0x5)},
	{NT_STATUS(0xc0000290), W_ERROR(0x5)},
	{NT_STATUS(0xc0000291), W_ERROR(0x1777)},
	{NT_STATUS(0xc0000292), W_ERROR(0x1778)},
	{NT_STATUS(0xc0000293), W_ERROR(0x1772)},
	{NT_STATUS(0xc0000295), W_ERROR(0x1068)},
	{NT_STATUS(0xc0000296), W_ERROR(0x1069)},
	{NT_STATUS(0xc0000297), W_ERROR(0x106a)},
	{NT_STATUS(0xc0000298), W_ERROR(0x106b)},
	{NT_STATUS(0xc0000299), W_ERROR(0x201a)},
	{NT_STATUS(0xc000029a), W_ERROR(0x201b)},
	{NT_STATUS(0xc000029b), W_ERROR(0x201c)},
	{NT_STATUS(0xc000029c), W_ERROR(0x1)},
	{NT_STATUS(0xc000029d), W_ERROR(0x10ff)},
	{NT_STATUS(0xc000029e), W_ERROR(0x1100)},
	{NT_STATUS(0xc000029f), W_ERROR(0x494)},
	{NT_STATUS(0xc00002a1), W_ERROR(0x200a)},
	{NT_STATUS(0xc00002a2), W_ERROR(0x200b)},
	{NT_STATUS(0xc00002a3), W_ERROR(0x200c)},
	{NT_STATUS(0xc00002a4), W_ERROR(0x200d)},
	{NT_STATUS(0xc00002a5), W_ERROR(0x200e)},
	{NT_STATUS(0xc00002a6), W_ERROR(0x200f)},
	{NT_STATUS(0xc00002a7), W_ERROR(0x2010)},
	{NT_STATUS(0xc00002a8), W_ERROR(0x2011)},
	{NT_STATUS(0xc00002a9), W_ERROR(0x2012)},
	{NT_STATUS(0xc00002aa), W_ERROR(0x2013)},
	{NT_STATUS(0xc00002ab), W_ERROR(0x2014)},
	{NT_STATUS(0xc00002ac), W_ERROR(0x2015)},
	{NT_STATUS(0xc00002ad), W_ERROR(0x2016)},
	{NT_STATUS(0xc00002ae), W_ERROR(0x2017)},
	{NT_STATUS(0xc00002af), W_ERROR(0x2018)},
	{NT_STATUS(0xc00002b0), W_ERROR(0x2019)},
	{NT_STATUS(0xc00002b1), W_ERROR(0x211e)},
	{NT_STATUS(0xc00002b2), W_ERROR(0x1127)},
	{NT_STATUS(0xc00002b6), W_ERROR(0x651)},
	{NT_STATUS(0xc00002b7), W_ERROR(0x49a)},
	{NT_STATUS(0xc00002b8), W_ERROR(0x49b)},
	{NT_STATUS(0xc00002c1), W_ERROR(0x2024)},
	{NT_STATUS(0xc00002c3), W_ERROR(0x575)},
	{NT_STATUS(0xc00002c5), W_ERROR(0x3e6)},
	{NT_STATUS(0xc00002c6), W_ERROR(0x1075)},
	{NT_STATUS(0xc00002c7), W_ERROR(0x1076)},
	{NT_STATUS(0xc00002ca), W_ERROR(0x10e8)},
	{NT_STATUS(0xc00002cb), W_ERROR(0x2138)},
	{NT_STATUS(0xc00002cc), W_ERROR(0x4e3)},
	{NT_STATUS(0xc00002cd), W_ERROR(0x2139)},
	{NT_STATUS(0xc00002cf), W_ERROR(0x49d)},
	{NT_STATUS(0xc00002d0), W_ERROR(0x213a)},
	{NT_STATUS(0xc00002d4), W_ERROR(0x2141)},
	{NT_STATUS(0xc00002d5), W_ERROR(0x2142)},
	{NT_STATUS(0xc00002d6), W_ERROR(0x2143)},
	{NT_STATUS(0xc00002d7), W_ERROR(0x2144)},
	{NT_STATUS(0xc00002d8), W_ERROR(0x2145)},
	{NT_STATUS(0xc00002d9), W_ERROR(0x2146)},
	{NT_STATUS(0xc00002da), W_ERROR(0x2147)},
	{NT_STATUS(0xc00002db), W_ERROR(0x2148)},
	{NT_STATUS(0xc00002dc), W_ERROR(0x2149)},
	{NT_STATUS(0xc00002dd), W_ERROR(0x32)},
	{NT_STATUS(0xc00002df), W_ERROR(0x2151)},
	{NT_STATUS(0xc00002e0), W_ERROR(0x2152)},
	{NT_STATUS(0xc00002e1), W_ERROR(0x2153)},
	{NT_STATUS(0xc00002e2), W_ERROR(0x2154)},
	{NT_STATUS(0xc00002e3), W_ERROR(0x215d)},
	{NT_STATUS(0xc00002e4), W_ERROR(0x2163)},
	{NT_STATUS(0xc00002e5), W_ERROR(0x2164)},
	{NT_STATUS(0xc00002e6), W_ERROR(0x2165)},
	{NT_STATUS(0xc00002e7), W_ERROR(0x216d)},
	{NT_STATUS(0xc00002fe), W_ERROR(0x45b)},
	{NT_STATUS(0xc00002ff), W_ERROR(0x4e7)},
	{NT_STATUS(0xc0000300), W_ERROR(0x4e6)},
	{NT_STATUS(0x80000001), W_ERROR(0x80000001)},
	{NT_STATUS(0x80000002), W_ERROR(0x3e6)},
	{NT_STATUS(0x80000003), W_ERROR(0x80000003)},
	{NT_STATUS(0x80000004), W_ERROR(0x80000004)},
	{NT_STATUS(0x80000005), W_ERROR(0xea)},
	{NT_STATUS(0x80000006), W_ERROR(0x12)},
	{NT_STATUS(0x8000000b), W_ERROR(0x56f)},
	{NT_STATUS(0x8000000d), W_ERROR(0x12b)},
	{NT_STATUS(0x8000000e), W_ERROR(0x1c)},
	{NT_STATUS(0x8000000f), W_ERROR(0x15)},
	{NT_STATUS(0x80000010), W_ERROR(0x15)},
	{NT_STATUS(0x80000011), W_ERROR(0xaa)},
	{NT_STATUS(0x80000012), W_ERROR(0x103)},
	{NT_STATUS(0x80000013), W_ERROR(0xfe)},
	{NT_STATUS(0x80000014), W_ERROR(0xff)},
	{NT_STATUS(0x80000015), W_ERROR(0xff)},
	{NT_STATUS(0x80000016), W_ERROR(0x456)},
	{NT_STATUS(0x8000001a), W_ERROR(0x103)},
	{NT_STATUS(0x8000001b), W_ERROR(0x44d)},
	{NT_STATUS(0x8000001c), W_ERROR(0x456)},
	{NT_STATUS(0x8000001d), W_ERROR(0x457)},
	{NT_STATUS(0x8000001e), W_ERROR(0x44c)},
	{NT_STATUS(0x8000001f), W_ERROR(0x44e)},
	{NT_STATUS(0x80000021), W_ERROR(0x44f)},
	{NT_STATUS(0x80000022), W_ERROR(0x450)},
	{NT_STATUS(0x80000025), W_ERROR(0x962)},
	{NT_STATUS(0x80000288), W_ERROR(0x48d)},
	{NT_STATUS(0x80000289), W_ERROR(0x48e)},
	{NT_STATUS_OK, WERR_OK}};


/*****************************************************************************
convert a dos eclas/ecode to a NT status32 code
 *****************************************************************************/
NTSTATUS dos_to_ntstatus(uint8 eclass, uint32 ecode)
{
	int i;
	if (eclass == 0 && ecode == 0) return NT_STATUS_OK;
	for (i=0; NT_STATUS_V(dos_to_ntstatus_map[i].ntstatus); i++) {
		if (eclass == dos_to_ntstatus_map[i].dos_class &&
		    ecode == dos_to_ntstatus_map[i].dos_code) {
			return dos_to_ntstatus_map[i].ntstatus;
		}
	}
	return NT_STATUS_UNSUCCESSFUL;
}


/*****************************************************************************
convert a NT status code to a dos class/code
 *****************************************************************************/
void ntstatus_to_dos(NTSTATUS ntstatus, uint8 *eclass, uint32 *ecode)
{
	int i;
	if (NT_STATUS_IS_OK(ntstatus)) {
		*eclass = 0;
		*ecode = 0;
		return;
	}
	for (i=0; NT_STATUS_V(ntstatus_to_dos_map[i].ntstatus); i++) {
		if (NT_STATUS_V(ntstatus) == 
		    NT_STATUS_V(ntstatus_to_dos_map[i].ntstatus)) {
			*eclass = ntstatus_to_dos_map[i].dos_class;
			*ecode = ntstatus_to_dos_map[i].dos_code;
			return;
		}
	}
	*eclass = ERRHRD;
	*ecode = ERRgeneral;
}


/*****************************************************************************
convert a WERROR to a NT status32 code
 *****************************************************************************/
NTSTATUS werror_to_ntstatus(WERROR error)
{
	int i;
	if (W_ERROR_IS_OK(error)) return NT_STATUS_OK;
	for (i=0; NT_STATUS_V(ntstatus_to_werror_map[i].ntstatus); i++) {
		if (W_ERROR_V(error) == 
		    W_ERROR_V(ntstatus_to_werror_map[i].werror)) {
			return ntstatus_to_werror_map[i].ntstatus;
		}
	}

	/* just guess ... */
	return NT_STATUS(W_ERROR_V(error) | 0xc0000000);
}

/*****************************************************************************
convert a NTSTATUS to a WERROR
 *****************************************************************************/
WERROR ntstatus_to_werror(NTSTATUS error)
{
	int i;
	if (NT_STATUS_IS_OK(error)) return WERR_OK;
	for (i=0; NT_STATUS_V(ntstatus_to_werror_map[i].ntstatus); i++) {
		if (NT_STATUS_V(error) == 
		    NT_STATUS_V(ntstatus_to_werror_map[i].ntstatus)) {
			return ntstatus_to_werror_map[i].werror;
		}
	}

	/* a lame guess */
	return W_ERROR(NT_STATUS_V(error) & 0xffff);
}

/* Mapping between Unix, DOS and NT error numbers */

const struct unix_error_map unix_dos_nt_errmap[] = {
	{ EPERM, ERRDOS, ERRnoaccess, NT_STATUS_ACCESS_DENIED },
	{ EACCES, ERRDOS, ERRnoaccess, NT_STATUS_ACCESS_DENIED },
	{ ENOENT, ERRDOS, ERRbadfile, NT_STATUS_OBJECT_NAME_NOT_FOUND },
	{ ENOTDIR, ERRDOS, ERRbadpath,  NT_STATUS_NOT_A_DIRECTORY },
	{ EIO, ERRHRD, ERRgeneral, NT_STATUS_IO_DEVICE_ERROR },
	{ EBADF, ERRSRV, ERRsrverror, NT_STATUS_INVALID_HANDLE },
	{ EINVAL, ERRSRV, ERRsrverror, NT_STATUS_INVALID_HANDLE },
	{ EEXIST, ERRDOS, ERRfilexists, NT_STATUS_OBJECT_NAME_COLLISION},
	{ ENFILE, ERRDOS, ERRnofids, NT_STATUS_TOO_MANY_OPENED_FILES },
	{ EMFILE, ERRDOS, ERRnofids, NT_STATUS_TOO_MANY_OPENED_FILES },
	{ ENOSPC, ERRHRD, ERRdiskfull, NT_STATUS_DISK_FULL },
	{ ENOMEM, ERRDOS, ERRnomem, NT_STATUS_NO_MEMORY },
#ifdef EDQUOT
	{ EDQUOT, ERRHRD, ERRdiskfull, NT_STATUS_DISK_FULL },
#endif
#ifdef ENOTEMPTY
	{ ENOTEMPTY, ERRDOS, ERRnoaccess, NT_STATUS_DIRECTORY_NOT_EMPTY },
#endif
#ifdef EXDEV
	{ EXDEV, ERRDOS, ERRdiffdevice, NT_STATUS_NOT_SAME_DEVICE },
#endif
#ifdef EROFS
	{ EROFS, ERRHRD, ERRnowrite, NT_STATUS_ACCESS_DENIED },
#endif
#ifdef ENAMETOOLONG
	{ ENAMETOOLONG, ERRDOS, 206, NT_STATUS_OBJECT_NAME_INVALID },
#endif
#ifdef EFBIG
	{ EFBIG, ERRHRD, ERRdiskfull, NT_STATUS_DISK_FULL },
#endif
	{ 0, 0, 0, NT_STATUS_OK }
};

/*********************************************************************
 Map an NT error code from a Unix error code.
*********************************************************************/

NTSTATUS map_nt_error_from_unix(int unix_error)
{
	int i = 0;

	if (unix_error == 0)
		return NT_STATUS_OK;

	/* Look through list */
	while(unix_dos_nt_errmap[i].unix_error != 0) {
		if (unix_dos_nt_errmap[i].unix_error == unix_error)
			return unix_dos_nt_errmap[i].nt_error;
		i++;
	}

	/* Default return */
	return NT_STATUS_ACCESS_DENIED;
}
