/*
 * JBoss, Home of Professional Open Source
 * Copyright 2011, Red Hat, Inc. and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jivesoftware.smackx.filetransfer;

import java.util.ArrayList;
import java.util.List;

import org.jivesoftware.smack.PacketListener;
import org.jivesoftware.smack.XMPPConnection;
import org.jivesoftware.smack.filter.AndFilter;
import org.jivesoftware.smack.filter.IQTypeFilter;
import org.jivesoftware.smack.filter.PacketTypeFilter;
import org.jivesoftware.smack.packet.IQ;
import org.jivesoftware.smack.packet.Packet;
import org.jivesoftware.smack.packet.XMPPError;
import org.jivesoftware.smack.util.StringUtils;
import org.jivesoftware.smackx.packet.StreamInitiation;

/**
 * The file transfer manager class handles the sending and recieving of files.
 * To send a file invoke the {@link #createOutgoingFileTransfer(String)} method.
 * <p>
 * And to recieve a file add a file transfer listener to the manager. The
 * listener will notify you when there is a new file transfer request. To create
 * the {@link IncomingFileTransfer} object accept the transfer, or, if the
 * transfer is not desirable reject it.
 * 
 * @author Alexander Wenckus
 * 
 */
public class FileTransferManager {

	private final FileTransferNegotiator fileTransferNegotiator;

	private List listeners;

	private XMPPConnection connection;

	/**
	 * Creates a file transfer manager to initiate and receive file transfers.
	 * 
	 * @param connection
	 *            The XMPPConnection that the file transfers will use.
	 */
	public FileTransferManager(XMPPConnection connection) {
		this.connection = connection;
		this.fileTransferNegotiator = FileTransferNegotiator
				.getInstanceFor(connection);
	}

	/**
	 * Add a file transfer listener to listen to incoming file transfer
	 * requests.
	 * 
	 * @param li
	 *            The listener
	 * @see #removeFileTransferListener(FileTransferListener)
	 * @see FileTransferListener
	 */
	public void addFileTransferListener(final FileTransferListener li) {
		if (listeners == null) {
			initListeners();
		}
		synchronized (this.listeners) {
			listeners.add(li);
		}
	}

	private void initListeners() {
		listeners = new ArrayList();

		connection.addPacketListener(new PacketListener() {
			public void processPacket(Packet packet) {
				fireNewRequest((StreamInitiation) packet);
			}
		}, new AndFilter(new PacketTypeFilter(StreamInitiation.class),
				new IQTypeFilter(IQ.Type.SET)));
	}

	protected void fireNewRequest(StreamInitiation initiation) {
		FileTransferListener[] listeners = null;
		synchronized (this.listeners) {
			listeners = new FileTransferListener[this.listeners.size()];
			this.listeners.toArray(listeners);
		}
		FileTransferRequest request = new FileTransferRequest(this, initiation);
		for (int i = 0; i < listeners.length; i++) {
			listeners[i].fileTransferRequest(request);
		}
	}

	/**
	 * Removes a file transfer listener.
	 * 
	 * @param li
	 *            The file transfer listener to be removed
	 * @see FileTransferListener
	 */
	public void removeFileTransferListener(final FileTransferListener li) {
		if (listeners == null) {
			return;
		}
		synchronized (this.listeners) {
			listeners.remove(li);
		}
	}

	/**
	 * Creates an OutgoingFileTransfer to send a file to another user.
	 * 
	 * @param userID
	 *            The fully qualified jabber ID with resource of the user to
	 *            send the file to.
	 * @return The send file object on which the negotiated transfer can be run.
	 */
	public OutgoingFileTransfer createOutgoingFileTransfer(String userID) {
		if (userID == null || StringUtils.parseName(userID).length() <= 0
				|| StringUtils.parseServer(userID).length() <= 0
				|| StringUtils.parseResource(userID).length() <= 0) {
			throw new IllegalArgumentException(
					"The provided user id was not fully qualified");
		}

		return new OutgoingFileTransfer(connection.getUser(), userID,
				fileTransferNegotiator.getNextStreamID(),
				fileTransferNegotiator);
	}

	/**
	 * When the file transfer request is acceptable, this method should be
	 * invoked. It will create an IncomingFileTransfer which allows the
	 * transmission of the file to procede.
	 * 
	 * @param request
	 *            The remote request that is being accepted.
	 * @return The IncomingFileTransfer which manages the download of the file
	 *         from the transfer initiator.
	 */
	protected IncomingFileTransfer createIncomingFileTransfer(
			FileTransferRequest request) {
		if (request == null) {
			throw new NullPointerException("RecieveRequest cannot be null");
		}

		IncomingFileTransfer transfer = new IncomingFileTransfer(request,
                fileTransferNegotiator);
		transfer.setFileInfo(request.getFileName(), request.getFileSize());

		return transfer;
	}

	protected void rejectIncomingFileTransfer(FileTransferRequest request) {
		StreamInitiation initiation = request.getStreamInitiation();

		IQ rejection = FileTransferNegotiator.createIQ(
				initiation.getPacketID(), initiation.getFrom(), initiation
						.getTo(), IQ.Type.ERROR);
		rejection.setError(new XMPPError(403));
		connection.sendPacket(rejection);
	}
}
