/*
 * JBoss, Home of Professional Open Source
 * Copyright 2011, Red Hat, Inc. and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.mobicents.media.server.impl.resource.audio;

import java.util.ArrayList;
import org.mobicents.media.Buffer;
import org.mobicents.media.Format;
import org.mobicents.media.server.impl.AbstractSource;
import org.mobicents.media.server.spi.dsp.Codec;
import org.mobicents.media.server.spi.dsp.CodecFactory;

/**
 * InbandGenerator generates Inband DTMF Tone only for uncompressed LINEAR
 * codec. After creating instance of InbandGenerator, it needs to be initialized
 * so that all the Tones are generated and kept ready for transmission once
 * start is called.
 * 
 * By default the Tone duration is 80ms. This is suited for Tone Detector who
 * has Tone duration of greater than 40 and less than 80ms. For Tone Detector
 * who's Tone duration is set greater than 80ms may fail to detect Tone
 * generated by InbandGenerator(with duration 80ms). In that case increase the
 * duration here too.
 * 
 * @author Oleg Kulikov
 * @author amit bhayani
 */
public class AudioNoiseGenerator extends AbstractSource  {

    private final static Format[] FORMATS = new Format[]{
        Codec.LINEAR_AUDIO,
        Codec.PCMA,
        Codec.PCMU,
        Codec.GSM,
        Codec.SPEEX,
        Codec.G729
    };

    private byte[] data = new byte[320];
    private Codec codec;

    private final static ArrayList<CodecFactory> codecFactories = new ArrayList();
    static {
        codecFactories.add(new org.mobicents.media.server.impl.dsp.audio.g711.alaw.DecoderFactory());
        codecFactories.add(new org.mobicents.media.server.impl.dsp.audio.g711.alaw.EncoderFactory());

        codecFactories.add(new org.mobicents.media.server.impl.dsp.audio.g711.ulaw.DecoderFactory());
        codecFactories.add(new org.mobicents.media.server.impl.dsp.audio.g711.ulaw.EncoderFactory());

        codecFactories.add(new org.mobicents.media.server.impl.dsp.audio.gsm.DecoderFactory());
        codecFactories.add(new org.mobicents.media.server.impl.dsp.audio.gsm.EncoderFactory());

        codecFactories.add(new org.mobicents.media.server.impl.dsp.audio.speex.DecoderFactory());
        codecFactories.add(new org.mobicents.media.server.impl.dsp.audio.speex.EncoderFactory());

        codecFactories.add(new org.mobicents.media.server.impl.dsp.audio.g729.DecoderFactory());
        codecFactories.add(new org.mobicents.media.server.impl.dsp.audio.g729.EncoderFactory());
    }
    
    public AudioNoiseGenerator(String name) {
        super(name);
    }

    @Override
    public void setPreffered(Format fmt) {
        codec = selectCodec(fmt);
        super.setPreffered(fmt);
    }
    
    private Codec selectCodec(Format f) {
        for (CodecFactory factory : codecFactories) {
            if (factory.getSupportedOutputFormat().matches(f) &&
                    factory.getSupportedInputFormat().matches(Codec.LINEAR_AUDIO)) {
                return factory.getCodec();
            }
        }
        return null;
    }
    

    public Format[] getFormats() {
        return FORMATS;
    }

    @Override
    public void evolve(Buffer buffer, long timestamp) {
        buffer.setData(data);
        buffer.setOffset(0);
        buffer.setLength(320);
        buffer.setFormat(Codec.LINEAR_AUDIO);
        buffer.setFlags(Buffer.FLAG_SILENCE);
        
        if (codec != null) {
            codec.process(buffer);
        }
    }

    @Override
    public void stop() {
    }
}
