package org.jboss.ip.dbtool.cliutil;

import org.jboss.ip.dbtool.Main;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.net.ConnectException;
import java.net.HttpURLConnection;
import java.net.URL;

/**
 * Class provides an abstraction to start a JBoss server.
 *
 * @author Alex Creasy <acreasy@redhat.com>
 */
public final class ServerLauncher {

    public static final String EXEC_DIR = "/bin/";
    public static final String STANDALONE_LAUNCHER_FILENAME = "standalone";
    public static final String DOMAIN_LAUNCHER_FILENAME = "domain";
    public static final String WINDOWS_EXTENSION = ".bat";
    public static final String STARNIX_EXTENSION = ".sh";
    public static final String ARGS = "--admin-only";

    private static final Logger logger = LoggerFactory.getLogger(Main.LOGGER_NAME);
    private final File jbossHomeDir;
    private ScriptLauncher jbossLauncher;

    /**
     * @param jbossHomeDir java.io.File pointing to the JBoss server's home directory
     *
     * @throws IllegalArgumentException if jbossHomeDir does not point to a valid directory or
     *                                  testURL is an empty String.
     */
    public ServerLauncher(final File jbossHomeDir) {
        logger.trace("ServerLauncher.ServerLauncher( jbossHome='{}' )", jbossHomeDir);

        if (!jbossHomeDir.isDirectory())
            throw new IllegalArgumentException("jbossHomeDir must point to a valid directory");

        this.jbossHomeDir = jbossHomeDir;
    }

    /**
     * @param domainMode true = start server in domain mode, false = start server in standalone mode
     */
    public void startServer(final boolean domainMode) throws ConnectException {
        logger.trace("ServerLauncher.startServer( domainMode='{}' )", domainMode);

        final String fileName = EXEC_DIR + (domainMode ? DOMAIN_LAUNCHER_FILENAME : STANDALONE_LAUNCHER_FILENAME)
                + (System.getProperty("os.name").toLowerCase().startsWith("windows")
                ? WINDOWS_EXTENSION
                : STARNIX_EXTENSION);

        final File executable = new File(jbossHomeDir, fileName);

        logger.info("Attempting to start JBoss server: '{}'", executable);

        if (!executable.exists()) {
            throw new IllegalStateException("Supplied jboss.home.dir does not contains necessary jboss files. " +
                    "File in question: " + executable.getAbsolutePath());
        }

        jbossLauncher = new ScriptLauncher(executable.getAbsolutePath(), ARGS);
        jbossLauncher.runScript();
    }

    /**
     * Terminate the operating system process the server is running in.
     * Blocks until the server has shutdown or the current thread is
     * interrupted.
     */
    public void terminateServerProcess() {
        jbossLauncher.stopScript();
    }


    // Modified class taken from JBoss installer
    private static class ScriptLauncher {

        private ProcessBuilder builder;
        private Process p;
        private BufferedReader reader;
        private BufferedWriter writer;
        private String line;

        private ScriptLauncher(String... command) {
            builder = new ProcessBuilder(command);
        }

        private void runScript() {
            logger.trace("ServerLauncher$ScriptLauncher.runScript()");
            try {
                p = builder.start();
                reader = new BufferedReader(new InputStreamReader(p.getInputStream()));
                writer = new BufferedWriter(new OutputStreamWriter(p.getOutputStream()));
                writer.newLine();
                writer.flush();
                Thread b = new Thread(new Runnable() {
                    @Override
                    public void run() {
                        try {
                            while ((line = reader.readLine()) != null) {
                                logger.trace("JBOSS STDOUT: {}", line);
                            }
                        } catch (IOException e) {
                            logger.error("ServerLauncher$ScriptLauncher.AnonymousRunnable: IOException caught", e);
                        }
                    }
                });
                b.setDaemon(true);
                b.start();
            } catch (Exception e) {
                logger.error("Exception caught in ServerLauncher$ScriptLauncher.runScript()", e);
            }
        }

        private void stopScript() {
            logger.trace("ServerLauncher$ScriptLauncher.stopScript()");
            p.destroy();
            try {
                p.waitFor();
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
    }
}
