/*
 * Copyright 2012 JBoss Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.overlord.sramp.atom.archive.expand;

import java.io.File;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.lang.reflect.Field;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;
import java.util.UUID;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.junit.Assert;
import org.junit.Test;
import org.oasis_open.docs.s_ramp.ns.s_ramp_v1.BaseArtifactType;
import org.oasis_open.docs.s_ramp.ns.s_ramp_v1.Document;
import org.oasis_open.docs.s_ramp.ns.s_ramp_v1.XsdDocument;
import org.overlord.sramp.atom.archive.SrampArchive;
import org.overlord.sramp.atom.archive.SrampArchiveEntry;

/**
 * Unit test for the {@link ZipToSrampArchive} class.
 *
 * @author eric.wittmann@redhat.com
 */
public class ZipToSrampArchiveTest {

	/**
	 * Test method for {@link org.overlord.sramp.atom.archive.jar.ZipToSrampArchive.jar.JarToSrampArchive#JarToSrampArchive(java.io.File)}.
	 */
	@Test
	public void testJarToSrampArchiveFile() throws Exception {
		InputStream resourceAsStream = null;
		File tempFile = null;
		FileOutputStream tempFileStream = null;
		ZipToSrampArchive j2sramp = null;

		try {
			resourceAsStream = ZipToSrampArchiveTest.class.getResourceAsStream("sample-webservice-0.0.1.jar"); //$NON-NLS-1$
			tempFile = File.createTempFile("j2sramp_test", ".jar"); //$NON-NLS-1$ //$NON-NLS-2$
			tempFileStream = FileUtils.openOutputStream(tempFile);
			IOUtils.copy(resourceAsStream, tempFileStream);
		} finally {
			IOUtils.closeQuietly(resourceAsStream);
			IOUtils.closeQuietly(tempFileStream);
		}

		try {
			j2sramp = new ZipToSrampArchive(tempFile) {};

			File jarWorkDir = getJarWorkDir(j2sramp);
			Assert.assertNotNull(jarWorkDir);
			Assert.assertTrue(jarWorkDir.isDirectory());
			Collection<File> files = FileUtils.listFiles(jarWorkDir, new String[] {"xsd", "wsdl"}, true); //$NON-NLS-1$ //$NON-NLS-2$
			Assert.assertEquals(2, files.size());
			Set<String> fnames = new HashSet<String>();
			for (File f : files) {
				fnames.add(f.getName());
			}
			Assert.assertTrue(fnames.contains("teetime.xsd")); //$NON-NLS-1$
			Assert.assertTrue(fnames.contains("teetime.wsdl")); //$NON-NLS-1$
		} finally {
			FileUtils.deleteQuietly(tempFile);
			ZipToSrampArchive.closeQuietly(j2sramp);
		}
	}

	/**
	 * Test method for {@link org.overlord.sramp.atom.archive.jar.ZipToSrampArchive.jar.JarToSrampArchive#JarToSrampArchive(java.io.InputStream)}.
	 */
	@Test
	public void testJarToSrampArchiveInputStream() throws Exception {
		InputStream resourceAsStream = ZipToSrampArchiveTest.class.getResourceAsStream("sample-webservice-0.0.1.jar"); //$NON-NLS-1$
		ZipToSrampArchive j2sramp = null;

		try {
			j2sramp = new ZipToSrampArchive(resourceAsStream){};

			File jarWorkDir = getJarWorkDir(j2sramp);
			Assert.assertNotNull(jarWorkDir);
			Assert.assertTrue(jarWorkDir.isDirectory());
			Collection<File> files = FileUtils.listFiles(jarWorkDir, new String[] {"xsd", "wsdl"}, true); //$NON-NLS-1$ //$NON-NLS-2$
			Assert.assertEquals(2, files.size());
			Set<String> fnames = new HashSet<String>();
			for (File f : files) {
				fnames.add(f.getName());
			}
			Assert.assertTrue(fnames.contains("teetime.xsd")); //$NON-NLS-1$
			Assert.assertTrue(fnames.contains("teetime.wsdl")); //$NON-NLS-1$
		} finally {
			if (j2sramp != null)
				j2sramp.close();
		}
	}

	/**
	 * Test method for {@link org.overlord.sramp.atom.archive.jar.ZipToSrampArchive.jar.JarToSrampArchive#createSrampArchive()}.
	 */
	@Test
	public void testCreateSrampArchive() throws Exception {
		InputStream resourceAsStream = ZipToSrampArchiveTest.class.getResourceAsStream("sample-webservice-0.0.1.jar"); //$NON-NLS-1$
		ZipToSrampArchive j2sramp = null;
		SrampArchive archive = null;

		try {
			j2sramp = new ZipToSrampArchive(resourceAsStream){};
			archive = j2sramp.createSrampArchive();
			Assert.assertNotNull(archive);
			Collection<SrampArchiveEntry> entries = archive.getEntries();
			Assert.assertEquals(2, entries.size());
			Set<String> paths = new HashSet<String>();
			for (SrampArchiveEntry entry : entries) {
				paths.add(entry.getPath());
			}
			Assert.assertEquals(2, entries.size());
			Assert.assertTrue(paths.contains("schema/teetime.xsd")); //$NON-NLS-1$
			Assert.assertTrue(paths.contains("wsdl/teetime.wsdl")); //$NON-NLS-1$
		} finally {
			ZipToSrampArchive.closeQuietly(j2sramp);
			SrampArchive.closeQuietly(archive);
		}
	}

	/**
	 * Test method for {@link org.overlord.sramp.atom.archive.jar.ZipToSrampArchive.jar.JarToSrampArchive#setArtifactFilter(ArtifactFilter)}.
	 */
	@Test
	public void testCustomArtifactFilter() throws Exception {
		InputStream resourceAsStream = ZipToSrampArchiveTest.class.getResourceAsStream("sample-webservice-0.0.1.jar"); //$NON-NLS-1$
		ZipToSrampArchive j2sramp = null;
		SrampArchive archive = null;

		try {
			j2sramp = new ZipToSrampArchive(resourceAsStream){};
			j2sramp.setArtifactFilter(new ArtifactFilter() {
			    @Override
			    public void setContext(ZipToSrampArchiveContext context) {
			    }
				@Override
				public boolean accepts(CandidateArtifact artifact) {
					return true;
				}
			});
			archive = j2sramp.createSrampArchive();
			Assert.assertNotNull(archive);
			Collection<SrampArchiveEntry> entries = archive.getEntries();
			Assert.assertEquals(16, entries.size());
			Set<String> paths = new HashSet<String>();
			for (SrampArchiveEntry entry : entries) {
				paths.add(entry.getPath());
			}
			Assert.assertEquals(16, entries.size());
			Assert.assertTrue(paths.contains("schema/teetime.xsd")); //$NON-NLS-1$
			Assert.assertTrue(paths.contains("wsdl/teetime.wsdl")); //$NON-NLS-1$
			Assert.assertTrue(paths.contains("com/redhat/ewittman/teetime/_2012/_09/wsdl/teetime_wsdl/TeeTimePortType.class")); //$NON-NLS-1$
			Assert.assertTrue(paths.contains("META-INF/maven/com.redhat.ewittman/sample-web-service/pom.properties")); //$NON-NLS-1$
		} finally {
			ZipToSrampArchive.closeQuietly(j2sramp);
			SrampArchive.closeQuietly(archive);
		}
	}

	/**
	 * Test method for {@link org.overlord.sramp.atom.archive.jar.ZipToSrampArchive.jar.JarToSrampArchive#createSrampArchive()}.
	 */
	@Test
	public void testMetaData() throws Exception {
		InputStream resourceAsStream = ZipToSrampArchiveTest.class.getResourceAsStream("sample-webservice-0.0.1.jar"); //$NON-NLS-1$
		ZipToSrampArchive j2sramp = null;
		SrampArchive archive = null;

		try {
			j2sramp = new ZipToSrampArchive(resourceAsStream){};
			archive = j2sramp.createSrampArchive();
			Assert.assertNotNull(archive);
			SrampArchiveEntry entry = archive.getEntry("schema/teetime.xsd"); //$NON-NLS-1$
			Assert.assertNotNull(entry);
			BaseArtifactType metaData = entry.getMetaData();
			Assert.assertNotNull(metaData);
			Assert.assertTrue(metaData instanceof XsdDocument);
			XsdDocument md = (XsdDocument) metaData;
			Assert.assertEquals("teetime.xsd", md.getName()); //$NON-NLS-1$
			Assert.assertNotNull(md.getUuid());
		} finally {
			ZipToSrampArchive.closeQuietly(j2sramp);
			SrampArchive.closeQuietly(archive);
		}
	}

	/**
	 * Test method for {@link org.overlord.sramp.atom.archive.jar.ZipToSrampArchive.jar.JarToSrampArchive#createSrampArchive()}.
	 */
	@Test
	public void testMetaDataFactory() throws Exception {
		InputStream resourceAsStream = ZipToSrampArchiveTest.class.getResourceAsStream("sample-webservice-0.0.1.jar"); //$NON-NLS-1$
		ZipToSrampArchive j2sramp = null;
		SrampArchive archive = null;

		try {
			j2sramp = new ZipToSrampArchive(resourceAsStream){};
			j2sramp.setMetaDataFactory(new MetaDataFactory() {
			    @Override
			    public void setContext(ZipToSrampArchiveContext context) {
			    }
				@Override
				public BaseArtifactType createMetaData(DiscoveredArtifact artifact) {
					BaseArtifactType md = new Document();
					md.setUuid(UUID.randomUUID().toString());
					md.setName("custom-" + artifact.getName()); //$NON-NLS-1$
					return md;
				}
			});
			archive = j2sramp.createSrampArchive();
			Assert.assertNotNull(archive);
			SrampArchiveEntry entry = archive.getEntry("schema/teetime.xsd"); //$NON-NLS-1$
			Assert.assertNotNull(entry);
			BaseArtifactType metaData = entry.getMetaData();
			Assert.assertNotNull(metaData);
			Assert.assertTrue(metaData instanceof Document);
			Document md = (Document) metaData;
			Assert.assertEquals("custom-teetime.xsd", md.getName()); //$NON-NLS-1$
			Assert.assertNotNull(md.getUuid());
		} finally {
			ZipToSrampArchive.closeQuietly(j2sramp);
			SrampArchive.closeQuietly(archive);
		}
	}

	/**
	 * Gets the JAR working directory.
	 * @param j2sramp
	 * @return the private JAR working directory
	 * @throws Exception
	 */
	public static File getJarWorkDir(ZipToSrampArchive j2sramp) throws Exception {
		Field field = getJarWorkDirField(j2sramp);
		boolean oldAccessible = field.isAccessible();
		field.setAccessible(true);
		File workDir = (File) field.get(j2sramp);
		field.setAccessible(oldAccessible);
		return workDir;
	}

    /**
     * Finds a field.
     * @param j2sramp
     * @throws NoSuchFieldException
     */
    protected static Field getJarWorkDirField(ZipToSrampArchive j2sramp) throws NoSuchFieldException {
        boolean found = false;
        Class<?> from = j2sramp.getClass();
        while (!found) {
            try {
                return from.getDeclaredField("jarWorkDir"); //$NON-NLS-1$
            } catch (NoSuchFieldException nsfe) {
                from = from.getSuperclass();
                if (from == null) {
                    break;
                }
            }
        }
        throw new NoSuchFieldException("jarWorkDir"); //$NON-NLS-1$
    }

}
