/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.activemq.artemis.tests.compatibility;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.activemq.artemis.tests.compatibility.base.ServerBase;
import org.junit.Assert;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import static org.apache.activemq.artemis.tests.compatibility.GroovyRun.SNAPSHOT;
import static org.apache.activemq.artemis.tests.compatibility.GroovyRun.TWO_FOUR;

/**
 * To run this test on the IDE and debug it, run the compatibility-tests through a command line once:
 *
 * cd /compatibility-tests
 * mvn install -Ptests | tee output.log
 *
 * on the output.log you will see the output generated by {@link #getClasspath(String)}
 *
 * On your IDE, edit the Run Configuration to your test and add those -D as parameters to your test.
 * On Idea you would do the following:
 *
 * Run->Edit Configuration->Add ArtemisMeshTest and add your properties.
 */
@RunWith(Parameterized.class)
public class Mesh2Test extends ServerBase {

   // this will ensure that all tests in this class are run twice,
   // once with "true" passed to the class' constructor and once with "false"
   @Parameterized.Parameters(name = "server={0}, producer={1}, consumer={2}")
   public static Collection getParameters() {
      // we don't need every single version ever released..
      // if we keep testing current one against 2.4 and 1.4.. we are sure the wire and API won't change over time
      List<Object[]> combinations = new ArrayList<>();
      combinations.add(new Object[]{SNAPSHOT, SNAPSHOT, SNAPSHOT});
      combinations.add(new Object[]{SNAPSHOT, TWO_FOUR, TWO_FOUR});
      combinations.add(new Object[]{TWO_FOUR, SNAPSHOT, SNAPSHOT});
      return combinations;
   }

   public Mesh2Test(String server, String sender, String receiver) throws Exception {
      super(server, sender, receiver);
   }

   @Test
   public void testSendReceiveTopicShared() throws Throwable {
      CountDownLatch latch = new CountDownLatch(1);

      setVariable(receiverClassloader, "latch", latch);
      AtomicInteger errors = new AtomicInteger(0);
      Thread t = new Thread() {
         @Override
         public void run() {
            try {
               evaluate(receiverClassloader, "meshTest/sendMessages.groovy", server, receiver, "receiveShared");
            } catch (Exception e) {
               e.printStackTrace();
               errors.incrementAndGet();
            }
         }
      };

      t.start();
      Assert.assertTrue(latch.await(10, TimeUnit.SECONDS));
      evaluate(senderClassloader,"meshTest/sendMessages.groovy", server, sender, "sendTopic");

      t.join();

      Assert.assertEquals(0, errors.get());
   }

}

