﻿//  ------------------------------------------------------------------------------------
//  Copyright (c) 2017 Red Hat, Inc.
//  All rights reserved.
//
//  Licensed under the Apache License, Version 2.0 (the ""License""); you may not use this
//  file except in compliance with the License. You may obtain a copy of the License at
//  http://www.apache.org/licenses/LICENSE-2.0
//
//  THIS CODE IS PROVIDED *AS IS* BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
//  EITHER EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION ANY IMPLIED WARRANTIES OR
//  CONDITIONS OF TITLE, FITNESS FOR A PARTICULAR PURPOSE, MERCHANTABLITY OR
//  NON-INFRINGEMENT.
//
//  See the Apache Version 2.0 License for specific language governing permissions and
//  limitations under the License.
//  ------------------------------------------------------------------------------------

//
// HelloWorld_async
//
// Command line:
//   HelloWorld_async [brokerUrl [brokerEndpointAddress]]
//
// Default:
//   HelloWorld_async amqp://localhost:5672 amq.topic
//
// Requires:
//   An unauthenticated broker or peer at the brokerUrl 
//   capable of receiving and delivering messages through 
//   the endpoint address.
//
using System;
using System.Threading.Tasks;
using Amqp;

namespace HelloWorld_async
{
    public class HelloWorld_async
    {
        public async static Task MessageSender(SenderLink sender, Message msg)
        {
            await sender.SendAsync(msg);
        }

        public async static Task<Message> MessageReceiver(ReceiverLink receiver)
        {
            Message msg = await receiver.ReceiveAsync();
            receiver.Accept(msg);
            return msg;
        }

        static async Task HelloAsync(string broker, string address)
        {
            Address brokerAddr = new Address(broker);
            Connection connection = await Connection.Factory.CreateAsync(brokerAddr);
            Session session = new Session(connection);

            ReceiverLink receiver = new ReceiverLink(session, "helloworld-receiver", address);
            var rTask = Task.Run(() => MessageReceiver(receiver));

            SenderLink sender = new SenderLink(session, "helloworld-sender", address);
            Message helloOut = new Message("Hello World!");
            var sTask = Task.Run(() => MessageSender(sender, helloOut));

            await Task.WhenAll(new Task[] { rTask, sTask });

            await receiver.CloseAsync();
            await sender.CloseAsync();
            await session.CloseAsync();
            await connection.CloseAsync();

            Message helloIn = rTask.Result;
            Console.WriteLine(helloIn.Body.ToString());
        }

        public static void Main(string[] args)
        {
            string broker  = args.Length >= 1 ? args[0] : "amqp://guest:guest@localhost:5672";
            string address = args.Length >= 2 ? args[1] : "q1";
            
            Task.Run(() => HelloAsync(broker, address)).Wait();
        }
    }
}
