package com.github.eyefloaters.console.api.model;

import java.net.URI;
import java.time.Instant;
import java.util.List;
import java.util.Map;

import jakarta.validation.constraints.AssertTrue;
import jakarta.validation.constraints.NotNull;
import jakarta.ws.rs.core.UriBuilder;

import org.eclipse.microprofile.openapi.annotations.media.Schema;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonFilter;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.github.eyefloaters.console.api.support.ErrorCategory;

import io.xlate.validation.constraints.Expression;

@Schema(name = "KafkaRecordAttributes")
@JsonFilter("fieldFilter")
public class KafkaRecord {

    public static final class Fields {
        public static final String PARTITION = "partition";
        public static final String OFFSET = "offset";
        public static final String TIMESTAMP = "timestamp";
        public static final String TIMESTAMP_TYPE = "timestampType";
        public static final String HEADERS = "headers";
        public static final String KEY = "key";
        public static final String VALUE = "value";
        public static final String SIZE = "size";

        public static final String DEFAULT =
                    PARTITION +
                    ", " + OFFSET +
                    ", " + TIMESTAMP +
                    ", " + TIMESTAMP_TYPE +
                    ", " + HEADERS +
                    ", " + KEY +
                    ", " + VALUE +
                    ", " + SIZE;

        public static final List<String> ALL = List.of(
                PARTITION,
                OFFSET,
                TIMESTAMP,
                TIMESTAMP_TYPE,
                HEADERS,
                KEY,
                VALUE,
                SIZE);

        private Fields() {
            // Prevent instances
        }
    }

    @Schema(name = "KafkaRecordListResponse")
    public static final class ListResponse extends DataList<RecordResource> {
        public ListResponse(List<KafkaRecord> data) {
            super(data.stream().map(RecordResource::new).toList());
        }
    }

    @Schema(name = "KafkaRecordDocument")
    public static final class KafkaRecordDocument extends DataSingleton<RecordResource> {
        @JsonCreator
        public KafkaRecordDocument(@JsonProperty("data") RecordResource data) {
            super(data);
        }

        public KafkaRecordDocument(KafkaRecord data) {
            super(new RecordResource(data));
        }
    }

    @Schema(name = "KafkaRecord")
    @Expression(
        when = "self.type != null",
        value = "self.type == 'records'",
        message = "resource type conflicts with operation",
        node = "type",
        payload = ErrorCategory.ResourceConflict.class
    )
    public static final class RecordResource extends Resource<KafkaRecord> {
        @JsonCreator
        public RecordResource(String type, KafkaRecord attributes) {
            super(null, type, attributes);
        }

        public RecordResource(KafkaRecord attributes) {
            super(null, "records", attributes);
        }
    }

    @JsonIgnore
    String topic;

    @Schema(description = "The record's partition within the topic")
    Integer partition;

    @Schema(readOnly = true, description = "The record's offset within the topic partition")
    Long offset;

    @Schema(description = "Timestamp associated with the record. The type is indicated by `timestampType`. When producing a record, this value will be used as the record's `CREATE_TIME`.", format = "date-time")
    Instant timestamp;

    @Schema(readOnly = true, description = "Type of timestamp associated with the record")
    String timestampType;

    @Schema(description = "Record headers, key/value pairs")
    Map<String, String> headers;

    @Schema(description = "Record key")
    String key;

    @NotNull
    @Schema(description = "Record value")
    String value;

    @Schema(readOnly = true, description = "Size of the uncompressed record, not including the overhead of the record in the log segment.")
    Long size;

    public KafkaRecord() {
        super();
    }

    public KafkaRecord(String topic) {
        this.topic = topic;
    }

    public KafkaRecord(String topic, Integer partition, Instant timestamp, Map<String, String> headers, String key, String value, Long size) {
        this(topic);
        this.partition = partition;
        this.timestamp = timestamp;
        this.headers = headers;
        this.key = key;
        this.value = value;
        this.size = size;
    }

    @JsonIgnore
    public URI buildUri(UriBuilder builder, String topicName) {
        builder.queryParam(Fields.PARTITION, partition);
        builder.queryParam(Fields.OFFSET, offset);
        return builder.build(topicName);
    }

    @AssertTrue(message = "invalid timestamp")
    @JsonIgnore
    public boolean isTimestampValid() {
        if (timestamp == null) {
            return true;
        }

        try {
            return timestamp.isAfter(Instant.ofEpochMilli(-1));
        } catch (Exception e) {
            return false;
        }
    }

    public Integer getPartition() {
        return partition;
    }

    public void setPartition(Integer partition) {
        this.partition = partition;
    }

    public Long getOffset() {
        return offset;
    }

    public void setOffset(Long offset) {
        this.offset = offset;
    }

    public Instant getTimestamp() {
        return timestamp;
    }

    public void setTimestamp(Instant timestamp) {
        this.timestamp = timestamp;
    }

    public String getTimestampType() {
        return timestampType;
    }

    public void setTimestampType(String timestampType) {
        this.timestampType = timestampType;
    }

    public Map<String, String> getHeaders() {
        return headers;
    }

    public void setHeaders(Map<String, String> headers) {
        this.headers = headers;
    }

    public String getKey() {
        return key;
    }

    public void setKey(String key) {
        this.key = key;
    }

    public String getValue() {
        return value;
    }

    public void setValue(String value) {
        this.value = value;
    }

    public Long getSize() {
        return size;
    }

    public void setSize(Long size) {
        this.size = size;
    }

}
