/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.component.kubernetes.processor;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import io.fabric8.kubernetes.api.model.EndpointAddress;
import io.fabric8.kubernetes.api.model.EndpointPort;
import io.fabric8.kubernetes.api.model.EndpointSubset;
import io.fabric8.kubernetes.api.model.Endpoints;
import io.fabric8.kubernetes.client.AutoAdaptableKubernetesClient;
import io.fabric8.kubernetes.client.ConfigBuilder;
import org.apache.camel.CamelContext;
import org.apache.camel.RuntimeCamelException;
import org.apache.camel.component.kubernetes.KubernetesConfiguration;
import org.apache.camel.impl.remote.DefaultServiceCallServer;
import org.apache.camel.spi.ServiceCallServer;
import org.apache.camel.util.IOHelper;
import org.apache.camel.util.ObjectHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class KubernetesServiceCallServerListStrategies {
    private KubernetesServiceCallServerListStrategies() {
    }

    // *************************************************************************
    // Client
    // *************************************************************************

    public static final class Client extends KubernetesServiceCallServerListStrategy {
        private static final Logger LOG = LoggerFactory.getLogger(Client.class);
        private static final int FIRST = 0;

        private AutoAdaptableKubernetesClient client;

        public Client(KubernetesConfiguration configuration) {
            super(configuration);

            this.client = null;
        }

        public List<ServiceCallServer> getUpdatedListOfServers(String name) {
            LOG.debug("Discovering endpoints from namespace: {} with name: {}", getNamespace(), name);
            Endpoints endpoints = client.endpoints().inNamespace(getNamespace()).withName(name).get();
            List<ServiceCallServer> result = new ArrayList<>();
            if (endpoints != null) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Found {} endpoints in namespace: {} for name: {} and portName: {}",
                        endpoints.getSubsets().size(), getNamespace(), name, getPortName());
                }
                for (EndpointSubset subset : endpoints.getSubsets()) {
                    if (subset.getPorts().size() == 1) {
                        addServers(result, subset.getPorts().get(FIRST), subset);
                    } else {
                        final List<EndpointPort> ports = subset.getPorts();
                        final int portSize = ports.size();

                        EndpointPort port;
                        for (int p = 0; p < portSize; p++) {
                            port = ports.get(p);
                            if (ObjectHelper.isEmpty(getPortName()) || getPortName().endsWith(port.getName())) {
                                addServers(result, port, subset);
                            }
                        }
                    }
                }
            }

            return result;
        }

        protected void addServers(List<ServiceCallServer> servers, EndpointPort port, EndpointSubset subset) {
            final List<EndpointAddress> addresses = subset.getAddresses();
            final int size = addresses.size();

            for (int i = 0; i < size; i++) {
                servers.add(new DefaultServiceCallServer(addresses.get(i).getIp(), port.getPort()));
            }
        }

        @Override
        protected void doStart() throws Exception {
            if (client != null) {
                return;
            }

            final KubernetesConfiguration configuration = getConfiguration();

            ConfigBuilder builder = new ConfigBuilder();
            builder.withMasterUrl(configuration.getMasterUrl());

            if ((ObjectHelper.isNotEmpty(configuration.getUsername())
                && ObjectHelper.isNotEmpty(configuration.getPassword()))
                && ObjectHelper.isEmpty(configuration.getOauthToken())) {
                builder.withUsername(configuration.getUsername());
                builder.withPassword(configuration.getPassword());
            } else {
                builder.withOauthToken(configuration.getOauthToken());
            }
            if (ObjectHelper.isNotEmpty(configuration.getCaCertData())) {
                builder.withCaCertData(configuration.getCaCertData());
            }
            if (ObjectHelper.isNotEmpty(configuration.getCaCertFile())) {
                builder.withCaCertFile(configuration.getCaCertFile());
            }
            if (ObjectHelper.isNotEmpty(configuration.getClientCertData())) {
                builder.withClientCertData(configuration.getClientCertData());
            }
            if (ObjectHelper.isNotEmpty(configuration.getClientCertFile())) {
                builder.withClientCertFile(configuration.getClientCertFile());
            }
            if (ObjectHelper.isNotEmpty(configuration.getApiVersion())) {
                builder.withApiVersion(configuration.getApiVersion());
            }
            if (ObjectHelper.isNotEmpty(configuration.getClientKeyAlgo())) {
                builder.withClientKeyAlgo(configuration.getClientKeyAlgo());
            }
            if (ObjectHelper.isNotEmpty(configuration.getClientKeyData())) {
                builder.withClientKeyData(configuration.getClientKeyData());
            }
            if (ObjectHelper.isNotEmpty(configuration.getClientKeyFile())) {
                builder.withClientKeyFile(configuration.getClientKeyFile());
            }
            if (ObjectHelper.isNotEmpty(configuration.getClientKeyPassphrase())) {
                builder.withClientKeyPassphrase(configuration.getClientKeyPassphrase());
            }
            if (ObjectHelper.isNotEmpty(configuration.getTrustCerts())) {
                builder.withTrustCerts(configuration.getTrustCerts());
            }

            client = new AutoAdaptableKubernetesClient(builder.build());
        }

        @Override
        protected void doStop() throws Exception {
            if (client != null) {
                IOHelper.close(client);
                client = null;
            }
        }

        @Override
        public String toString() {
            return "KubernetesServiceDiscovery.Client";
        }
    }

    // *************************************************************************
    // DNS
    // *************************************************************************

    private abstract static class StaticServerListStrategy extends KubernetesServiceCallServerListStrategy {
        private Map<String, List<ServiceCallServer>> servers;

        StaticServerListStrategy(KubernetesConfiguration configuration) {
            super(configuration);

            servers = new ConcurrentHashMap<>();
        }

        @Override
        public List<ServiceCallServer> getUpdatedListOfServers(String name) {
            List<ServiceCallServer> list = servers.get(name);
            if (list == null) {
                synchronized (servers) {
                    list = servers.computeIfAbsent(name, this::createServerList);
                }
            }

            return list;
        }

        protected abstract List<ServiceCallServer> createServerList(String name);
    }

    // *************************************************************************
    // DNS
    // *************************************************************************

    public static final class DNS extends StaticServerListStrategy {
        public DNS(KubernetesConfiguration configuration) {
            super(configuration);
        }

        @Override
        protected List<ServiceCallServer> createServerList(String name) {
            return Collections.singletonList(new DefaultServiceCallServer(
                name + "." + getConfiguration().getNamespace() + ".svc." + getConfiguration().getDnsDomain(),
                -1)
            );
        }

        @Override
        public String toString() {
            return "KubernetesServiceDiscovery.DNS";
        }
    }

    // *************************************************************************
    // Environment
    // *************************************************************************

    public static final class Environment extends StaticServerListStrategy {
        public Environment(KubernetesConfiguration configuration) {
            super(configuration);
        }

        @Override
        protected List<ServiceCallServer> createServerList(String name) {
            try {
                final CamelContext ctx = getCamelContext();
                final String ip = ctx.resolvePropertyPlaceholders("{{service.host:" + name + "}}");
                final String num = ctx.resolvePropertyPlaceholders("{{service.port:" + name + "}}");
                final int port = ctx.getTypeConverter().tryConvertTo(int.class, num);

                return Collections.singletonList(new DefaultServiceCallServer(ip, port));
            } catch (Exception e) {
                throw new RuntimeCamelException(e);
            }
        }

        @Override
        public String toString() {
            return "KubernetesServiceDiscovery.Environment";
        }
    }
}
