"use strict";
/*********************************************************************
 * Copyright (c) 2019 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 **********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenshiftTasks = void 0;
const tslib_1 = require("tslib");
const commandExists = require("command-exists");
const execa = require("execa");
const Listr = require("listr");
const version_1 = require("../../api/version");
class OpenshiftTasks {
    /**
     * Returns tasks list which perform preflight platform checks.
     */
    preflightCheckTasks(flags, command) {
        return new Listr([
            {
                title: 'Verify if oc is installed',
                task: (_ctx, task) => {
                    if (!commandExists.sync('oc')) {
                        command.error('E_REQUISITE_NOT_FOUND');
                    }
                    else {
                        task.title = `${task.title}...done.`;
                    }
                }
            },
            {
                title: 'Verify if openshift is running',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const openshiftIsRunning = yield this.isOpenshiftRunning();
                    if (!openshiftIsRunning) {
                        command.error(`E_PLATFORM_NOT_READY: oc status command failed. If there is no project, please create it before by running "oc new-project ${flags.chenamespace}"`);
                    }
                    else {
                        task.title = `${task.title}...done.`;
                    }
                })
            },
            version_1.VersionHelper.getOpenShiftCheckVersionTask(flags),
            version_1.VersionHelper.getK8sCheckVersionTask(flags),
        ], { renderer: flags['listr-renderer'] });
    }
    isOpenshiftRunning() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { exitCode } = yield execa('oc', ['status'], { timeout: 60000, reject: false });
            return exitCode === 0;
        });
    }
}
exports.OpenshiftTasks = OpenshiftTasks;
//# sourceMappingURL=openshift.js.map