"use strict";
/*********************************************************************
 * Copyright (c) 2020 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 **********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
exports.getPrintHighlightedMessagesTask = exports.getRetrieveKeycloakCredentialsTask = exports.getMessageImportCaCertIntoBrowser = exports.retrieveCheCaCertificateTask = exports.patchingEclipseCheCluster = exports.createEclipseCheCluster = exports.copyOperatorResources = exports.createNamespaceTask = void 0;
const tslib_1 = require("tslib");
const ansi = require("ansi-colors");
const fs_extra_1 = require("fs-extra");
const Listr = require("listr");
const lodash_1 = require("lodash");
const path = require("path");
const che_1 = require("../../api/che");
const context_1 = require("../../api/context");
const kube_1 = require("../../api/kube");
const constants_1 = require("../../constants");
function createNamespaceTask(namespaceName, labels) {
    return {
        title: `Create Namespace (${namespaceName})`,
        task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            const kube = new kube_1.KubeHelper();
            const che = new che_1.CheHelper({});
            const namespace = yield kube.getNamespace(namespaceName);
            if (namespace) {
                yield che.waitNamespaceActive(namespaceName);
                task.title = `${task.title}...It already exists.`;
            }
            else {
                yield kube.createNamespace(namespaceName, labels);
                yield che.waitNamespaceActive(namespaceName);
                task.title = `${task.title}...Done.`;
            }
        })
    };
}
exports.createNamespaceTask = createNamespaceTask;
function copyOperatorResources(flags, cacheDir) {
    return {
        title: 'Copying operator resources',
        task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            ctx.resourcesPath = yield copyCheOperatorResources(flags.templates, cacheDir);
            task.title = `${task.title}...done.`;
        })
    };
}
exports.copyOperatorResources = copyOperatorResources;
function copyCheOperatorResources(templatesDir, cacheDir) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const srcDir = path.join(templatesDir, '/codeready-workspaces-operator/');
        const destDir = path.join(cacheDir, '/templates/codeready-workspaces-operator/');
        yield fs_extra_1.remove(destDir);
        yield fs_extra_1.mkdirp(destDir);
        yield fs_extra_1.copy(srcDir, destDir);
        return destDir;
    });
}
function createEclipseCheCluster(flags, kube) {
    return {
        title: `Create the Custom Resource of type ${constants_1.CHE_CLUSTER_CRD} in the namespace ${flags.chenamespace}`,
        enabled: ctx => !!ctx.customCR || !!ctx.defaultCR,
        task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            ctx.isCheDeployed = true;
            ctx.isPostgresDeployed = true;
            ctx.isKeycloakDeployed = true;
            // plugin and devfile registry will be deployed only when external ones are not configured
            ctx.isPluginRegistryDeployed = !flags['plugin-registry-url'];
            ctx.isDevfileRegistryDeployed = !flags['devfile-registry-url'];
            const cheClusterCR = ctx.customCR || ctx.defaultCR;
            const cr = yield kube.createCheCluster(cheClusterCR, flags, ctx, !ctx.customCR);
            ctx.isKeycloakReady = ctx.isKeycloakReady || cr.spec.auth.externalIdentityProvider;
            ctx.isPostgresReady = ctx.isPostgresReady || cr.spec.database.externalDb;
            ctx.isDevfileRegistryReady = ctx.isDevfileRegistryReady || cr.spec.server.externalDevfileRegistry;
            ctx.isPluginRegistryReady = ctx.isPluginRegistryReady || cr.spec.server.externalPluginRegistry;
            if (cr.spec.server.customCheProperties && cr.spec.server.customCheProperties.CHE_MULTIUSER === 'false') {
                flags.multiuser = false;
            }
            task.title = `${task.title}...done.`;
        })
    };
}
exports.createEclipseCheCluster = createEclipseCheCluster;
/**
 * Update CheCluster CR object using CR patch file.
 * Clean up custom images if they weren't defined in the CR patch file to prevent update failing.
 * @param flags - parent command flags
 * @param kube - kubeHelper util
 * @param command - parent command
 */
function patchingEclipseCheCluster(flags, kube, command) {
    return {
        title: `Patching the Custom Resource of type '${constants_1.CHE_CLUSTER_CRD}' in the namespace '${flags.chenamespace}'`,
        skip: (ctx) => lodash_1.isEmpty(ctx[context_1.ChectlContext.CR_PATCH]),
        task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            const cheCluster = yield kube.getCheCluster(flags.chenamespace);
            if (!cheCluster) {
                command.error(`CodeReady Workspaces cluster CR is not found in the namespace '${flags.chenamespace}'`);
            }
            yield kube.patchCheCluster(cheCluster.metadata.name, flags.chenamespace, ctx[context_1.ChectlContext.CR_PATCH]);
            task.title = `${task.title}...done.`;
        })
    };
}
exports.patchingEclipseCheCluster = patchingEclipseCheCluster;
function retrieveCheCaCertificateTask(flags) {
    return {
        title: 'Retrieving Che self-signed CA certificate',
        // It makes sense to retrieve CA certificate only if self-signed certificate is used.
        enabled: () => flags.tls && flags.installer !== 'helm',
        task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            const che = new che_1.CheHelper(flags);
            const kube = new kube_1.KubeHelper();
            const cheCaCert = yield che.retrieveCheCaCert(flags.chenamespace);
            if (cheCaCert) {
                const targetFile = yield che.saveCheCaCert(cheCaCert);
                task.title = `${task.title}... done`;
                const serverStrategy = yield kube.getConfigMapValue('che', flags.chenamespace, 'CHE_INFRA_KUBERNETES_SERVER__STRATEGY');
                if (serverStrategy !== 'single-host') {
                    ctx.highlightedMessages.push(getMessageImportCaCertIntoBrowser(targetFile));
                }
            }
            else {
                task.title = `${task.title}... commonly trusted certificate is used.`;
            }
        })
    };
}
exports.retrieveCheCaCertificateTask = retrieveCheCaCertificateTask;
function getMessageImportCaCertIntoBrowser(caCertFileLocation) {
    const message = `${ansi.yellow('[ACTION REQUIRED]')} Please add Che self-signed CA certificate into your browser: ${caCertFileLocation}.\n` +
        `Documentation how to add a CA certificate into a browser: ${constants_1.DOCS_LINK_IMPORT_CA_CERT_INTO_BROWSER}`;
    return message;
}
exports.getMessageImportCaCertIntoBrowser = getMessageImportCaCertIntoBrowser;
function getRetrieveKeycloakCredentialsTask(flags) {
    return {
        title: 'Retrieving Keycloak admin credentials',
        enabled: () => (flags.installer !== 'helm'),
        task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            const che = new che_1.CheHelper(flags);
            const [login, password] = yield che.retrieveKeycloakAdminCredentials(flags.chenamespace);
            if (login && password) {
                ctx.identityProviderUsername = login;
                ctx.identityProviderPassword = password;
                task.title = `${task.title}...done`;
            }
            else {
                task.title = `${task.title}...failed.`;
            }
        })
    };
}
exports.getRetrieveKeycloakCredentialsTask = getRetrieveKeycloakCredentialsTask;
/**
 * Prints important to user messages which are stored in ctx.highlightedMessages
 * Typically this task is the last task of a command.
 */
function getPrintHighlightedMessagesTask() {
    return {
        title: 'Show important messages',
        enabled: ctx => ctx.highlightedMessages && ctx.highlightedMessages.length > 0,
        task: (ctx) => {
            const printMessageTasks = new Listr([], ctx.listrOptions);
            for (const message of ctx.highlightedMessages) {
                printMessageTasks.add({
                    title: message,
                    task: () => { }
                });
            }
            return printMessageTasks;
        }
    };
}
exports.getPrintHighlightedMessagesTask = getPrintHighlightedMessagesTask;
//# sourceMappingURL=common-tasks.js.map