"use strict";
/*********************************************************************
 * Copyright (c) 2020 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 **********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
exports.CertManagerTasks = exports.DEFAULT_CHE_CLUSTER_ISSUER_NAME = exports.CERT_MANAGER_CA_SECRET_NAME = void 0;
const tslib_1 = require("tslib");
const path = require("path");
const che_1 = require("../../api/che");
const kube_1 = require("../../api/kube");
const constants_1 = require("../../constants");
const util_1 = require("../../util");
const common_tasks_1 = require("../installers/common-tasks");
exports.CERT_MANAGER_CA_SECRET_NAME = 'ca';
exports.DEFAULT_CHE_CLUSTER_ISSUER_NAME = 'che-cluster-issuer';
class CertManagerTasks {
    constructor(flags) {
        this.kubeHelper = new kube_1.KubeHelper(flags);
        this.cheHelper = new che_1.CheHelper(flags);
    }
    /**
     * Returns list of tasks which perform cert-manager checks and deploy and requests self-signed certificate for Che.
     */
    getTasks(flags) {
        return [
            {
                title: 'Check Cert Manager deployment',
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    // Check only one CRD of cert-manager assuming that it is installed or not.
                    ctx.certManagerInstalled = (yield this.kubeHelper.getNamespace(constants_1.CERT_MANAGER_NAMESPACE_NAME)) && (yield this.kubeHelper.crdExist('certificates.cert-manager.io'));
                    if (ctx.certManagerInstalled) {
                        task.title = `${task.title}...already deployed`;
                    }
                    else {
                        task.title = `${task.title}...not deployed`;
                    }
                })
            },
            {
                title: 'Deploy cert-manager',
                enabled: ctx => !ctx.certManagerInstalled,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const yamlPath = path.join(flags.templates, '..', 'installers', 'cert-manager.yml');
                    // Apply additional --validate=false flag to be able to deploy Cert Manager on Kubernetes v1.15.4 or below
                    yield this.kubeHelper.applyResource(yamlPath, '--validate=false');
                    ctx.certManagerInstalled = true;
                    task.title = `${task.title}...done`;
                })
            },
            {
                title: 'Wait for cert-manager',
                enabled: ctx => ctx.certManagerInstalled,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (!ctx.certManagerInstalled) {
                        throw new Error('Cert Manager should be deployed before.');
                    }
                    ctx.certManagerK8sApiVersion = yield this.kubeHelper.getCertManagerK8sApiVersion();
                    const timeout = 5 * 60 * 1000;
                    yield this.kubeHelper.waitForPodReady('app.kubernetes.io/name=cert-manager', constants_1.CERT_MANAGER_NAMESPACE_NAME, 1000, timeout);
                    yield this.kubeHelper.waitForPodReady('app.kubernetes.io/name=webhook', constants_1.CERT_MANAGER_NAMESPACE_NAME, 1000, timeout);
                    yield this.kubeHelper.waitForPodReady('app.kubernetes.io/name=cainjector', constants_1.CERT_MANAGER_NAMESPACE_NAME, 1000, timeout);
                    task.title = `${task.title}...ready`;
                })
            },
            {
                title: 'Check Cert Manager CA certificate',
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (!ctx.certManagerInstalled) {
                        throw new Error('Cert manager must be installed before.');
                    }
                    // To be able to use self-signed sertificate it is required to provide CA private key & certificate to cert-manager
                    const caSelfSignedCertSecret = yield this.kubeHelper.getSecret(exports.CERT_MANAGER_CA_SECRET_NAME, constants_1.CERT_MANAGER_NAMESPACE_NAME);
                    if (!caSelfSignedCertSecret) {
                        // First run, generate CA self-signed certificate
                        task.title = `${task.title}...generating new one`;
                        const CA_CERT_GENERATION_SERVICE_ACCOUNT_NAME = 'ca-cert-generator';
                        const CA_CERT_GENERATION_JOB_NAME = 'ca-cert-generation-job';
                        try {
                            // Configure permissions for CA key pair generation job
                            yield this.kubeHelper.createServiceAccount(CA_CERT_GENERATION_SERVICE_ACCOUNT_NAME, constants_1.CERT_MANAGER_NAMESPACE_NAME);
                            yield this.kubeHelper.createRoleFromFile(path.join(flags.templates, 'cert-manager', 'ca-cert-generator-role.yml'), constants_1.CERT_MANAGER_NAMESPACE_NAME);
                            yield this.kubeHelper.createRoleBindingFromFile(path.join(flags.templates, 'cert-manager', 'ca-cert-generator-role-binding.yml'), constants_1.CERT_MANAGER_NAMESPACE_NAME);
                            // Await created resources to be available
                            yield this.kubeHelper.waitServiceAccount(CA_CERT_GENERATION_SERVICE_ACCOUNT_NAME, constants_1.CERT_MANAGER_NAMESPACE_NAME);
                            // Run CA key pair generation job
                            try {
                                yield this.kubeHelper.createJob(CA_CERT_GENERATION_JOB_NAME, constants_1.CA_CERT_GENERATION_JOB_IMAGE, CA_CERT_GENERATION_SERVICE_ACCOUNT_NAME, constants_1.CERT_MANAGER_NAMESPACE_NAME);
                                yield this.kubeHelper.waitJob(CA_CERT_GENERATION_JOB_NAME, constants_1.CERT_MANAGER_NAMESPACE_NAME);
                            }
                            catch (_a) {
                                throw new Error('Failed to generate self-signed CA certificate: generating job failed.');
                            }
                        }
                        finally {
                            // Clean up resources
                            try {
                                // Do not change order of statements.
                                // Despite logically it is better to remove role binding first, we should delete items here in order of their creation.
                                // Such approach will resolve situation if only subset of items were created during previos run.
                                yield this.kubeHelper.deleteServiceAccount(CA_CERT_GENERATION_SERVICE_ACCOUNT_NAME, constants_1.CERT_MANAGER_NAMESPACE_NAME);
                                yield this.kubeHelper.deleteRole('ca-cert-generator-role', constants_1.CERT_MANAGER_NAMESPACE_NAME);
                                yield this.kubeHelper.deleteRoleBinding('ca-cert-generator-role-binding', constants_1.CERT_MANAGER_NAMESPACE_NAME);
                                yield this.kubeHelper.deleteJob(CA_CERT_GENERATION_JOB_NAME, constants_1.CERT_MANAGER_NAMESPACE_NAME);
                            }
                            catch (_b) {
                                // Do nothing
                            }
                        }
                        // Wait until the secret is available
                        yield this.kubeHelper.waitSecret('ca', constants_1.CERT_MANAGER_NAMESPACE_NAME);
                    }
                    else {
                        task.title = `${task.title}...already exists`;
                    }
                })
            },
            {
                title: 'Set up CodeReady Workspaces certificates issuer',
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    let clusterIssuers = yield this.kubeHelper.listClusterIssuers(ctx.certManagerK8sApiVersion, constants_1.CHE_RELATED_COMPONENT_LABEL);
                    if (clusterIssuers.length > 1) {
                        throw new Error(`Found more than one cluster issuer with "${constants_1.CHE_RELATED_COMPONENT_LABEL}" label`);
                    }
                    else if (clusterIssuers.length === 1) {
                        // Found already configured cluster issuer
                        ctx.clusterIssuersName = clusterIssuers[0].metadata.name;
                        task.title = `${task.title}...found existing one: ${ctx.clusterIssuersName}`;
                        return;
                    }
                    // There is no labeled cluster issuers, check if there is only one configured
                    clusterIssuers = yield this.kubeHelper.listClusterIssuers(ctx.certManagerK8sApiVersion);
                    if (clusterIssuers.length === 1) {
                        // Using the cluster issuer
                        ctx.clusterIssuersName = clusterIssuers[0].metadata.name;
                        task.title = `${task.title}...found existing one: ${ctx.clusterIssuersName}`;
                        return;
                    }
                    ctx.clusterIssuersName = exports.DEFAULT_CHE_CLUSTER_ISSUER_NAME;
                    const cheClusterIssuerExists = yield this.kubeHelper.clusterIssuerExists(exports.DEFAULT_CHE_CLUSTER_ISSUER_NAME, ctx.certManagerK8sApiVersion);
                    if (!cheClusterIssuerExists) {
                        const cheCertificateClusterIssuerTemplatePath = path.join(flags.templates, '/cert-manager/che-cluster-issuer.yml');
                        yield this.kubeHelper.createCheClusterIssuer(cheCertificateClusterIssuerTemplatePath, ctx.certManagerK8sApiVersion);
                        task.title = `${task.title}...done`;
                    }
                    else {
                        task.title = `${task.title}...already exists`;
                    }
                })
            },
            {
                title: 'Request certificate',
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (ctx.cheCertificateExists) {
                        throw new Error('CodeReady Workspaces certificate already exists.');
                    }
                    if (ctx.clusterIssuersName === exports.DEFAULT_CHE_CLUSTER_ISSUER_NAME) {
                        task.title = 'Request self-signed certificate';
                    }
                    const certificateTemplatePath = path.join(flags.templates, '/cert-manager/che-certificate.yml');
                    const certifiateYaml = this.kubeHelper.safeLoadFromYamlFile(certificateTemplatePath);
                    const CN = '*.' + flags.domain;
                    certifiateYaml.spec.commonName = CN;
                    certifiateYaml.spec.dnsNames = [flags.domain, CN];
                    if (ctx.clusterIssuersName) {
                        certifiateYaml.spec.issuerRef.name = ctx.clusterIssuersName;
                    }
                    certifiateYaml.metadata.namespace = flags.chenamespace;
                    yield this.kubeHelper.createCheClusterCertificate(certifiateYaml, ctx.certManagerK8sApiVersion);
                    ctx.cheCertificateExists = true;
                    task.title = `${task.title}...done`;
                })
            },
            {
                title: 'Wait for certificate',
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (ctx.clusterIssuersName === exports.DEFAULT_CHE_CLUSTER_ISSUER_NAME) {
                        task.title = 'Wait for self-signed certificate';
                    }
                    yield this.kubeHelper.waitSecret(constants_1.CHE_TLS_SECRET_NAME, flags.chenamespace, ['tls.key', 'tls.crt', 'ca.crt']);
                    task.title = `${task.title}...ready`;
                })
            },
            {
                title: 'Retrieving Che CA certificate',
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (ctx.clusterIssuersName === exports.DEFAULT_CHE_CLUSTER_ISSUER_NAME) {
                        task.title = 'Retrieving Che self-signed CA certificate';
                    }
                    const cheSecret = yield this.kubeHelper.getSecret(constants_1.CHE_TLS_SECRET_NAME, flags.chenamespace);
                    if (cheSecret && cheSecret.data) {
                        const cheCaCrt = util_1.base64Decode(cheSecret.data['ca.crt']);
                        const cheCaCertPath = yield this.cheHelper.saveCheCaCert(cheCaCrt);
                        // We need to put self-signed CA certificate separately into CHE_ROOT_CA_SECRET_NAME secret
                        yield this.kubeHelper.createSecret(constants_1.CHE_ROOT_CA_SECRET_NAME, { 'ca.crt': cheCaCrt }, flags.chenamespace);
                        const serverStrategy = yield this.kubeHelper.getConfigMapValue('che', flags.chenamespace, 'CHE_INFRA_KUBERNETES_SERVER__STRATEGY');
                        if (serverStrategy !== 'single-host') {
                            ctx.highlightedMessages.push(common_tasks_1.getMessageImportCaCertIntoBrowser(cheCaCertPath));
                        }
                        task.title = `${task.title}... done`;
                    }
                    else {
                        throw new Error('Failed to get Cert Manager CA secret');
                    }
                })
            }
        ];
    }
}
exports.CertManagerTasks = CertManagerTasks;
//# sourceMappingURL=cert-manager.js.map