"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OperatorTasks = void 0;
const tslib_1 = require("tslib");
const cli_ux_1 = require("cli-ux");
const fs = require("fs");
const yaml = require("js-yaml");
const Listr = require("listr");
const kube_1 = require("../../api/kube");
const constants_1 = require("../../constants");
const util_1 = require("../../util");
const kube_2 = require("../kube");
const common_tasks_1 = require("./common-tasks");
class OperatorTasks {
    constructor() {
        this.operatorServiceAccount = 'codeready-operator';
        this.operatorRole = 'codeready-operator';
        this.operatorClusterRole = 'codeready-operator';
        this.operatorRoleBinding = 'codeready-operator';
        this.operatorClusterRoleBinding = 'codeready-operator';
        this.namespaceEditorClusterRole = 'che-namespace-editor';
        this.operatorNamespaceEditorClusterRoleBinding = 'codeready-operator-namespace-editor';
        this.cheClusterCrd = 'checlusters.org.eclipse.che';
    }
    /**
     * Returns tasks list which perform preflight platform checks.
     */
    startTasks(flags, command) {
        const clusterRoleName = `${flags.chenamespace}-${this.operatorClusterRole}`;
        const clusterRoleBindingName = `${flags.chenamespace}-${this.operatorClusterRoleBinding}`;
        const namespaceEditorClusterRoleName = `${flags.chenamespace}-${this.namespaceEditorClusterRole}`;
        const operatorNamespaceEditorClusterRoleBindingName = `${flags.chenamespace}-${this.operatorNamespaceEditorClusterRoleBinding}`;
        const kube = new kube_1.KubeHelper(flags);
        const kubeTasks = new kube_2.KubeTasks(flags);
        if (util_1.isStableVersion(flags)) {
            command.warn('Consider using the more reliable \'OLM\' installer when deploying a stable release of CodeReady Workspaces (--installer=olm).');
        }
        return new Listr([
            common_tasks_1.copyOperatorResources(flags, command.config.cacheDir),
            common_tasks_1.createNamespaceTask(flags.chenamespace, {}),
            {
                title: `Create ServiceAccount ${this.operatorServiceAccount} in namespace ${flags.chenamespace}`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield kube.serviceAccountExist(this.operatorServiceAccount, flags.chenamespace);
                    if (exist) {
                        task.title = `${task.title}...It already exists.`;
                    }
                    else {
                        const yamlFilePath = ctx.resourcesPath + 'service_account.yaml';
                        yield kube.createServiceAccountFromFile(yamlFilePath, flags.chenamespace);
                        task.title = `${task.title}...done.`;
                    }
                })
            },
            {
                title: `Create Role ${this.operatorRole} in namespace ${flags.chenamespace}`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield kube.roleExist(this.operatorRole, flags.chenamespace);
                    if (exist) {
                        task.title = `${task.title}...It already exists.`;
                    }
                    else {
                        const yamlFilePath = ctx.resourcesPath + 'role.yaml';
                        yield kube.createRoleFromFile(yamlFilePath, flags.chenamespace);
                        task.title = `${task.title}...done.`;
                    }
                })
            },
            {
                title: `Create ClusterRole ${clusterRoleName}`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield kube.clusterRoleExist(clusterRoleName);
                    if (exist) {
                        task.title = `${task.title}...It already exists.`;
                    }
                    else {
                        const yamlFilePath = ctx.resourcesPath + 'cluster_role.yaml';
                        yield kube.createClusterRoleFromFile(yamlFilePath, clusterRoleName);
                        task.title = `${task.title}...done.`;
                    }
                })
            },
            {
                title: `Create ClusterRole ${namespaceEditorClusterRoleName}`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    ctx.namespaceEditorClusterRoleName = namespaceEditorClusterRoleName;
                    const exist = yield kube.clusterRoleExist(namespaceEditorClusterRoleName);
                    if (exist) {
                        task.title = `${task.title}...It already exists.`;
                    }
                    else {
                        const yamlFilePath = ctx.resourcesPath + 'namespaces_cluster_role.yaml';
                        yield kube.createClusterRoleFromFile(yamlFilePath, namespaceEditorClusterRoleName);
                        task.title = `${task.title}...done.`;
                    }
                })
            },
            {
                title: `Create ClusterRoleBinding ${operatorNamespaceEditorClusterRoleBindingName}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield kube.clusterRoleBindingExist(operatorNamespaceEditorClusterRoleBindingName);
                    if (exist) {
                        task.title = `${task.title}...It already exists.`;
                    }
                    else {
                        yield kube.createClusterRoleBinding(operatorNamespaceEditorClusterRoleBindingName, this.operatorServiceAccount, flags.chenamespace, namespaceEditorClusterRoleName);
                        task.title = `${task.title}...done.`;
                    }
                })
            },
            {
                title: `Create RoleBinding ${this.operatorRoleBinding} in namespace ${flags.chenamespace}`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield kube.roleBindingExist(this.operatorRoleBinding, flags.chenamespace);
                    if (exist) {
                        task.title = `${task.title}...It already exists.`;
                    }
                    else {
                        const yamlFilePath = ctx.resourcesPath + 'role_binding.yaml';
                        yield kube.createRoleBindingFromFile(yamlFilePath, flags.chenamespace);
                        task.title = `${task.title}...done.`;
                    }
                })
            },
            {
                title: `Create ClusterRoleBinding ${clusterRoleBindingName}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield kube.clusterRoleBindingExist(clusterRoleBindingName);
                    if (exist) {
                        task.title = `${task.title}...It already exists.`;
                    }
                    else {
                        yield kube.createClusterRoleBinding(clusterRoleBindingName, this.operatorServiceAccount, flags.chenamespace, clusterRoleName);
                        task.title = `${task.title}...done.`;
                    }
                })
            },
            {
                title: `Create CRD ${this.cheClusterCrd}`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield kube.crdExist(this.cheClusterCrd);
                    const yamlFilePath = ctx.resourcesPath + 'crds/org_v1_che_crd.yaml';
                    if (exist) {
                        const checkCRD = yield kube.isCRDCompatible(this.cheClusterCrd, yamlFilePath);
                        if (!checkCRD) {
                            cli_ux_1.cli.error(`It is not possible to proceed the installation of CodeReady Workspaces. The existed ${this.cheClusterCrd} is different from a new one. Please update it to continue the installation.`);
                        }
                        task.title = `${task.title}...It already exists.`;
                    }
                    else {
                        yield kube.createCrdFromFile(yamlFilePath);
                        task.title = `${task.title}...done.`;
                    }
                })
            },
            {
                title: 'Waiting 5 seconds for the new Kubernetes resources to get flushed',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield cli_ux_1.cli.wait(5000);
                    task.title = `${task.title}...done.`;
                })
            },
            {
                title: `Create deployment ${constants_1.OPERATOR_DEPLOYMENT_NAME} in namespace ${flags.chenamespace}`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield kube.deploymentExist(constants_1.OPERATOR_DEPLOYMENT_NAME, flags.chenamespace);
                    if (exist) {
                        task.title = `${task.title}...It already exists.`;
                    }
                    else {
                        yield kube.createDeploymentFromFile(ctx.resourcesPath + 'operator.yaml', flags.chenamespace, flags['che-operator-image']);
                        task.title = `${task.title}...done.`;
                    }
                })
            },
            {
                title: 'Operator pod bootstrap',
                task: () => kubeTasks.podStartTasks(constants_1.CHE_OPERATOR_SELECTOR, flags.chenamespace)
            },
            {
                title: 'Prepare CodeReady Workspaces cluster CR',
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const cheCluster = yield kube.getCheCluster(flags.chenamespace);
                    if (cheCluster) {
                        task.title = `${task.title}...It already exists..`;
                        return;
                    }
                    if (!ctx.customCR) {
                        const yamlFilePath = ctx.resourcesPath + 'crds/org_v1_che_cr.yaml';
                        ctx.defaultCR = yaml.safeLoad(fs.readFileSync(yamlFilePath).toString());
                    }
                    task.title = `${task.title}...Done.`;
                })
            },
            common_tasks_1.createEclipseCheCluster(flags, kube)
        ], { renderer: flags['listr-renderer'] });
    }
    preUpdateTasks(flags, command) {
        const kube = new kube_1.KubeHelper(flags);
        return new Listr([
            {
                title: 'Checking versions compatibility before updating',
                task: (ctx, _task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const operatorDeployment = yield kube.getDeployment(constants_1.OPERATOR_DEPLOYMENT_NAME, flags.chenamespace);
                    if (!operatorDeployment) {
                        command.error(`${constants_1.OPERATOR_DEPLOYMENT_NAME} deployment is not found in namespace ${flags.chenamespace}.\nProbably CodeReady Workspaces was initially deployed with another installer`);
                    }
                    const deployedCheOperator = this.retrieveContainerImage(operatorDeployment);
                    const deployedCheOperatorImageAndTag = deployedCheOperator.split(':', 2);
                    ctx.deployedCheOperatorImage = deployedCheOperatorImageAndTag[0];
                    ctx.deployedCheOperatorTag = deployedCheOperatorImageAndTag.length === 2 ? deployedCheOperatorImageAndTag[1] : 'latest';
                    const newCheOperatorImageAndTag = flags['che-operator-image'].split(':', 2);
                    ctx.newCheOperatorImage = newCheOperatorImageAndTag[0];
                    ctx.newCheOperatorTag = newCheOperatorImageAndTag.length === 2 ? newCheOperatorImageAndTag[1] : 'latest';
                })
            }
        ]);
    }
    updateTasks(flags, command) {
        const kube = new kube_1.KubeHelper(flags);
        const clusterRoleName = `${flags.chenamespace}-${this.operatorClusterRole}`;
        const clusterRoleBindingName = `${flags.chenamespace}-${this.operatorClusterRoleBinding}`;
        const namespaceEditorClusterRoleName = `${flags.chenamespace}-${this.namespaceEditorClusterRole}`;
        const operatorNamespaceEditorClusterRoleBindingName = `${flags.chenamespace}-${this.operatorNamespaceEditorClusterRoleBinding}`;
        return new Listr([
            common_tasks_1.copyOperatorResources(flags, command.config.cacheDir),
            {
                title: `Updating ServiceAccount ${this.operatorServiceAccount} in namespace ${flags.chenamespace}`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield kube.serviceAccountExist(this.operatorServiceAccount, flags.chenamespace);
                    const yamlFilePath = ctx.resourcesPath + 'service_account.yaml';
                    if (exist) {
                        yield kube.replaceServiceAccountFromFile(yamlFilePath, flags.chenamespace);
                        task.title = `${task.title}...updated.`;
                    }
                    else {
                        yield kube.createServiceAccountFromFile(yamlFilePath, flags.chenamespace);
                        task.title = `${task.title}...created new one.`;
                    }
                })
            },
            {
                title: `Updating Role ${this.operatorRole} in namespace ${flags.chenamespace}`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield kube.roleExist(this.operatorRole, flags.chenamespace);
                    const yamlFilePath = ctx.resourcesPath + 'role.yaml';
                    if (exist) {
                        yield kube.replaceRoleFromFile(yamlFilePath, flags.chenamespace);
                        task.title = `${task.title}...updated.`;
                    }
                    else {
                        yield kube.createRoleFromFile(yamlFilePath, flags.chenamespace);
                        task.title = `${task.title}...created new one.`;
                    }
                })
            },
            {
                title: `Updating ClusterRole ${clusterRoleName}`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const clusterRoleExists = yield kube.clusterRoleExist(clusterRoleName);
                    const legacyClusterRoleExists = yield kube.clusterRoleExist(this.operatorClusterRole);
                    const yamlFilePath = ctx.resourcesPath + 'cluster_role.yaml';
                    if (clusterRoleExists) {
                        yield kube.replaceClusterRoleFromFile(yamlFilePath, clusterRoleName);
                        task.title = `${task.title}...updated.`;
                        // it is needed to check the legacy cluster object name to be compatible with previous installations
                    }
                    else if (legacyClusterRoleExists) {
                        yield kube.replaceClusterRoleFromFile(yamlFilePath, this.operatorClusterRole);
                        task.title = `Updating ClusterRole ${this.operatorClusterRole}...updated.`;
                    }
                    else {
                        yield kube.createClusterRoleFromFile(yamlFilePath, clusterRoleName);
                        task.title = `${task.title}...created a new one.`;
                    }
                })
            },
            {
                title: `Updating ClusterRole ${namespaceEditorClusterRoleName}`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const clusterRoleExists = yield kube.clusterRoleExist(namespaceEditorClusterRoleName);
                    const yamlFilePath = ctx.resourcesPath + 'namespaces_cluster_role.yaml';
                    if (clusterRoleExists) {
                        yield kube.replaceClusterRoleFromFile(yamlFilePath, namespaceEditorClusterRoleName);
                        task.title = `${task.title}...updated.`;
                    }
                    else {
                        yield kube.createClusterRoleFromFile(yamlFilePath, namespaceEditorClusterRoleName);
                        task.title = `${task.title}...created a new one.`;
                    }
                })
            },
            {
                title: `Updating RoleBinding ${this.operatorRoleBinding} in namespace ${flags.chenamespace}`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield kube.roleBindingExist(this.operatorRoleBinding, flags.chenamespace);
                    const yamlFilePath = ctx.resourcesPath + 'role_binding.yaml';
                    if (exist) {
                        yield kube.replaceRoleBindingFromFile(yamlFilePath, flags.chenamespace);
                        task.title = `${task.title}...updated.`;
                    }
                    else {
                        yield kube.createRoleBindingFromFile(yamlFilePath, flags.chenamespace);
                        task.title = `${task.title}...created new one.`;
                    }
                })
            },
            {
                title: `Updating ClusterRoleBinding ${clusterRoleBindingName}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const clusterRoleBindExists = yield kube.clusterRoleBindingExist(clusterRoleBindingName);
                    const legacyClusterRoleBindExists = yield kube.clusterRoleBindingExist(this.operatorClusterRoleBinding);
                    if (clusterRoleBindExists) {
                        yield kube.replaceClusterRoleBinding(clusterRoleBindingName, this.operatorServiceAccount, flags.chenamespace, clusterRoleName);
                        task.title = `${task.title}...updated.`;
                        // it is needed to check the legacy cluster object name to be compatible with previous installations
                    }
                    else if (legacyClusterRoleBindExists) {
                        yield kube.replaceClusterRoleBinding(this.operatorClusterRoleBinding, this.operatorServiceAccount, flags.chenamespace, this.operatorClusterRole);
                        task.title = `Updating ClusterRoleBinding ${this.operatorClusterRoleBinding}...updated.`;
                    }
                    else {
                        yield kube.createClusterRoleBinding(clusterRoleBindingName, this.operatorServiceAccount, flags.chenamespace, clusterRoleName);
                        task.title = `${task.title}...created new one.`;
                    }
                })
            },
            {
                title: `Updating ClusterRoleBinding ${operatorNamespaceEditorClusterRoleBindingName}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const clusterRoleBindExists = yield kube.clusterRoleBindingExist(operatorNamespaceEditorClusterRoleBindingName);
                    if (clusterRoleBindExists) {
                        yield kube.replaceClusterRoleBinding(operatorNamespaceEditorClusterRoleBindingName, this.operatorServiceAccount, flags.chenamespace, namespaceEditorClusterRoleName);
                        task.title = `${task.title}...updated.`;
                    }
                    else {
                        yield kube.createClusterRoleBinding(operatorNamespaceEditorClusterRoleBindingName, this.operatorServiceAccount, flags.chenamespace, namespaceEditorClusterRoleName);
                        task.title = `${task.title}...created new one.`;
                    }
                })
            },
            {
                title: `Updating CodeReady Workspaces cluster CRD ${this.cheClusterCrd}`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const crd = yield kube.getCrd(this.cheClusterCrd);
                    const yamlFilePath = ctx.resourcesPath + 'crds/org_v1_che_crd.yaml';
                    if (crd) {
                        if (!crd.metadata || !crd.metadata.resourceVersion) {
                            throw new Error(`Fetched CRD ${this.cheClusterCrd} without resource version`);
                        }
                        yield kube.replaceCrdFromFile(yamlFilePath, crd.metadata.resourceVersion);
                        task.title = `${task.title}...updated.`;
                    }
                    else {
                        yield kube.createCrdFromFile(yamlFilePath);
                        task.title = `${task.title}...created new one.`;
                    }
                })
            },
            {
                title: 'Waiting 5 seconds for the new Kubernetes resources to get flushed',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield cli_ux_1.cli.wait(5000);
                    task.title = `${task.title}...done.`;
                })
            },
            {
                title: `Updating deployment ${constants_1.OPERATOR_DEPLOYMENT_NAME} in namespace ${flags.chenamespace}`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield kube.deploymentExist(constants_1.OPERATOR_DEPLOYMENT_NAME, flags.chenamespace);
                    if (exist) {
                        yield kube.replaceDeploymentFromFile(ctx.resourcesPath + 'operator.yaml', flags.chenamespace, flags['che-operator-image']);
                        task.title = `${task.title}...updated.`;
                    }
                    else {
                        yield kube.createDeploymentFromFile(ctx.resourcesPath + 'operator.yaml', flags.chenamespace, flags['che-operator-image']);
                        task.title = `${task.title}...created new one.`;
                    }
                })
            },
            {
                title: 'Waiting newer operator to be run',
                task: (_ctx, _task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield cli_ux_1.cli.wait(1000);
                    yield kube.waitLatestReplica(constants_1.OPERATOR_DEPLOYMENT_NAME, flags.chenamespace);
                })
            },
            common_tasks_1.patchingEclipseCheCluster(flags, kube, command),
        ], { renderer: flags['listr-renderer'] });
    }
    /**
     * Returns list of tasks which remove CodeReady Workspaces operator related resources
     */
    deleteTasks(flags) {
        let kh = new kube_1.KubeHelper(flags);
        const clusterRoleName = `${flags.chenamespace}-${this.operatorClusterRole}`;
        const clusterRoleBindingName = `${flags.chenamespace}-${this.operatorClusterRoleBinding}`;
        const namespaceEditorClusterRoleName = `${flags.chenamespace}-${this.namespaceEditorClusterRole}`;
        const operatorNamespaceEditorClusterRoleBindingName = `${flags.chenamespace}-${this.operatorNamespaceEditorClusterRoleBinding}`;
        return [{
                title: 'Delete oauthClientAuthorizations',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const checluster = yield kh.getCheCluster(flags.chenamespace);
                    if (checluster && checluster.spec && checluster.spec.auth && checluster.spec.auth.oAuthClientName) {
                        const oAuthClientAuthorizations = yield kh.getOAuthClientAuthorizations(checluster.spec.auth.oAuthClientName);
                        yield kh.deleteOAuthClientAuthorizations(oAuthClientAuthorizations);
                    }
                    task.title = `${task.title}...OK`;
                })
            },
            {
                title: `Delete the Custom Resource of type ${constants_1.CHE_CLUSTER_CRD}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const checluster = yield kh.getCheCluster(flags.chenamespace);
                    if (checluster) {
                        yield kh.patchCheClusterCustomResource(checluster.metadata.name, flags.chenamespace, { metadata: { finalizers: null } });
                    }
                    yield kh.deleteCheCluster(flags.chenamespace);
                    do {
                        yield cli_ux_1.cli.wait(2000); //wait a couple of secs for the finalizers to be executed
                    } while (yield kh.getCheCluster(flags.chenamespace));
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: `Delete CRD ${this.cheClusterCrd}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const crdExists = yield kh.crdExist(this.cheClusterCrd);
                    const checlusters = yield kh.getAllCheClusters();
                    if (checlusters.length > 0) {
                        task.title = yield `${task.title}...Skipped: another CodeReady Workspaces deployment found.`;
                    }
                    else {
                        // Check if CRD exist. When installer is helm the CRD are not created
                        if (crdExists) {
                            yield kh.deleteCrd(this.cheClusterCrd);
                        }
                        task.title = yield `${task.title}...OK`;
                    }
                })
            },
            {
                title: `Delete role binding ${this.operatorRoleBinding}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (yield kh.roleBindingExist(this.operatorRoleBinding, flags.chenamespace)) {
                        yield kh.deleteRoleBinding(this.operatorRoleBinding, flags.chenamespace);
                    }
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: `Delete role ${this.operatorRole}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (yield kh.roleExist(this.operatorRole, flags.chenamespace)) {
                        yield kh.deleteRole(this.operatorRole, flags.chenamespace);
                    }
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: `Delete cluster role binding ${clusterRoleBindingName}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const clusterRoleBindExists = yield kh.clusterRoleBindingExist(clusterRoleBindingName);
                    const legacyClusterRoleBindExists = yield kh.clusterRoleBindingExist(this.operatorClusterRoleBinding);
                    if (clusterRoleBindExists) {
                        yield kh.deleteClusterRoleBinding(clusterRoleBindingName);
                        task.title = yield `${task.title}...OK`;
                        // it is needed to check the legacy cluster object name to be compatible with previous installations
                    }
                    else if (legacyClusterRoleBindExists) {
                        yield kh.deleteClusterRoleBinding(this.operatorClusterRoleBinding);
                        task.title = yield `Delete cluster role binding ${this.operatorClusterRoleBinding}...OK`;
                    }
                })
            },
            {
                title: `Delete cluster role ${clusterRoleName}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const clusterRoleExists = yield kh.clusterRoleExist(clusterRoleName);
                    const legacyClusterRoleExists = yield kh.clusterRoleExist(this.operatorClusterRole);
                    if (clusterRoleExists) {
                        yield kh.deleteClusterRole(clusterRoleName);
                        task.title = yield `${task.title}...OK`;
                        // it is needed to check the legacy cluster object name to be compatible with previous installations
                    }
                    else if (legacyClusterRoleExists) {
                        yield kh.deleteClusterRole(this.operatorClusterRole);
                        task.title = yield `Delete cluster role ${this.operatorClusterRole}...OK`;
                    }
                })
            },
            {
                title: `Delete cluster role binding ${operatorNamespaceEditorClusterRoleBindingName}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const clusterRoleBindExists = yield kh.clusterRoleBindingExist(operatorNamespaceEditorClusterRoleBindingName);
                    if (clusterRoleBindExists) {
                        yield kh.deleteClusterRoleBinding(operatorNamespaceEditorClusterRoleBindingName);
                        task.title = yield `${task.title}...OK`;
                    }
                })
            },
            {
                title: `Delete cluster role ${namespaceEditorClusterRoleName}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const clusterRoleExists = yield kh.clusterRoleExist(namespaceEditorClusterRoleName);
                    if (clusterRoleExists) {
                        yield kh.deleteClusterRole(namespaceEditorClusterRoleName);
                        task.title = yield `${task.title}...OK`;
                    }
                })
            },
            {
                title: 'Delete server and workspace rolebindings',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (yield kh.roleBindingExist('che', flags.chenamespace)) {
                        yield kh.deleteRoleBinding('che', flags.chenamespace);
                    }
                    if (yield kh.roleBindingExist('che-workspace-exec', flags.chenamespace)) {
                        yield kh.deleteRoleBinding('che-workspace-exec', flags.chenamespace);
                    }
                    if (yield kh.roleBindingExist('che-workspace-view', flags.chenamespace)) {
                        yield kh.deleteRoleBinding('che-workspace-view', flags.chenamespace);
                    }
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: `Delete service accounts ${this.operatorServiceAccount}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (yield kh.serviceAccountExist(this.operatorServiceAccount, flags.chenamespace)) {
                        yield kh.deleteServiceAccount(this.operatorServiceAccount, flags.chenamespace);
                    }
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: 'Delete PVC codeready-operator',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (yield kh.persistentVolumeClaimExist('codeready-operator', flags.chenamespace)) {
                        yield kh.deletePersistentVolumeClaim('codeready-operator', flags.chenamespace);
                    }
                    task.title = yield `${task.title}...OK`;
                })
            },
        ];
    }
    evaluateTemplateOperatorImage(flags) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (flags['che-operator-image']) {
                return flags['che-operator-image'];
            }
            else {
                const filePath = flags.templates + '/codeready-workspaces-operator/operator.yaml';
                const yamlFile = fs.readFileSync(filePath);
                const yamlDeployment = yaml.safeLoad(yamlFile.toString());
                return yamlDeployment.spec.template.spec.containers[0].image;
            }
        });
    }
    retrieveContainerImage(deployment) {
        const containers = deployment.spec.template.spec.containers;
        const namespace = deployment.metadata.namespace;
        const name = deployment.metadata.name;
        if (containers.length === 0) {
            throw new Error(`Can not evaluate image of ${namespace}/${name} deployment. Containers list are empty`);
        }
        if (containers.length > 1) {
            throw new Error(`Can not evaluate image of ${namespace}/${name} deployment. It has multiple containers`);
        }
        const container = containers[0];
        if (!container.image) {
            throw new Error(`Container ${container.name} in deployment ${namespace}/${name} must have image specified`);
        }
        return container.image;
    }
}
exports.OperatorTasks = OperatorTasks;
//# sourceMappingURL=operator.js.map