"use strict";
/*********************************************************************
 * Copyright (c) 2020 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 **********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
exports.findWorkingNamespace = exports.getLatestChectlVersion = exports.readPackageJson = exports.getProjectName = exports.getProjectVersion = exports.getCommandErrorMessage = exports.isDirEmpty = exports.notifyCommandCompletedSuccessfully = exports.getCommandSuccessMessage = exports.readCRFile = exports.sleep = exports.getImageTag = exports.isStableVersion = exports.base64Decode = exports.generatePassword = exports.isOpenshiftPlatformFamily = exports.isKubernetesPlatformFamily = exports.getClusterClientCommand = exports.OPENSHIFT_CLI = exports.KUBERNETES_CLI = void 0;
const tslib_1 = require("tslib");
const axios_1 = require("axios");
const commandExists = require("command-exists");
const fs = require("fs-extra");
const https = require("https");
const yaml = require("js-yaml");
const notifier = require("node-notifier");
const pkjson = require('../package.json');
const context_1 = require("./api/context");
const kube_1 = require("./api/kube");
const constants_1 = require("./constants");
exports.KUBERNETES_CLI = 'kubectl';
exports.OPENSHIFT_CLI = 'oc';
function getClusterClientCommand() {
    const clusterClients = [exports.KUBERNETES_CLI, exports.OPENSHIFT_CLI];
    for (const command of clusterClients) {
        if (commandExists.sync(command)) {
            return command;
        }
    }
    throw new Error('No cluster CLI client is installed.');
}
exports.getClusterClientCommand = getClusterClientCommand;
function isKubernetesPlatformFamily(platform) {
    return platform === 'k8s' || platform === 'minikube' || platform === 'microk8s';
}
exports.isKubernetesPlatformFamily = isKubernetesPlatformFamily;
function isOpenshiftPlatformFamily(platform) {
    return platform === 'openshift' || platform === 'minishift' || platform === 'crc';
}
exports.isOpenshiftPlatformFamily = isOpenshiftPlatformFamily;
function generatePassword(passwodLength, charactersSet = '') {
    let dictionary;
    if (!charactersSet) {
        const ZERO_CHAR_CODE = 48;
        const NINE_CHAR_CODE = 57;
        const A_CHAR_CODE = 65;
        const Z_CHAR_CODE = 90;
        const a_CHAR_CODE = 97;
        const z_CHAR_CODE = 122;
        const ranges = [[ZERO_CHAR_CODE, NINE_CHAR_CODE], [A_CHAR_CODE, Z_CHAR_CODE], [a_CHAR_CODE, z_CHAR_CODE]];
        dictionary = [];
        for (let range of ranges) {
            for (let charCode = range[0]; charCode <= range[1]; charCode++) {
                dictionary.push(String.fromCharCode(charCode));
            }
        }
    }
    else {
        dictionary = [...charactersSet];
    }
    let generatedPassword = '';
    for (let i = 0; i < passwodLength; i++) {
        const randomIndex = Math.floor(Math.random() * dictionary.length);
        generatedPassword += dictionary[randomIndex];
    }
    return generatedPassword;
}
exports.generatePassword = generatePassword;
function base64Decode(arg) {
    return Buffer.from(arg, 'base64').toString('ascii');
}
exports.base64Decode = base64Decode;
/**
 * Indicates if stable version of `crwctl` is used.
 */
function isStableVersion(flags) {
    const operatorImage = flags['che-operator-image'] || constants_1.DEFAULT_CHE_OPERATOR_IMAGE;
    const cheVersion = getImageTag(operatorImage);
    return cheVersion !== 'nightly' && cheVersion !== 'latest' && !flags['catalog-source-yaml'] && !flags['catalog-source-name'];
}
exports.isStableVersion = isStableVersion;
/**
 * Returns the tag of the image.
 */
function getImageTag(image) {
    let entries = image.split('@');
    if (entries.length === 2) {
        // digest
        return entries[1];
    }
    entries = image.split(':');
    // tag
    return entries[1];
}
exports.getImageTag = getImageTag;
function sleep(ms) {
    // tslint:disable-next-line no-string-based-set-timeout
    return new Promise(resolve => setTimeout(resolve, ms));
}
exports.sleep = sleep;
/**
 * Returns CR file content. Throws an error, if file doesn't exist.
 * @param flags - parent command flags
 * @param CRKey - key for CR file flag
 * @param command - parent command
 */
function readCRFile(flags, CRKey) {
    const CRFilePath = flags[CRKey];
    if (!CRFilePath) {
        return;
    }
    if (fs.existsSync(CRFilePath)) {
        return yaml.safeLoad(fs.readFileSync(CRFilePath).toString());
    }
    throw new Error(`Unable to find file defined in the flag '--${CRKey}'`);
}
exports.readCRFile = readCRFile;
/**
 * Returns command success message with execution time.
 */
function getCommandSuccessMessage() {
    const ctx = context_1.ChectlContext.get();
    if (ctx[context_1.ChectlContext.START_TIME]) {
        if (!ctx[context_1.ChectlContext.END_TIME]) {
            ctx[context_1.ChectlContext.END_TIME] = Date.now();
        }
        const workingTimeInSeconds = Math.round((ctx[context_1.ChectlContext.END_TIME] - ctx[context_1.ChectlContext.START_TIME]) / 1000);
        const minutes = Math.floor(workingTimeInSeconds / 60);
        const seconds = (workingTimeInSeconds - minutes * 60) % 60;
        const minutesToStr = minutes.toLocaleString([], { minimumIntegerDigits: 2 });
        const secondsToStr = seconds.toLocaleString([], { minimumIntegerDigits: 2 });
        return `Command ${ctx[context_1.ChectlContext.COMMAND_ID]} has completed successfully in ${minutesToStr}:${secondsToStr}.`;
    }
    return `Command ${ctx[context_1.ChectlContext.COMMAND_ID]} has completed successfully.`;
}
exports.getCommandSuccessMessage = getCommandSuccessMessage;
function notifyCommandCompletedSuccessfully() {
    notifier.notify({
        title: 'crwctl',
        message: getCommandSuccessMessage()
    });
}
exports.notifyCommandCompletedSuccessfully = notifyCommandCompletedSuccessfully;
/**
 * Determine if a directory is empty.
 */
function isDirEmpty(dirname) {
    try {
        return fs.readdirSync(dirname).length === 0;
        // Fails in case if directory doesn't exist
    }
    catch (_a) {
        return true;
    }
}
exports.isDirEmpty = isDirEmpty;
/**
 * Returns command success message with execution time.
 */
function getCommandErrorMessage(err) {
    const ctx = context_1.ChectlContext.get();
    const logDirectory = ctx[context_1.ChectlContext.LOGS_DIRECTORY];
    let message = `${err}\nCommand ${ctx[context_1.ChectlContext.COMMAND_ID]} failed. Error log: ${ctx[context_1.ChectlContext.ERROR_LOG]}`;
    if (logDirectory && isDirEmpty(logDirectory)) {
        message += ` CodeReady Workspaces logs: ${logDirectory}`;
    }
    return message;
}
exports.getCommandErrorMessage = getCommandErrorMessage;
/**
 * Returns current crwctl version defined in package.json.
 */
function getProjectVersion() {
    return pkjson.version;
}
exports.getProjectVersion = getProjectVersion;
/**
 * Returns current crwctl version defined in package.json.
 */
function getProjectName() {
    return pkjson.name;
}
exports.getProjectName = getProjectName;
function readPackageJson() {
    return JSON.parse(fs.readFileSync('../package.json').toString());
}
exports.readPackageJson = readPackageJson;
/**
 * Returns latest crwctl version for the given channel.
 */
function getLatestChectlVersion(channel) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        if (getProjectName() !== 'crwctl') {
            return;
        }
        const axiosInstance = axios_1.default.create({
            httpsAgent: new https.Agent({})
        });
        try {
            const { data } = yield axiosInstance.get(`https://che-incubator.github.io/crwctl/channels/${channel}/linux-x64`);
            return data.version;
        }
        catch (_a) {
            return;
        }
    });
}
exports.getLatestChectlVersion = getLatestChectlVersion;
/**
 * The default CodeReady Workspaces namespace has been changed from 'che' to 'eclipse-che'.
 * It checks if legacy namespace 'che' exists. If so all crwctl commands
 * will launched against that namespace otherwise default 'eclipse-che' namespace will be used.
 */
function findWorkingNamespace(flags) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        if (flags.chenamespace) {
            // use user specified namespace
            return flags.chenamespace;
        }
        const kubeHelper = new kube_1.KubeHelper(flags);
        if (yield kubeHelper.getNamespace(constants_1.DEFAULT_CHE_NAMESPACE)) {
            return constants_1.DEFAULT_CHE_NAMESPACE;
        }
        if (yield kubeHelper.getNamespace(constants_1.LEGACY_CHE_NAMESPACE)) {
            return constants_1.LEGACY_CHE_NAMESPACE;
        }
        return constants_1.DEFAULT_CHE_NAMESPACE;
    });
}
exports.findWorkingNamespace = findWorkingNamespace;
//# sourceMappingURL=util.js.map