"use strict";
/*********************************************************************
 * Copyright (c) 2019 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 **********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const command_1 = require("@oclif/command");
const flags_1 = require("@oclif/parser/lib/flags");
const cli_ux_1 = require("cli-ux");
const fs = require("fs-extra");
const Listr = require("listr");
const lodash_1 = require("lodash");
const path = require("path");
const context_1 = require("../../api/context");
const kube_1 = require("../../api/kube");
const common_flags_1 = require("../../common-flags");
const constants_1 = require("../../constants");
const common_tasks_1 = require("../../tasks/installers/common-tasks");
const installer_1 = require("../../tasks/installers/installer");
const api_1 = require("../../tasks/platforms/api");
const util_1 = require("../../util");
class Update extends command_1.Command {
    static getTemplatesDir() {
        // return local templates folder if present
        const TEMPLATES = 'templates';
        const templatesDir = path.resolve(TEMPLATES);
        const exists = fs.pathExistsSync(templatesDir);
        if (exists) {
            return TEMPLATES;
        }
        // else use the location from modules
        return path.join(__dirname, '../../../templates');
    }
    run() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { flags } = this.parse(Update);
            flags.chenamespace = yield util_1.findWorkingNamespace(flags);
            const ctx = yield context_1.ChectlContext.initAndGet(flags, this);
            yield this.setDomainFlag(flags);
            if (!flags.installer) {
                yield this.setDefaultInstaller(flags);
                cli_ux_1.cli.info(`› Installer type is set to: '${flags.installer}'`);
            }
            yield this.config.runHook(constants_1.DEFAULT_ANALYTIC_HOOK_NAME, { command: Update.id, flags });
            const kubeHelper = new kube_1.KubeHelper(flags);
            const installerTasks = new installer_1.InstallerTasks();
            // pre update tasks
            const apiTasks = new api_1.ApiTasks();
            const preUpdateTasks = new Listr([], ctx.listrOptions);
            preUpdateTasks.add(apiTasks.testApiTasks(flags, this));
            preUpdateTasks.add(installerTasks.preUpdateTasks(flags, this));
            // update tasks
            const updateTasks = new Listr([], ctx.listrOptions);
            updateTasks.add({
                title: '↺  Updating...',
                task: () => new Listr(installerTasks.updateTasks(flags, this))
            });
            // post update tasks
            const postUpdateTasks = new Listr([], ctx.listrOptions);
            postUpdateTasks.add(common_tasks_1.getPrintHighlightedMessagesTask());
            try {
                yield preUpdateTasks.run(ctx);
            }
            catch (err) {
                this.error(util_1.getCommandErrorMessage(err));
            }
            if (flags.installer === 'operator') {
                const existedOperatorImage = `${ctx.deployedCheOperatorImage}:${ctx.deployedCheOperatorTag}`;
                const newOperatorImage = `${ctx.newCheOperatorImage}:${ctx.newCheOperatorTag}`;
                cli_ux_1.cli.info(`Existed CodeReady Workspaces operator: ${existedOperatorImage}.`);
                cli_ux_1.cli.info(`New CodeReady Workspaces operator    : ${newOperatorImage}.`);
                const defaultOperatorImageTag = util_1.getImageTag(constants_1.DEFAULT_CHE_OPERATOR_IMAGE);
                const crwctlChannel = defaultOperatorImageTag === 'nightly' ? 'next' : 'stable';
                const currentChectlVersion = util_1.getProjectVersion();
                const latestChectlVersion = yield util_1.getLatestChectlVersion(crwctlChannel);
                const crwctlName = util_1.getProjectName();
                // the same version is already installed
                if (newOperatorImage === existedOperatorImage) {
                    if (crwctlName === 'crwctl' && latestChectlVersion) {
                        // suggest update crwctl first
                        if (currentChectlVersion !== latestChectlVersion) {
                            cli_ux_1.cli.warn(`It is not possible to update CodeReady Workspaces to a newer version
using the current '${currentChectlVersion}' version of crwctl. Please, update 'crwctl'
to a newer version '${latestChectlVersion}' with the command 'crwctl update ${crwctlChannel}'
and then try again.`);
                        }
                        else if (!flags[common_flags_1.CHE_OPERATOR_CR_PATCH_YAML_KEY]) {
                            // same version, no patch then nothing to update
                            cli_ux_1.cli.info('CodeReady Workspaces is already up to date.');
                            this.exit(0);
                        }
                    }
                    else {
                        // unknown project, no patch file then suggest to update
                        if (!flags[common_flags_1.CHE_OPERATOR_CR_PATCH_YAML_KEY]) {
                            cli_ux_1.cli.warn(`It is not possible to update CodeReady Workspaces to a newer version
using the current '${currentChectlVersion}' version of '${util_1.getProjectName()}'.
Please, update '${util_1.getProjectName()}' and then try again.`);
                            this.exit(0);
                        }
                    }
                    // custom operator image is used
                }
                else if (newOperatorImage !== constants_1.DEFAULT_CHE_OPERATOR_IMAGE) {
                    cli_ux_1.cli.warn(`CodeReady Workspaces operator deployment will be updated with the provided image,
but other CodeReady Workspaces components will be updated to the ${defaultOperatorImageTag} version.
Consider removing '--che-operator-image' to update CodeReady Workspaces operator to the same version.`);
                }
                if (!flags.yes && !(yield cli_ux_1.cli.confirm('If you want to continue - press Y'))) {
                    cli_ux_1.cli.info('Update cancelled by user.');
                    this.exit(0);
                }
            }
            const cheCluster = yield kubeHelper.getCheCluster(flags.chenamespace);
            if (cheCluster.spec.server.cheImage
                || cheCluster.spec.server.cheImageTag
                || cheCluster.spec.server.devfileRegistryImage
                || cheCluster.spec.database.postgresImage
                || cheCluster.spec.server.pluginRegistryImage
                || cheCluster.spec.auth.identityProviderImage) {
                let imagesListMsg = '';
                const crPatch = ctx[context_1.ChectlContext.CR_PATCH] || {};
                if (cheCluster.spec.server.pluginRegistryImage
                    && (!crPatch.spec || !crPatch.spec.server || !crPatch.spec.server.pluginRegistryImage)) {
                    imagesListMsg += `\n - Plugin registry image: ${cheCluster.spec.server.pluginRegistryImage}`;
                    lodash_1.merge(crPatch, { spec: { server: { pluginRegistryImage: '' } } });
                }
                if (cheCluster.spec.server.devfileRegistryImage
                    && (!crPatch.spec || !crPatch.spec.server || !crPatch.spec.server.devfileRegistryImage)) {
                    imagesListMsg += `\n - Devfile registry image: ${cheCluster.spec.server.devfileRegistryImage}`;
                    lodash_1.merge(crPatch, { spec: { server: { devfileRegistryImage: '' } } });
                }
                if (cheCluster.spec.server.postgresImage
                    && (!crPatch.spec || !crPatch.spec.database || !crPatch.spec.database.postgresImage)) {
                    imagesListMsg += `\n - Postgres image: ${cheCluster.spec.database.postgresImage}`;
                    lodash_1.merge(crPatch, { spec: { database: { postgresImage: '' } } });
                }
                if (cheCluster.spec.server.identityProviderImage
                    && (!crPatch.spec || !crPatch.spec.auth || !crPatch.spec.auth.identityProviderImage)) {
                    imagesListMsg += `\n - Identity provider image: ${cheCluster.spec.auth.identityProviderImage}`;
                    lodash_1.merge(crPatch, { spec: { auth: { identityProviderImage: '' } } });
                }
                if (cheCluster.spec.server.cheImage
                    && (!crPatch.spec || !crPatch.spec.server || !crPatch.spec.server.cheImage)) {
                    imagesListMsg += `\n - CodeReady Workspaces server image name: ${cheCluster.spec.server.cheImage}`;
                    lodash_1.merge(crPatch, { spec: { server: { cheImage: '' } } });
                }
                if (cheCluster.spec.server.cheImageTag
                    && (!crPatch.spec || !crPatch.spec.server || !crPatch.spec.server.cheImageTag)) {
                    imagesListMsg += `\n - CodeReady Workspaces server image tag: ${cheCluster.spec.server.cheImageTag}`;
                    lodash_1.merge(crPatch, { spec: { server: { cheImageTag: '' } } });
                }
                ctx[context_1.ChectlContext.CR_PATCH] = crPatch;
                if (imagesListMsg) {
                    cli_ux_1.cli.warn(`In order to update CodeReady Workspaces to a newer version the fields defining the images in the '${cheCluster.metadata.name}'
Custom Resource in the '${flags.chenamespace}' namespace will be cleaned up:${imagesListMsg}`);
                    if (!flags.yes && !(yield cli_ux_1.cli.confirm('If you want to continue - press Y'))) {
                        cli_ux_1.cli.info('Update cancelled by user.');
                        this.exit(0);
                    }
                }
            }
            try {
                yield updateTasks.run(ctx);
                yield postUpdateTasks.run(ctx);
                this.log(util_1.getCommandSuccessMessage());
            }
            catch (err) {
                this.error(util_1.getCommandErrorMessage(err));
            }
            util_1.notifyCommandCompletedSuccessfully();
            this.exit(0);
        });
    }
    /**
     * Copies spec.k8s.ingressDomain. It is needed later for updates.
     */
    setDomainFlag(flags) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const kubeHelper = new kube_1.KubeHelper(flags);
            const cheCluster = yield kubeHelper.getCheCluster(flags.chenamespace);
            if (cheCluster && cheCluster.spec.k8s && cheCluster.spec.k8s.ingressDomain) {
                flags.domain = cheCluster.spec.k8s.ingressDomain;
            }
        });
    }
    /**
     * Sets installer type depending on the previous installation.
     */
    setDefaultInstaller(flags) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const kubeHelper = new kube_1.KubeHelper(flags);
            try {
                yield kubeHelper.getOperatorSubscription(constants_1.SUBSCRIPTION_NAME, flags.chenamespace);
                flags.installer = 'olm';
            }
            catch (_a) {
                flags.installer = 'operator';
            }
        });
    }
}
exports.default = Update;
Update.description = 'Update CodeReady Workspaces server.';
Update.examples = [
    '# Update CodeReady Workspaces:\n' +
        'crwctl server:update',
    '\n# Update CodeReady Workspaces in \'eclipse-che\' namespace:\n' +
        'crwctl server:update -n eclipse-che',
    '\n# Update CodeReady Workspaces and update its configuration in the custom resource:\n' +
        `crwctl server:update --${common_flags_1.CHE_OPERATOR_CR_PATCH_YAML_KEY} patch.yaml`,
];
Update.flags = {
    installer: flags_1.string({
        char: 'a',
        description: 'Installer type. If not set, default is autodetected depending on previous installation.',
        options: ['operator', 'olm'],
        hidden: true,
    }),
    platform: flags_1.string({
        char: 'p',
        description: 'Type of OpenShift platform. Valid values are \"openshift\", \"crc (for CodeReady Containers)\".',
        options: ['openshift', 'crc'],
        default: 'openshift'
    }),
    chenamespace: common_flags_1.cheNamespace,
    templates: flags_1.string({
        char: 't',
        description: 'Path to the templates folder',
        default: Update.getTemplatesDir(),
        env: 'CHE_TEMPLATES_FOLDER'
    }),
    'che-operator-image': flags_1.string({
        description: 'Container image of the operator. This parameter is used only when the installer is the operator',
        default: constants_1.DEFAULT_CHE_OPERATOR_IMAGE
    }),
    'skip-version-check': command_1.flags.boolean({
        description: 'Skip minimal versions check.',
        default: false,
        hidden: true,
    }),
    'deployment-name': common_flags_1.cheDeployment,
    'listr-renderer': common_flags_1.listrRenderer,
    'skip-kubernetes-health-check': common_flags_1.skipKubeHealthzCheck,
    yes: common_flags_1.assumeYes,
    help: command_1.flags.help({ char: 'h' }),
    [common_flags_1.CHE_OPERATOR_CR_PATCH_YAML_KEY]: common_flags_1.cheOperatorCRPatchYaml,
    telemetry: common_flags_1.CHE_TELEMETRY
};
//# sourceMappingURL=update.js.map