"use strict";
/*********************************************************************
 * Copyright (c) 2020 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 **********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const command_1 = require("@oclif/command");
const cli_ux_1 = require("cli-ux");
const inquirer = require("inquirer");
const che_api_client_1 = require("../../api/che-api-client");
const che_login_manager_1 = require("../../api/che-login-manager");
const context_1 = require("../../api/context");
const common_flags_1 = require("../../common-flags");
const constants_1 = require("../../constants");
class Use extends command_1.Command {
    run() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { args, flags } = this.parse(Use);
            yield context_1.ChectlContext.init(flags, this);
            yield this.config.runHook(constants_1.DEFAULT_ANALYTIC_HOOK_NAME, { command: Use.id, flags });
            if (flags.interactive) {
                yield this.interactiveSwitch();
                return;
            }
            let cheApiEndpoint = args[common_flags_1.CHE_API_ENDPOINT_KEY];
            let username = flags[common_flags_1.USERNAME_KEY];
            if (!cheApiEndpoint && !username) {
                throw new Error('No arguments provided');
            }
            const loginManager = yield che_login_manager_1.CheServerLoginManager.getInstance();
            if (!cheApiEndpoint) {
                // Try to use current server
                const currentLogin = loginManager.getCurrentLoginInfo();
                cheApiEndpoint = currentLogin.cheApiEndpoint;
                if (!cheApiEndpoint) {
                    // There is no current server to switch user on
                    throw new Error('No current login session. Please specify it directly.');
                }
                if (username === currentLogin.username) {
                    // This is already current context
                    cli_ux_1.cli.info(`Already logged in as ${username} on ${cheApiEndpoint} server`);
                    return;
                }
            }
            else {
                cheApiEndpoint = che_api_client_1.CheApiClient.normalizeCheApiEndpointUrl(cheApiEndpoint);
                // Check if any login exist for provided Che server
                if (!loginManager.hasLoginFor(cheApiEndpoint)) {
                    // Maybe /api suffix isn't provided
                    const cheApiEndpointGuess = cheApiEndpoint + '/api';
                    if (!loginManager.hasLoginFor(cheApiEndpointGuess)) {
                        cli_ux_1.cli.info(`No registered login sessions on server ${cheApiEndpoint}`);
                        return;
                    }
                    cheApiEndpoint = cheApiEndpointGuess;
                }
            }
            if (!username) {
                // Check if given server has only one login session to use
                const serverLogins = loginManager.getAllLogins().get(cheApiEndpoint);
                if (!serverLogins || (serverLogins && serverLogins.length < 1)) {
                    cli_ux_1.cli.info(`No registered login sessions for ${cheApiEndpoint} server`);
                    return;
                }
                if (serverLogins.length !== 1) {
                    throw new Error(`Username on ${cheApiEndpoint} server is expected. Please provide "--${common_flags_1.USERNAME_KEY}" parameter`);
                }
                // Use the only logged in user on the server
                username = serverLogins[0];
            }
            yield loginManager.switchLoginContext(cheApiEndpoint, username);
            cli_ux_1.cli.info(`Now active login is ${username} on ${cheApiEndpoint} server`);
        });
    }
    interactiveSwitch() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const loginManager = yield che_login_manager_1.CheServerLoginManager.getInstance();
            const allLogins = loginManager.getAllLogins();
            const currentLogin = loginManager.getCurrentLoginInfo();
            let cheApiEndpoint = '';
            let username = '';
            if (allLogins.size === 0) {
                cli_ux_1.cli.info('No login session exists');
                return;
            }
            else if (allLogins.size === 1) {
                // Retrieve the only login info
                cheApiEndpoint = allLogins.keys().next().value;
                username = allLogins.get(cheApiEndpoint)[0];
            }
            else {
                // Ask user to interactively select
                const choices = [];
                let current;
                allLogins.forEach((serverLogins, serverUrl) => {
                    choices.push(new inquirer.Separator(serverUrl));
                    for (const login of serverLogins) {
                        const choise = {
                            name: `   ${login}`,
                            value: { cheApiEndpoint: serverUrl, username: login }
                        };
                        choices.push(choise);
                        if (currentLogin.cheApiEndpoint === serverUrl && currentLogin.username === login) {
                            current = choise;
                        }
                    }
                });
                const userResponse = yield inquirer.prompt([{
                        name: 'context',
                        type: 'list',
                        message: 'Select login session',
                        choices,
                        default: current ? current.value : undefined,
                    }]);
                if (userResponse && userResponse.context) {
                    cheApiEndpoint = userResponse.context.cheApiEndpoint;
                    username = userResponse.context.username;
                }
            }
            if (cheApiEndpoint && username) {
                if (currentLogin.cheApiEndpoint === cheApiEndpoint && currentLogin.username === username) {
                    cli_ux_1.cli.info(`Already logged in as ${username} on ${cheApiEndpoint} server`);
                    return;
                }
                yield loginManager.switchLoginContext(cheApiEndpoint, username);
                cli_ux_1.cli.info(`Now active login is ${username} on ${cheApiEndpoint} server`);
            }
            else {
                cli_ux_1.cli.info('Nothing to change');
            }
        });
    }
}
exports.default = Use;
Use.description = 'Set active login session';
Use.args = [
    {
        name: common_flags_1.CHE_API_ENDPOINT_KEY,
        description: 'CodeReady Workspaces server API endpoint',
        required: false
    }
];
Use.flags = {
    help: command_1.flags.help({ char: 'h' }),
    [common_flags_1.USERNAME_KEY]: common_flags_1.username,
    interactive: command_1.flags.boolean({
        char: 'i',
        description: 'Select an active login session in interactive mode',
        required: false,
        exclusive: [common_flags_1.USERNAME_KEY]
    }),
    telemetry: common_flags_1.CHE_TELEMETRY
};
Use.examples = [
    '# Set an active login session for the specified user on the given cluster:\n' +
        'crwctl auth:use che-che.apps-crc.testing/api -u username',
    '\n\n# Switch to another user on the same cluster:\n' +
        'crwctl auth:use -u another-user-on-this-server',
    '\n\n# Switch to the only user on the given cluster:\n' +
        'crwctl auth:use my.cluster.net',
    '\n\n# Select an active login session in interactive mode:\n' +
        'crwctl auth:use -i',
];
//# sourceMappingURL=use.js.map