"use strict";
/*********************************************************************
 * Copyright (c) 2020 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 **********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const command_1 = require("@oclif/command");
const cli_ux_1 = require("cli-ux");
const che_api_client_1 = require("../../api/che-api-client");
const che_login_manager_1 = require("../../api/che-login-manager");
const context_1 = require("../../api/context");
const common_flags_1 = require("../../common-flags");
const constants_1 = require("../../constants");
class Delete extends command_1.Command {
    run() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { args, flags } = this.parse(Delete);
            yield context_1.ChectlContext.init(flags, this);
            yield this.config.runHook(constants_1.DEFAULT_ANALYTIC_HOOK_NAME, { command: Delete.id, flags });
            let cheApiEndpoint = che_api_client_1.CheApiClient.normalizeCheApiEndpointUrl(args[common_flags_1.CHE_API_ENDPOINT_KEY]);
            const username = flags[common_flags_1.USERNAME_KEY];
            const loginManager = yield che_login_manager_1.CheServerLoginManager.getInstance();
            if (!loginManager.hasLoginFor(cheApiEndpoint)) {
                // Maybe /api suffix isn't provided
                const cheApiEndpointGuess = cheApiEndpoint + '/api';
                if (!loginManager.hasLoginFor(cheApiEndpointGuess)) {
                    cli_ux_1.cli.info(`No registered login sessions on server ${cheApiEndpoint}`);
                    return;
                }
                cheApiEndpoint = cheApiEndpointGuess;
            }
            if (username) {
                if (!loginManager.hasLoginFor(cheApiEndpoint, username)) {
                    cli_ux_1.cli.info(`${username} is not logged in on ${cheApiEndpoint}. Nothing to delete.`);
                    return;
                }
            }
            loginManager.deleteLoginContext(cheApiEndpoint, username);
            if (username) {
                cli_ux_1.cli.info(`Successfully logged out ${username} on ${cheApiEndpoint}`);
            }
            else {
                cli_ux_1.cli.info(`Successfully logged out all users on ${cheApiEndpoint}`);
            }
        });
    }
}
exports.default = Delete;
Delete.description = 'Delete specified login session(s)';
Delete.args = [
    {
        name: common_flags_1.CHE_API_ENDPOINT_KEY,
        description: 'CodeReady Workspaces server API endpoint',
        required: true
    }
];
Delete.flags = {
    help: command_1.flags.help({ char: 'h' }),
    [common_flags_1.USERNAME_KEY]: common_flags_1.username,
    telemetry: common_flags_1.CHE_TELEMETRY
};
Delete.examples = [
    '# Delete login session of the specified user on the cluster:\n' +
        'crwctl auth:delete che-che.apps-crc.testing/api -u username',
    '\n\n# Delete all login sessions on the cluster:\n' +
        'crwctl auth:delete che-che.apps-crc.testing',
];
//# sourceMappingURL=delete.js.map