"use strict";
/*********************************************************************
 * Copyright (c) 2020 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 **********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
exports.DevWorkspaceTasks = void 0;
const tslib_1 = require("tslib");
const Listr = require("listr");
const path = require("path");
const che_1 = require("../../api/che");
const kube_1 = require("../../api/kube");
const openshift_1 = require("../../api/openshift");
const constants_1 = require("../../constants");
const cert_manager_1 = require("../component-installers/cert-manager");
const common_tasks_1 = require("../installers/common-tasks");
/**
 * Handle setup of the dev workspace operator controller.
 */
class DevWorkspaceTasks {
    constructor(flags) {
        this.flags = flags;
        this.devWorkspaceServiceAccount = 'devworkspace-controller-serviceaccount';
        // DevWorkspace Controller Roles
        this.devWorkspaceLeaderElectionRole = 'devworkspace-controller-leader-election-role';
        // DevWorkspace Controller Role Bindings
        this.devWorkspaceLeaderElectionRoleBinding = 'devworkspace-controller-leader-election-role';
        // DevWorkspace Controller Cluster Roles
        this.devWorkspaceEditWorkspaceClusterRole = 'devworkspace-controller-edit-workspaces';
        this.devworkspaceProxyClusterRole = 'devworkspace-controller-proxy-role';
        this.devworkspaceClusterRole = 'devworkspace-controller-role';
        this.devWorkspaceViewWorkspaceClusterRole = 'devworkspace-controller-view-workspaces';
        this.devWorkspaceClusterRoleWebhook = 'devworkspace-webhook-server';
        // DevWorkspace Controller ClusterRole Bindings
        this.devworkspaceProxyClusterRoleBinding = 'devworkspace-controller-proxy-rolebinding';
        this.devWorkspaceRoleBinding = 'devworkspace-controller-rolebinding';
        // DevWorkspace Che Roles
        this.devWorkspaceCheLeaderElectionRole = 'devworkspace-che-leader-election-role';
        // DevWorkspace Che RoleBindings
        this.devWorkspaceCheLeaderElectionRoleBinding = 'devworkspace-che-leader-election-rolebinding';
        // DevWorkspace Che Cluster Roles
        this.devWorkspaceCheMetricsReader = 'devworkspace-che-metrics-reader';
        this.devWorkspaceCheProxyRole = 'devworkspace-che-proxy-role';
        this.devWorkspaceCheRole = 'devworkspace-che-role';
        // ClusterRoleBindings DevWorkspaceChe
        this.devWorkspaceCheProxyClusterRolebinding = 'devworkspace-che-proxy-rolebinding';
        this.devWorkspaceCheClusterRolebinding = 'devworkspace-che-rolebinding';
        this.devWorkspaceWebhookServerClusterRolebinding = 'devworkspace-webhook-server';
        // Deployment names
        this.deploymentName = 'devworkspace-controller-manager';
        // ConfigMap names
        this.devWorkspaceConfigMap = 'devworkspace-controller-configmap';
        this.devworkspaceCheConfigmap = 'devworkspace-che-configmap';
        this.devWorkspaceCertificate = 'devworkspace-controller-serving-cert';
        this.devWorkspaceCertIssuer = 'devworkspace-controller-selfsigned-issuer';
        // DevWorkspace CRD Names
        this.devWorkspacesCrdName = 'devworkspaces.workspace.devfile.io';
        this.devWorkspaceTemplatesCrdName = 'devworkspacetemplates.workspace.devfile.io';
        this.workspaceRoutingsCrdName = 'devworkspaceroutings.controller.devfile.io';
        this.webhooksName = 'controller.devfile.io';
        // Web Terminal Operator constants
        this.WTOSubscriptionName = 'web-terminal';
        this.WTONamespace = 'openshift-operators';
        this.kubeHelper = new kube_1.KubeHelper(flags);
        this.cheHelper = new che_1.CheHelper(flags);
        this.openShiftHelper = new openshift_1.OpenShiftHelper();
        this.certManagerTask = new cert_manager_1.CertManagerTasks({ flags });
    }
    getTemplatePath(ctx) {
        if (ctx.isOpenShift) {
            return path.join(this.flags.templates, 'devworkspace', 'deployment', 'openshift', 'objects');
        }
        return path.join(this.flags.templates, 'devworkspace', 'deployment', 'kubernetes', 'objects');
    }
    /**
     * Returns list of tasks which setup dev-workspace.
     */
    getInstallTasks(flags) {
        return [
            common_tasks_1.createNamespaceTask(constants_1.DEFAULT_DEV_WORKSPACE_CONTROLLER_NAMESPACE, {}),
            {
                title: 'Verify cert-manager installation',
                enabled: (ctx) => !ctx.isOpenShift,
                task: (ctx, _task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    return new Listr(this.certManagerTask.getDeployCertManagerTasks(flags), ctx.listrOptions);
                })
            },
            // WARNING: Issuer and Certificate should be moved to codeready-operator side. Depends on issue: https://github.com/eclipse/che/issues/19502
            {
                title: `Create certificate issuer ${this.devWorkspaceCertIssuer}`,
                enabled: (ctx) => !ctx.isOpenShift,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const certIssuerExist = yield this.kubeHelper.isCertificateIssuerExists(this.devWorkspaceCertIssuer, ctx.certManagerK8sApiVersion, constants_1.DEFAULT_DEV_WORKSPACE_CONTROLLER_NAMESPACE);
                    if (certIssuerExist) {
                        task.title = `${task.title}...It already exists.`;
                        return;
                    }
                    const devWorkspaceIssuerCertFilePath = path.join(this.getTemplatePath(ctx), 'devworkspace-controller-selfsigned-issuer.Issuer.yaml');
                    yield this.kubeHelper.createCertificateIssuer(devWorkspaceIssuerCertFilePath, ctx.certManagerK8sApiVersion, constants_1.DEFAULT_DEV_WORKSPACE_CONTROLLER_NAMESPACE);
                    task.title = `${task.title}...Done.`;
                })
            },
            {
                title: `Create self signed certificate ${this.devWorkspaceCertificate}`,
                enabled: (ctx) => !ctx.isOpenShift,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const certExists = yield this.kubeHelper.isNamespacedCertificateExists(this.devWorkspaceCertificate, ctx.certManagerK8sApiVersion, constants_1.DEFAULT_DEV_WORKSPACE_CONTROLLER_NAMESPACE);
                    if (certExists) {
                        task.title = `${task.title}...It already exists.`;
                        return;
                    }
                    const certificateTemplatePath = path.join(this.getTemplatePath(ctx), 'devworkspace-controller-serving-cert.Certificate.yaml');
                    const certifiateYaml = this.kubeHelper.safeLoadFromYamlFile(certificateTemplatePath);
                    yield this.kubeHelper.createCheClusterCertificate(certifiateYaml, ctx.certManagerK8sApiVersion);
                    task.title = `${task.title}...Done.`;
                })
            },
        ];
    }
    /**
     * Returns list of tasks which uninstall dev-workspace.
     */
    getUninstallTasks() {
        return [
            {
                title: 'Delete all DevWorkspace Controller and DevWorkspace Che deployments',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kubeHelper.deleteAllDeployments(constants_1.DEFAULT_DEV_WORKSPACE_CONTROLLER_NAMESPACE);
                    yield this.kubeHelper.deleteAllDeployments(constants_1.DEFAULT_DEV_WORKSPACE_CHE_NAMESPACE);
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: 'Delete all DevWorkspace Controller and DevWorkspace Che services',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kubeHelper.deleteAllServices(constants_1.DEFAULT_DEV_WORKSPACE_CONTROLLER_NAMESPACE);
                    yield this.kubeHelper.deleteAllServices(constants_1.DEFAULT_DEV_WORKSPACE_CHE_NAMESPACE);
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: 'Delete all DevWorkspace Controller and DevWorkspace Che routes',
                enabled: (ctx) => !ctx.isOpenShift,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kubeHelper.deleteAllIngresses(constants_1.DEFAULT_DEV_WORKSPACE_CONTROLLER_NAMESPACE);
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: 'Delete all DevWorkspace Controller and DevWorkspace Che routes',
                enabled: (ctx) => ctx.isOpenShift,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.openShiftHelper.deleteAllRoutes(constants_1.DEFAULT_DEV_WORKSPACE_CONTROLLER_NAMESPACE);
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: 'Delete DevWorkspace Controller and DevWorkspace Che configmaps',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kubeHelper.deleteConfigMap(this.devWorkspaceConfigMap, constants_1.DEFAULT_DEV_WORKSPACE_CONTROLLER_NAMESPACE);
                    yield this.kubeHelper.deleteConfigMap(this.devworkspaceCheConfigmap, constants_1.DEFAULT_DEV_WORKSPACE_CHE_NAMESPACE);
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: 'Delete DevWorkspace Controller and DevWorkspace Che ClusterRoleBindings',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kubeHelper.deleteClusterRoleBinding(this.devWorkspaceRoleBinding);
                    yield this.kubeHelper.deleteClusterRoleBinding(this.devworkspaceProxyClusterRoleBinding);
                    yield this.kubeHelper.deleteClusterRoleBinding(this.devWorkspaceCheProxyClusterRolebinding);
                    yield this.kubeHelper.deleteClusterRoleBinding(this.devWorkspaceCheClusterRolebinding);
                    yield this.kubeHelper.deleteClusterRoleBinding(this.devWorkspaceWebhookServerClusterRolebinding);
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: 'Delete DevWorkspace Controller and DevWorkspace Che role',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kubeHelper.deleteRole(this.devWorkspaceLeaderElectionRole, constants_1.DEFAULT_DEV_WORKSPACE_CONTROLLER_NAMESPACE);
                    yield this.kubeHelper.deleteRole(this.devWorkspaceCheLeaderElectionRole, constants_1.DEFAULT_DEV_WORKSPACE_CHE_NAMESPACE);
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: 'Delete DevWorkspace Controller and DevWorkspace Che roleBinding',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kubeHelper.deleteRoleBinding(this.devWorkspaceLeaderElectionRoleBinding, constants_1.DEFAULT_DEV_WORKSPACE_CONTROLLER_NAMESPACE);
                    yield this.kubeHelper.deleteRoleBinding(this.devWorkspaceCheLeaderElectionRoleBinding, constants_1.DEFAULT_DEV_WORKSPACE_CHE_NAMESPACE);
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: 'Delete DevWorkspace Controller and DevWorkspace Che cluster roles',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kubeHelper.deleteClusterRole(this.devWorkspaceEditWorkspaceClusterRole);
                    yield this.kubeHelper.deleteClusterRole(this.devWorkspaceViewWorkspaceClusterRole);
                    yield this.kubeHelper.deleteClusterRole(this.devworkspaceProxyClusterRole);
                    yield this.kubeHelper.deleteClusterRole(this.devworkspaceClusterRole);
                    yield this.kubeHelper.deleteClusterRole(this.devWorkspaceClusterRoleWebhook);
                    yield this.kubeHelper.deleteClusterRole(this.devWorkspaceCheMetricsReader);
                    yield this.kubeHelper.deleteClusterRole(this.devWorkspaceCheProxyRole);
                    yield this.kubeHelper.deleteClusterRole(this.devWorkspaceCheRole);
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: 'Delete DevWorkspace Controller service account',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kubeHelper.deleteServiceAccount(this.devWorkspaceServiceAccount, constants_1.DEFAULT_DEV_WORKSPACE_CONTROLLER_NAMESPACE);
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: 'Delete DevWorkspace Controller self-signed certificates',
                enabled: (ctx) => tslib_1.__awaiter(this, void 0, void 0, function* () { return !ctx.IsOpenshift; }),
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kubeHelper.deleteNamespacedCertificate(this.devWorkspaceCertificate, 'v1', constants_1.DEFAULT_DEV_WORKSPACE_CONTROLLER_NAMESPACE);
                    yield this.kubeHelper.deleteNamespacedIssuer(this.devWorkspaceCertIssuer, 'v1', constants_1.DEFAULT_DEV_WORKSPACE_CONTROLLER_NAMESPACE);
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: 'Delete DevWorkspace Controller webhooks configurations',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kubeHelper.deleteMutatingWebhookConfiguration(this.webhooksName);
                    yield this.kubeHelper.deleteValidatingWebhookConfiguration(this.webhooksName);
                    task.title = yield `${task.title} ...OK`;
                }),
            },
            {
                title: 'Delete DevWorkspace Controller CRDs',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kubeHelper.deleteCrd(this.devWorkspacesCrdName);
                    yield this.kubeHelper.deleteCrd(this.devWorkspaceTemplatesCrdName);
                    yield this.kubeHelper.deleteCrd(this.workspaceRoutingsCrdName);
                    task.title = yield `${task.title}...OK`;
                })
            },
        ];
    }
}
exports.DevWorkspaceTasks = DevWorkspaceTasks;
//# sourceMappingURL=devfile-workspace-operator-installer.js.map