"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OperatorTasks = void 0;
const tslib_1 = require("tslib");
const cli_ux_1 = require("cli-ux");
const fs = require("fs");
const Listr = require("listr");
const path = require("path");
const context_1 = require("../../api/context");
const kube_1 = require("../../api/kube");
const version_1 = require("../../api/version");
const constants_1 = require("../../constants");
const util_1 = require("../../util");
const kube_2 = require("../kube");
const common_tasks_1 = require("./common-tasks");
class OperatorTasks {
    constructor() {
        this.operatorServiceAccount = 'codeready-operator';
        this.cheClusterCrd = 'checlusters.org.eclipse.che';
        this.cheManagerCRD = 'chemanagers.che.eclipse.org';
        this.dwRoutingCRD = 'devworkspaceroutings.controller.devfile.io';
        this.legacyClusterResourcesName = 'codeready-operator';
    }
    getReadRolesAndBindingsTask(kube) {
        return {
            title: 'Read Roles and Bindings',
            task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                ctx.roles = [];
                ctx.roleBindings = [];
                ctx.clusterRoles = [];
                ctx.clusterRoleBindings = [];
                const filesList = fs.readdirSync(ctx.resourcesPath);
                for (const fileName of filesList) {
                    if (!fileName.endsWith('.yaml')) {
                        continue;
                    }
                    const yamlFilePath = path.join(ctx.resourcesPath, fileName);
                    const yamlContent = kube.safeLoadFromYamlFile(yamlFilePath);
                    if (!(yamlContent && yamlContent.kind)) {
                        continue;
                    }
                    switch (yamlContent.kind) {
                        case 'Role':
                            ctx.roles.push(yamlContent);
                            break;
                        case 'RoleBinding':
                            ctx.roleBindings.push(yamlContent);
                            break;
                        case 'ClusterRole':
                            ctx.clusterRoles.push(yamlContent);
                            break;
                        case 'ClusterRoleBinding':
                            ctx.clusterRoleBindings.push(yamlContent);
                            break;
                        default:
                        // Ignore this object kind
                    }
                }
                // Check consistancy
                if (ctx.roles.length !== ctx.roleBindings.length) {
                    cli_ux_1.cli.warn('Number of Roles and Role Bindings is different');
                }
                if (ctx.clusterRoles.length !== ctx.clusterRoleBindings.length) {
                    cli_ux_1.cli.warn('Number of Cluster Roles and Cluster Role Bindings is different');
                }
                task.title = `${task.title}...done.`;
            })
        };
    }
    getCreateOrUpdateRolesAndBindingsTask(flags, taskTitle, shouldUpdate = false) {
        const kube = new kube_1.KubeHelper(flags);
        return {
            title: taskTitle,
            task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                if (!ctx.roles) {
                    // Should never happen. 'Read Roles and Bindings' task should be called first.
                    throw new Error('Should read Roles and Bindings first');
                }
                for (const role of ctx.roles) {
                    if (yield kube.roleExist(role.metadata.name, flags.chenamespace)) {
                        if (shouldUpdate) {
                            yield kube.replaceRoleFrom(role, flags.chenamespace);
                        }
                    }
                    else {
                        yield kube.createRoleFrom(role, flags.chenamespace);
                    }
                }
                for (const roleBinding of ctx.roleBindings) {
                    if (yield kube.roleBindingExist(roleBinding.metadata.name, flags.chenamespace)) {
                        if (shouldUpdate) {
                            yield kube.replaceRoleBindingFrom(roleBinding, flags.chenamespace);
                        }
                    }
                    else {
                        yield kube.createRoleBindingFrom(roleBinding, flags.chenamespace);
                    }
                }
                // For Cluster Roles and Cluster Role Bindings use prefix to allow several Che installations
                const clusterObjectNamePrefix = `${flags.chenamespace}-`;
                for (const clusterRole of ctx.clusterRoles) {
                    const clusterRoleName = clusterObjectNamePrefix + clusterRole.metadata.name;
                    if (yield kube.clusterRoleExist(clusterRoleName)) {
                        if (shouldUpdate) {
                            yield kube.replaceClusterRoleFrom(clusterRole, clusterRoleName);
                        }
                    }
                    else {
                        yield kube.createClusterRoleFrom(clusterRole, clusterRoleName);
                    }
                }
                for (const clusterRoleBinding of ctx.clusterRoleBindings) {
                    clusterRoleBinding.metadata.name = clusterObjectNamePrefix + clusterRoleBinding.metadata.name;
                    clusterRoleBinding.roleRef.name = clusterObjectNamePrefix + clusterRoleBinding.roleRef.name;
                    for (const subj of clusterRoleBinding.subjects || []) {
                        subj.namespace = flags.chenamespace;
                    }
                    if (yield kube.clusterRoleBindingExist(clusterRoleBinding.metadata.name)) {
                        if (shouldUpdate) {
                            yield kube.replaceClusterRoleBindingFrom(clusterRoleBinding);
                        }
                    }
                    else {
                        yield kube.createClusterRoleBindingFrom(clusterRoleBinding);
                    }
                }
                task.title = `${task.title}...done.`;
            })
        };
    }
    /**
     * Returns tasks list which perform preflight platform checks.
     */
    deployTasks(flags, command) {
        const kube = new kube_1.KubeHelper(flags);
        const kubeTasks = new kube_2.KubeTasks(flags);
        const ctx = context_1.ChectlContext.get();
        ctx.resourcesPath = path.join(flags.templates, constants_1.OPERATOR_TEMPLATE_DIR);
        if (version_1.VersionHelper.isDeployingStableVersion(flags)) {
            command.warn('Consider using the more reliable \'OLM\' installer when deploying a stable release of CodeReady Workspaces (--installer=olm).');
        }
        return new Listr([
            common_tasks_1.createNamespaceTask(flags.chenamespace, {}),
            {
                title: `Create ServiceAccount ${this.operatorServiceAccount} in namespace ${flags.chenamespace}`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield kube.serviceAccountExist(this.operatorServiceAccount, flags.chenamespace);
                    if (exist) {
                        task.title = `${task.title}...It already exists.`;
                    }
                    else {
                        const yamlFilePath = path.join(ctx.resourcesPath, 'service_account.yaml');
                        yield kube.createServiceAccountFromFile(yamlFilePath, flags.chenamespace);
                        task.title = `${task.title}...done.`;
                    }
                })
            },
            this.getReadRolesAndBindingsTask(kube),
            this.getCreateOrUpdateRolesAndBindingsTask(flags, 'Creating Roles and Bindings', false),
            {
                title: `Create CRD ${this.cheClusterCrd}`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const existedCRD = yield kube.getCrd(this.cheClusterCrd);
                    if (existedCRD) {
                        task.title = `${task.title}...It already exists.`;
                    }
                    else {
                        const newCRDPath = yield this.getCRDPath(ctx, flags);
                        yield kube.createCrdFromFile(newCRDPath);
                        task.title = `${task.title}...done.`;
                    }
                })
            },
            {
                title: `Create CRD ${this.cheManagerCRD}`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (!(yield kube.IsAPIExtensionSupported('v1'))) {
                        task.title = `${task.title}...Skipped.`;
                        return;
                    }
                    const existedCRD = yield kube.getCrd(this.cheManagerCRD);
                    if (existedCRD) {
                        task.title = `${task.title}...It already exists.`;
                    }
                    else {
                        const newCRDPath = path.join(ctx.resourcesPath, 'crds', 'chemanagers.che.eclipse.org.CustomResourceDefinition.yaml');
                        if (!fs.existsSync(newCRDPath)) {
                            task.title = `${task.title}...Skipped.`;
                            return;
                        }
                        yield kube.createCrdFromFile(newCRDPath);
                        task.title = `${task.title}...done.`;
                    }
                })
            },
            {
                title: `Create CRD ${this.dwRoutingCRD}`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (!(yield kube.IsAPIExtensionSupported('v1'))) {
                        task.title = `${task.title}...Skipped.`;
                        return;
                    }
                    const existedCRD = yield kube.getCrd(this.dwRoutingCRD);
                    if (existedCRD) {
                        task.title = `${task.title}...It already exists.`;
                    }
                    else {
                        const newCRDPath = path.join(ctx.resourcesPath, 'crds', 'devworkspaceroutings.controller.devfile.io.CustomResourceDefinition.yaml');
                        if (!fs.existsSync(newCRDPath)) {
                            task.title = `${task.title}...Skipped.`;
                            return;
                        }
                        yield kube.createCrdFromFile(newCRDPath);
                        task.title = `${task.title}...done.`;
                    }
                })
            },
            {
                title: 'Waiting 5 seconds for the new Kubernetes resources to get flushed',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield cli_ux_1.cli.wait(5000);
                    task.title = `${task.title}...done.`;
                })
            },
            {
                title: `Create deployment ${constants_1.OPERATOR_DEPLOYMENT_NAME} in namespace ${flags.chenamespace}`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield kube.deploymentExist(constants_1.OPERATOR_DEPLOYMENT_NAME, flags.chenamespace);
                    if (exist) {
                        task.title = `${task.title}...It already exists.`;
                    }
                    else {
                        const deploymentPath = path.join(ctx.resourcesPath, 'operator.yaml');
                        const operatorDeployment = yield this.readOperatorDeployment(deploymentPath, flags);
                        yield kube.createDeploymentFrom(operatorDeployment);
                        task.title = `${task.title}...done.`;
                    }
                })
            },
            {
                title: 'Operator pod bootstrap',
                task: () => kubeTasks.podStartTasks(constants_1.CHE_OPERATOR_SELECTOR, flags.chenamespace)
            },
            {
                title: 'Prepare CodeReady Workspaces cluster CR',
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const cheCluster = yield kube.getCheCluster(flags.chenamespace);
                    if (cheCluster) {
                        task.title = `${task.title}...It already exists..`;
                        return;
                    }
                    if (!ctx.customCR) {
                        const yamlFilePath = path.join(ctx.resourcesPath, 'crds', 'org_v1_che_cr.yaml');
                        ctx.defaultCR = util_1.safeLoadFromYamlFile(yamlFilePath);
                    }
                    task.title = `${task.title}...Done.`;
                })
            },
            common_tasks_1.createEclipseCheCluster(flags, kube)
        ], { renderer: flags['listr-renderer'] });
    }
    preUpdateTasks(flags, command) {
        const kube = new kube_1.KubeHelper(flags);
        return new Listr([
            {
                title: 'Checking existing operator deployment before update',
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const operatorDeployment = yield kube.getDeployment(constants_1.OPERATOR_DEPLOYMENT_NAME, flags.chenamespace);
                    if (!operatorDeployment) {
                        command.error(`${constants_1.OPERATOR_DEPLOYMENT_NAME} deployment is not found in namespace ${flags.chenamespace}.\nProbably CodeReady Workspaces was initially deployed with another installer`);
                    }
                    ctx.deployedCheOperatorYaml = operatorDeployment;
                    task.title = `${task.title}...done`;
                })
            },
            {
                title: 'Detecting existing version...',
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    ctx.deployedCheOperatorImage = this.retrieveContainerImage(ctx.deployedCheOperatorYaml);
                    const deployedCheOperatorImageAndTag = ctx.deployedCheOperatorImage.split(':', 2);
                    ctx.deployedCheOperatorImageName = deployedCheOperatorImageAndTag[0];
                    ctx.deployedCheOperatorImageTag = deployedCheOperatorImageAndTag.length === 2 ? deployedCheOperatorImageAndTag[1] : 'latest';
                    ctx.deployedCheOperatorImage = ctx.deployedCheOperatorImageName + ':' + ctx.deployedCheOperatorImageTag;
                    if (flags['che-operator-image']) {
                        ctx.newCheOperatorImage = flags['che-operator-image'];
                    }
                    else {
                        // Load new operator image from templates
                        const newCheOperatorYaml = util_1.safeLoadFromYamlFile(path.join(flags.templates, constants_1.OPERATOR_TEMPLATE_DIR, 'operator.yaml'));
                        ctx.newCheOperatorImage = this.retrieveContainerImage(newCheOperatorYaml);
                    }
                    const newCheOperatorImageAndTag = ctx.newCheOperatorImage.split(':', 2);
                    ctx.newCheOperatorImageName = newCheOperatorImageAndTag[0];
                    ctx.newCheOperatorImageTag = newCheOperatorImageAndTag.length === 2 ? newCheOperatorImageAndTag[1] : 'latest';
                    ctx.newCheOperatorImage = ctx.newCheOperatorImageName + ':' + ctx.newCheOperatorImageTag;
                    task.title = `${task.title} ${ctx.deployedCheOperatorImageTag} -> ${ctx.newCheOperatorImageTag}`;
                })
            }
        ]);
    }
    updateTasks(flags, command) {
        const kube = new kube_1.KubeHelper(flags);
        const ctx = context_1.ChectlContext.get();
        ctx.resourcesPath = path.join(flags.templates, constants_1.OPERATOR_TEMPLATE_DIR);
        return new Listr([
            {
                title: `Updating ServiceAccount ${this.operatorServiceAccount} in namespace ${flags.chenamespace}`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield kube.serviceAccountExist(this.operatorServiceAccount, flags.chenamespace);
                    const yamlFilePath = path.join(ctx.resourcesPath, 'service_account.yaml');
                    if (exist) {
                        yield kube.replaceServiceAccountFromFile(yamlFilePath, flags.chenamespace);
                        task.title = `${task.title}...updated.`;
                    }
                    else {
                        yield kube.createServiceAccountFromFile(yamlFilePath, flags.chenamespace);
                        task.title = `${task.title}...created new one.`;
                    }
                })
            },
            this.getReadRolesAndBindingsTask(kube),
            this.getCreateOrUpdateRolesAndBindingsTask(flags, 'Updating Roles and Bindings', true),
            {
                title: `Updating CodeReady Workspaces cluster CRD ${this.cheClusterCrd}`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const existedCRD = yield kube.getCrd(this.cheClusterCrd);
                    const newCRDPath = yield this.getCRDPath(ctx, flags);
                    if (existedCRD) {
                        if (!existedCRD.metadata || !existedCRD.metadata.resourceVersion) {
                            throw new Error(`Fetched CRD ${this.cheClusterCrd} without resource version`);
                        }
                        yield kube.replaceCrdFromFile(newCRDPath, existedCRD.metadata.resourceVersion);
                        task.title = `${task.title}...updated.`;
                    }
                    else {
                        yield kube.createCrdFromFile(newCRDPath);
                        task.title = `${task.title}...created new one.`;
                    }
                })
            },
            {
                title: `Updating CRD ${this.cheManagerCRD}`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (!(yield kube.IsAPIExtensionSupported('v1'))) {
                        task.title = `${task.title}...Skipped.`;
                        return;
                    }
                    const newCRDPath = path.join(ctx.resourcesPath, 'crds', 'chemanagers.che.eclipse.org.CustomResourceDefinition.yaml');
                    if (!fs.existsSync(newCRDPath)) {
                        task.title = `${task.title}...Skipped.`;
                        return;
                    }
                    const existedCRD = yield kube.getCrd(this.cheManagerCRD);
                    if (existedCRD) {
                        yield kube.replaceCrdFromFile(newCRDPath, existedCRD.metadata.resourceVersion);
                        task.title = `${task.title}...updated.`;
                    }
                    else {
                        yield kube.createCrdFromFile(newCRDPath);
                        task.title = `${task.title}...created new one.`;
                    }
                })
            },
            {
                title: `Updating CRD ${this.dwRoutingCRD}`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (!(yield kube.IsAPIExtensionSupported('v1'))) {
                        task.title = `${task.title}...Skipped.`;
                        return;
                    }
                    const newCRDPath = path.join(ctx.resourcesPath, 'crds', 'devworkspaceroutings.controller.devfile.io.CustomResourceDefinition.yaml');
                    if (!fs.existsSync(newCRDPath)) {
                        task.title = `${task.title}...Skipped.`;
                        return;
                    }
                    const existedCRD = yield kube.getCrd(this.dwRoutingCRD);
                    if (existedCRD) {
                        yield kube.replaceCrdFromFile(newCRDPath, existedCRD.metadata.resourceVersion);
                        task.title = `${task.title}...updated.`;
                    }
                    else {
                        yield kube.createCrdFromFile(newCRDPath);
                        task.title = `${task.title}...created new one.`;
                    }
                })
            },
            {
                title: 'Waiting 5 seconds for the new Kubernetes resources to get flushed',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield cli_ux_1.cli.wait(5000);
                    task.title = `${task.title}...done.`;
                })
            },
            {
                title: `Updating deployment ${constants_1.OPERATOR_DEPLOYMENT_NAME} in namespace ${flags.chenamespace}`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield kube.deploymentExist(constants_1.OPERATOR_DEPLOYMENT_NAME, flags.chenamespace);
                    const deploymentPath = path.join(ctx.resourcesPath, 'operator.yaml');
                    const operatorDeployment = yield this.readOperatorDeployment(deploymentPath, flags);
                    if (exist) {
                        yield kube.replaceDeploymentFrom(operatorDeployment);
                        task.title = `${task.title}...updated.`;
                    }
                    else {
                        yield kube.createDeploymentFrom(operatorDeployment);
                        task.title = `${task.title}...created new one.`;
                    }
                })
            },
            {
                title: 'Waiting newer operator to be run',
                task: (_ctx, _task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield cli_ux_1.cli.wait(1000);
                    yield kube.waitLatestReplica(constants_1.OPERATOR_DEPLOYMENT_NAME, flags.chenamespace);
                })
            },
            common_tasks_1.patchingEclipseCheCluster(flags, kube, command),
        ], { renderer: flags['listr-renderer'] });
    }
    /**
     * Returns list of tasks which remove CodeReady Workspaces operator related resources
     */
    deleteTasks(flags) {
        let kh = new kube_1.KubeHelper(flags);
        return [{
                title: 'Delete oauthClientAuthorizations',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const checluster = yield kh.getCheCluster(flags.chenamespace);
                    if (checluster && checluster.spec && checluster.spec.auth && checluster.spec.auth.oAuthClientName) {
                        const oAuthClientAuthorizations = yield kh.getOAuthClientAuthorizations(checluster.spec.auth.oAuthClientName);
                        yield kh.deleteOAuthClientAuthorizations(oAuthClientAuthorizations);
                    }
                    task.title = `${task.title}...OK`;
                })
            },
            {
                title: `Delete the Custom Resource of type ${constants_1.CHE_CLUSTER_CRD}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield kh.deleteCheCluster(flags.chenamespace);
                    // wait 20 seconds, default timeout in che operator
                    for (let index = 0; index < 20; index++) {
                        yield cli_ux_1.cli.wait(1000);
                        if (!(yield kh.getCheCluster(flags.chenamespace))) {
                            task.title = `${task.title}...OK`;
                            return;
                        }
                    }
                    // if checluster still exists then remove finalizers and delete again
                    const checluster = yield kh.getCheCluster(flags.chenamespace);
                    if (checluster) {
                        try {
                            yield kh.patchCheClusterCustomResource(checluster.metadata.name, flags.chenamespace, { metadata: { finalizers: null } });
                        }
                        catch (error) {
                            if (yield kh.getCheCluster(flags.chenamespace)) {
                                task.title = `${task.title}...OK`;
                                return; // successfully removed
                            }
                            throw error;
                        }
                        // wait 2 seconds
                        yield cli_ux_1.cli.wait(2000);
                    }
                    if (!(yield kh.getCheCluster(flags.chenamespace))) {
                        task.title = `${task.title}...OK`;
                    }
                    else {
                        task.title = `${task.title}...Failed`;
                    }
                })
            },
            {
                title: `Delete CRD ${this.cheClusterCrd}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const checlusters = yield kh.getAllCheClusters();
                    if (checlusters.length > 0) {
                        task.title = `${task.title}...Skipped: another CodeReady Workspaces deployment found.`;
                    }
                    else {
                        yield kh.deleteCrd(this.cheClusterCrd);
                        task.title = `${task.title}...OK`;
                    }
                })
            },
            {
                title: 'Delete Roles and Bindings',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const roleBindings = yield kh.listRoleBindings(flags.chenamespace);
                    for (const roleBinding of roleBindings.items) {
                        yield kh.deleteRoleBinding(roleBinding.metadata.name, flags.chenamespace);
                    }
                    const roles = yield kh.listRoles(flags.chenamespace);
                    for (const role of roles.items) {
                        yield kh.deleteRole(role.metadata.name, flags.chenamespace);
                    }
                    // Count existing pairs of cluster roles and thier bindings
                    let pairs = 0;
                    const clusterRoleBindings = yield kh.listClusterRoleBindings();
                    for (const clusterRoleBinding of clusterRoleBindings.items) {
                        const name = clusterRoleBinding.metadata && clusterRoleBinding.metadata.name || '';
                        if (name.startsWith(flags.chenamespace)) {
                            pairs++;
                            yield kh.deleteClusterRoleBinding(name);
                        }
                    }
                    const clusterRoles = yield kh.listClusterRoles();
                    for (const clusterRole of clusterRoles.items) {
                        const name = clusterRole.metadata && clusterRole.metadata.name || '';
                        if (name.startsWith(flags.chenamespace)) {
                            yield kh.deleteClusterRole(name);
                        }
                    }
                    // If no pairs were deleted, then legacy names is used
                    if (pairs === 0) {
                        yield kh.deleteClusterRoleBinding(this.legacyClusterResourcesName);
                        yield kh.deleteClusterRole(this.legacyClusterResourcesName);
                    }
                    task.title = `${task.title}...OK`;
                })
            },
            {
                title: `Delete service accounts ${this.operatorServiceAccount}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield kh.deleteServiceAccount(this.operatorServiceAccount, flags.chenamespace);
                    task.title = `${task.title}...OK`;
                })
            },
            {
                title: 'Delete PVC codeready-operator',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield kh.deletePersistentVolumeClaim('codeready-operator', flags.chenamespace);
                    task.title = `${task.title}...OK`;
                })
            },
        ];
    }
    retrieveContainerImage(deployment) {
        const containers = deployment.spec.template.spec.containers;
        const namespace = deployment.metadata.namespace;
        const name = deployment.metadata.name;
        const container = containers.find(c => c.name === 'codeready-operator');
        if (!container) {
            throw new Error(`Can not evaluate image of ${namespace}/${name} deployment. Containers list are empty`);
        }
        if (!container.image) {
            throw new Error(`Container ${container.name} in deployment ${namespace}/${name} must have image specified`);
        }
        return container.image;
    }
    getCRDPath(ctx, flags) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            let newCRDFilePath;
            const kube = new kube_1.KubeHelper(flags);
            if (!(yield kube.IsAPIExtensionSupported('v1'))) {
                // try to get CRD v1beta1 if platform doesn't support v1
                newCRDFilePath = path.join(ctx.resourcesPath, 'crds', 'org_v1_che_crd-v1beta1.yaml');
                if (fs.existsSync(newCRDFilePath)) {
                    return newCRDFilePath;
                }
            }
            return path.join(ctx.resourcesPath, 'crds', 'org_v1_che_crd.yaml');
        });
    }
    /**
     * Reads and patch 'codeready-operator' deployment:
     * - sets operator image
     * - sets deployment namespace
     * - removes other containers for ocp 3.11
     */
    readOperatorDeployment(path, flags) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const operatorDeployment = util_1.safeLoadFromYamlFile(path);
            if (!operatorDeployment.metadata || !operatorDeployment.metadata.name) {
                throw new Error(`Deployment read from ${path} must have name specified`);
            }
            if (flags['che-operator-image']) {
                const container = operatorDeployment.spec.template.spec.containers.find(c => c.name === 'codeready-operator');
                if (container) {
                    container.image = flags['che-operator-image'];
                }
                else {
                    throw new Error(`Container 'codeready-operator' not found in deployment '${operatorDeployment.metadata.name}'`);
                }
            }
            if (flags.chenamespace) {
                operatorDeployment.metadata.namespace = flags.chenamespace;
            }
            const kube = new kube_1.KubeHelper(flags);
            if (!(yield kube.IsAPIExtensionSupported('v1'))) {
                const containers = operatorDeployment.spec.template.spec.containers || [];
                operatorDeployment.spec.template.spec.containers = containers.filter(c => c.name === 'codeready-operator');
            }
            return operatorDeployment;
        });
    }
}
exports.OperatorTasks = OperatorTasks;
//# sourceMappingURL=operator.js.map