"use strict";
/*********************************************************************
 * Copyright (c) 2020 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 **********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
exports.getPrintHighlightedMessagesTask = exports.getRetrieveKeycloakCredentialsTask = exports.getMessageImportCaCertIntoBrowser = exports.retrieveCheCaCertificateTask = exports.patchingEclipseCheCluster = exports.createEclipseCheCluster = exports.downloadTemplates = exports.checkChectlAndCheVersionCompatibility = exports.createNamespaceTask = void 0;
const tslib_1 = require("tslib");
const ansi = require("ansi-colors");
const fs = require("fs-extra");
const Listr = require("listr");
const lodash_1 = require("lodash");
const path = require("path");
const semver = require("semver");
const che_1 = require("../../api/che");
const context_1 = require("../../api/context");
const github_client_1 = require("../../api/github-client");
const kube_1 = require("../../api/kube");
const version_1 = require("../../api/version");
const constants_1 = require("../../constants");
const util_1 = require("../../util");
function createNamespaceTask(namespaceName, labels) {
    return {
        title: `Create Namespace (${namespaceName})`,
        task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            const kube = new kube_1.KubeHelper();
            const che = new che_1.CheHelper({});
            const namespace = yield kube.getNamespace(namespaceName);
            if (namespace) {
                yield che.waitNamespaceActive(namespaceName);
                task.title = `${task.title}...It already exists.`;
            }
            else {
                yield kube.createNamespace(namespaceName, labels);
                yield che.waitNamespaceActive(namespaceName);
                task.title = `${task.title}...Done.`;
            }
        })
    };
}
exports.createNamespaceTask = createNamespaceTask;
function checkChectlAndCheVersionCompatibility(flags) {
    return {
        title: 'Check versions compatibility',
        enabled: ctx => ctx.isChectl && !flags.templates && flags.version && flags.installer !== 'olm',
        task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            const githubClient = new github_client_1.CheGithubClient();
            const verInfo = yield githubClient.getTemplatesTagInfo(flags.installer, flags.version);
            if (!verInfo) {
                throw new Error(`Version ${flags.version} does not exist`);
            }
            ctx.versionInfo = verInfo;
            flags.version = version_1.VersionHelper.removeVPrefix(verInfo.name, true);
            if (!ctx.isNightly && semver.lt(util_1.getProjectVersion(), flags.version)) {
                throw new Error(`To deploy CodeReady Workspaces ${flags.version}, please update your crwctl first by running "crwctl update".`);
            }
            task.title = `${task.title}... OK`;
        })
    };
}
exports.checkChectlAndCheVersionCompatibility = checkChectlAndCheVersionCompatibility;
/**
 * Sets flags.templates based on required version and installer.
 * Does not support OLM.
 */
function downloadTemplates(flags) {
    return {
        title: 'Download templates',
        enabled: ctx => ctx.isChectl && flags.version && !flags.templates && flags.installer !== 'olm',
        task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            // All templates are stored in the cache directory
            // Example path: ~/.cache/crwctl/templates/7.15.1/
            const templatesRootDir = path.join(ctx[context_1.ChectlContext.CACHE_DIR], 'templates');
            let installerTemplatesSubDir;
            switch (flags.installer) {
                case 'operator':
                    installerTemplatesSubDir = constants_1.OPERATOR_TEMPLATE_DIR;
                    break;
                case 'helm':
                    installerTemplatesSubDir = 'kubernetes';
                    break;
                case 'olm':
                    // Should be handled on install phase when catalog source is deployed
                    return;
                default:
                    throw new Error(`Unknow installer ${flags.installer}`);
            }
            const versionTemplatesDirPath = path.join(templatesRootDir, flags.version);
            flags.templates = versionTemplatesDirPath;
            const installerTemplatesDirPath = path.join(versionTemplatesDirPath, installerTemplatesSubDir);
            if (yield fs.pathExists(installerTemplatesDirPath)) {
                // Use cached templates
                task.title = `${task.title}... found cached templates for version ${flags.version}`;
                return;
            }
            // Download templates
            task.title = `${task.title} for version ${flags.version}`;
            const cheHelper = new che_1.CheHelper(flags);
            yield cheHelper.downloadAndUnpackTemplates(flags.installer, ctx.versionInfo.zipball_url, versionTemplatesDirPath);
            task.title = `${task.title} ... OK`;
        })
    };
}
exports.downloadTemplates = downloadTemplates;
function createEclipseCheCluster(flags, kube) {
    return {
        title: `Create the Custom Resource of type ${constants_1.CHE_CLUSTER_CRD} in the namespace ${flags.chenamespace}`,
        enabled: ctx => !!ctx.customCR || !!ctx.defaultCR,
        task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            ctx.isCheDeployed = true;
            ctx.isPostgresDeployed = true;
            ctx.isKeycloakDeployed = true;
            ctx.isDashboardDeployed = false;
            // Check if the installed version support dashboard deployment checking `RELATED_IMAGE_dashboard` operator environment
            const operatorDeployment = yield kube.getDeployment('codeready-operator', flags.chenamespace);
            if (operatorDeployment && operatorDeployment.spec && operatorDeployment.spec.template.spec) {
                const operatorContainer = operatorDeployment.spec.template.spec.containers.find(c => c.name === 'codeready-operator');
                if (operatorContainer && operatorContainer.env) {
                    ctx.isDashboardDeployed = operatorContainer.env.some(env => env.name === 'RELATED_IMAGE_dashboard');
                }
            }
            // plugin and devfile registry will be deployed only when external ones are not configured
            ctx.isPluginRegistryDeployed = !flags['plugin-registry-url'];
            ctx.isDevfileRegistryDeployed = !flags['devfile-registry-url'];
            const cheClusterCR = ctx.customCR || ctx.defaultCR;
            const cr = yield kube.createCheCluster(cheClusterCR, flags, ctx, !ctx.customCR);
            ctx.isKeycloakReady = ctx.isKeycloakReady || cr.spec.auth.externalIdentityProvider;
            ctx.isPostgresReady = ctx.isPostgresReady || cr.spec.database.externalDb;
            ctx.isDevfileRegistryReady = ctx.isDevfileRegistryReady || cr.spec.server.externalDevfileRegistry;
            ctx.isPluginRegistryReady = ctx.isPluginRegistryReady || cr.spec.server.externalPluginRegistry;
            if (cr.spec.server.customCheProperties && cr.spec.server.customCheProperties.CHE_MULTIUSER === 'false') {
                flags.multiuser = false;
            }
            task.title = `${task.title}...done.`;
        })
    };
}
exports.createEclipseCheCluster = createEclipseCheCluster;
/**
 * Update CheCluster CR object using CR patch file.
 * Clean up custom images if they weren't defined in the CR patch file to prevent update failing.
 * @param flags - parent command flags
 * @param kube - kubeHelper util
 * @param command - parent command
 */
function patchingEclipseCheCluster(flags, kube, command) {
    return {
        title: `Patching the Custom Resource of type '${constants_1.CHE_CLUSTER_CRD}' in the namespace '${flags.chenamespace}'`,
        skip: (ctx) => lodash_1.isEmpty(ctx[context_1.ChectlContext.CR_PATCH]),
        task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            const cheCluster = yield kube.getCheCluster(flags.chenamespace);
            if (!cheCluster) {
                command.error(`CodeReady Workspaces cluster CR is not found in the namespace '${flags.chenamespace}'`);
            }
            yield kube.patchCheCluster(cheCluster.metadata.name, flags.chenamespace, ctx[context_1.ChectlContext.CR_PATCH]);
            task.title = `${task.title}...done.`;
        })
    };
}
exports.patchingEclipseCheCluster = patchingEclipseCheCluster;
function retrieveCheCaCertificateTask(flags) {
    return {
        title: 'Retrieving Che self-signed CA certificate',
        // It makes sense to retrieve CA certificate only if self-signed certificate is used.
        enabled: () => flags.tls && flags.installer !== 'helm',
        task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            const che = new che_1.CheHelper(flags);
            const kube = new kube_1.KubeHelper();
            const cheCaCert = yield che.retrieveCheCaCert(flags.chenamespace);
            if (cheCaCert) {
                const targetFile = yield che.saveCheCaCert(cheCaCert);
                task.title = `${task.title}... done`;
                const serverStrategy = yield kube.getConfigMapValue('che', flags.chenamespace, 'CHE_INFRA_KUBERNETES_SERVER__STRATEGY');
                if (serverStrategy !== 'single-host') {
                    ctx.highlightedMessages.push(getMessageImportCaCertIntoBrowser(targetFile));
                }
            }
            else {
                task.title = `${task.title}... commonly trusted certificate is used.`;
            }
        })
    };
}
exports.retrieveCheCaCertificateTask = retrieveCheCaCertificateTask;
function getMessageImportCaCertIntoBrowser(caCertFileLocation) {
    const message = `${ansi.yellow('[ACTION REQUIRED]')} Please add Che self-signed CA certificate into your browser: ${caCertFileLocation}.\n` +
        `Documentation how to add a CA certificate into a browser: ${constants_1.DOCS_LINK_IMPORT_CA_CERT_INTO_BROWSER}`;
    return message;
}
exports.getMessageImportCaCertIntoBrowser = getMessageImportCaCertIntoBrowser;
function getRetrieveKeycloakCredentialsTask(flags) {
    return {
        title: 'Retrieving Keycloak admin credentials',
        enabled: () => (flags.installer !== 'helm'),
        task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            const che = new che_1.CheHelper(flags);
            const [login, password] = yield che.retrieveKeycloakAdminCredentials(flags.chenamespace);
            if (login && password) {
                ctx.identityProviderUsername = login;
                ctx.identityProviderPassword = password;
                task.title = `${task.title}...done`;
            }
            else {
                task.title = `${task.title}...failed.`;
            }
        })
    };
}
exports.getRetrieveKeycloakCredentialsTask = getRetrieveKeycloakCredentialsTask;
/**
 * Prints important to user messages which are stored in ctx.highlightedMessages
 * Typically this task is the last task of a command.
 */
function getPrintHighlightedMessagesTask() {
    return {
        title: 'Show important messages',
        enabled: ctx => ctx.highlightedMessages && ctx.highlightedMessages.length > 0,
        task: (ctx) => {
            const printMessageTasks = new Listr([], ctx.listrOptions);
            for (const message of ctx.highlightedMessages) {
                printMessageTasks.add({
                    title: message,
                    task: () => { }
                });
            }
            return printMessageTasks;
        }
    };
}
exports.getPrintHighlightedMessagesTask = getPrintHighlightedMessagesTask;
//# sourceMappingURL=common-tasks.js.map