__author__ = 'slaskawi@redhat.com'

import os
import logging
import subprocess


class RemoteHostReleaseHandler:
    def __init__(self, configuration):
        self._configuration = configuration

    def upload_dependency_information(self):
        logging.info("Uploading Brew dependency information")
        what = [self._configuration.get_packaging_repo_location() + "/target/dependency-report/jdg-artifacts.txt"]
        where = [self._configuration.get_release_destination_for_artifacts()]
        self._scp_to_remote_host(what, where)

    def prepare_release_directory(self):
        logging.info("Preparing release directory")
        self._run_command_on_remote_host("mkdir -p " + self._configuration.get_release_scripts_destination_directory())

        logging.info("Uploading scripts")
        scripts_path = self._configuration.get_packaging_repo_location() + "/bin"
        what = self._list_files_only(scripts_path)
        where = [self._configuration.get_release_destination_scripts()]
        self._scp_to_remote_host(what, where)

    def run_release_script(self):
        logging.info("Invoking release script")
        command = "cd " + self._configuration.get_release_scripts_destination_directory() + ";./jdg-release.sh " + self._configuration.get_version_modifier()
        self._run_command_on_remote_host(command)

    def clean_release_area(self):
        logging.info("Cleanning up release scripts")
        scripts_path = self._configuration.get_packaging_repo_location() + "/bin"
        command = "rm " + " ".join(self._list_files_only(scripts_path, self._configuration.get_release_scripts_destination_directory()))
        self._run_command_on_remote_host(command)
        
    def upload_quickstarts(self):
        logging.info("Uploading Quickstarts package")   
        what = [self._configuration.get_quickstarts_zip_location()]
        where = [self._configuration.get_release_destination_scripts()]
        self._scp_to_remote_host(what, where)
        self._run_command_on_remote_host("cd " + self._configuration.get_release_scripts_destination_directory() + ";./checksums.sh ")

    def update_release_permissions(self, group, mode):
        logging.info("Uploading Permissions on release")
        self._run_command_on_remote_host("chgrp -R " + group + " " +
                                          self._configuration.get_release_scripts_destination_directory())
        self._run_command_on_remote_host("chmod -R " + mode + " " +
                                          self._configuration.get_release_scripts_destination_directory())           
        self._run_command_on_remote_host("chmod " + mode + " " + 
                                         self._configuration.get_relese_destination_directory() + "/jdg-artifacts.txt")
        self._run_command_on_remote_host("chgrp " + group + " " + 
                                         self._configuration.get_relese_destination_directory() + "/jdg-artifacts.txt")
                
    def _run_command_on_remote_host(self, command_to_execute):
        command = ["ssh", "-K", self._configuration.release_destination_host()]
        command.extend([command_to_execute])
        logging.debug("Invoking remote ssh command with arguments: " + str(command))
        subprocess.check_call(command)

    def _scp_to_remote_host(self, what, where):
        command = ["scp"]
        command.extend(what)
        command.extend(where)
        logging.debug("Invoking scp with arguments: " + str(command))
        subprocess.check_call(command)

    def _list_files_only(self, path, override_path=None):
        files = []
        for name in os.listdir(path):
            if os.path.isfile(os.path.join(path, name)):
                if override_path is None:
                    files.append(os.path.join(path, name))
                else:
                    files.append(os.path.join(override_path, name))
        return files