/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.component.infinispan;

import org.apache.camel.Exchange;
import org.apache.camel.Processor;
import org.apache.camel.impl.DefaultConsumer;
import org.infinispan.Cache;
import org.infinispan.client.hotrod.RemoteCache;
import org.infinispan.client.hotrod.RemoteCacheManager;
import org.infinispan.commons.api.BasicCacheContainer;
import org.infinispan.manager.EmbeddedCacheManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class InfinispanConsumer extends DefaultConsumer {
    private static final transient Logger LOGGER = LoggerFactory.getLogger(InfinispanProducer.class);
    private final InfinispanConfiguration configuration;
    private final InfinispanEventListener listener;
    private EmbeddedCacheManager cacheManager;
    private RemoteCache<Object, Object> remoteCache;

    public InfinispanConsumer(InfinispanEndpoint endpoint, Processor processor, InfinispanConfiguration configuration) {
        super(endpoint, processor);
        this.configuration = configuration;
        if (configuration.isCustom()) {
            InfinispanCustomListener customListener = configuration.getCustomListener();
            customListener.setInfinispanConsumer(this);
            listener = customListener;
        } else if (configuration.isClustered()) {
            if (configuration.isSync()) {
                listener = new InfinispanSyncClusteredEventListener(this, configuration.getEventTypes());
            } else {
                listener = new InfinispanAsyncClusteredEventListener(this, configuration.getEventTypes());
            }
        } else {
            if (configuration.isSync()) {
                listener = new InfinispanSyncLocalEventListener(this, configuration.getEventTypes());
            } else {
                listener = new InfinispanAsyncLocalEventListener(this, configuration.getEventTypes());
            }
        }
    }

    public void processEvent(String eventType, boolean isPre, String cacheName, Object key) {
        processEvent(eventType, isPre, cacheName, key, null);
    }

    public void processEvent(String eventType, boolean isPre, String cacheName, Object key, Object eventData) {
        Exchange exchange = getEndpoint().createExchange();
        exchange.getOut().setHeader(InfinispanConstants.EVENT_TYPE, eventType);
        exchange.getOut().setHeader(InfinispanConstants.IS_PRE, isPre);
        exchange.getOut().setHeader(InfinispanConstants.CACHE_NAME, cacheName);
        exchange.getOut().setHeader(InfinispanConstants.KEY, key);
        if (eventData != null) {
            exchange.getOut().setHeader(InfinispanConstants.EVENT_DATA, eventData);
        }

        try {
            getProcessor().process(exchange);
        } catch (Exception e) {
            LOGGER.error("Error processing event ", e);
        }
    }

    @Override
    protected void doStart() throws Exception {
        BasicCacheContainer cacheContainer = configuration.getCacheContainer();
        if (cacheContainer instanceof EmbeddedCacheManager) {
            cacheManager = (EmbeddedCacheManager) cacheContainer;
            Cache<Object, Object> cache;
            if (configuration.getCacheName() != null) {
                cache = cacheManager.getCache(configuration.getCacheName());
            } else {
                cache = cacheManager.getCache();
            }
            cache.addListener(listener);
        } else if (cacheContainer instanceof RemoteCacheManager) {
            if (configuration.isSync()) {
                throw new UnsupportedOperationException("Sync listeners not supported for remote caches.");
            }
            RemoteCacheManager remoteCacheManager = (RemoteCacheManager) cacheContainer;
            RemoteCache<Object, Object> cache;
            if (configuration.getCacheName() != null) {
                cache = remoteCacheManager.getCache(configuration.getCacheName());
            } else {
                cache = remoteCacheManager.getCache();
            }
            listener.setCacheName(cache.getName());
            cache.addClientListener(listener);
        } else {
            throw new UnsupportedOperationException("Consumer not support for CacheContainer: " + cacheContainer);
        }
        super.doStart();
    }

    @Override
    protected void doStop() throws Exception {
        if (cacheManager != null) {
            cacheManager.removeListener(listener);
        }
        if (remoteCache != null) {
            remoteCache.removeClientListener(listener);
        }
        super.doStop();
    }
}
