
 package org.infinispan.marshall.exts;

import static org.testng.AssertJUnit.assertEquals;
import static org.testng.AssertJUnit.assertSame;

import java.io.IOException;
import java.io.ObjectOutput;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.TreeMap;

import org.infinispan.commons.CacheException;
import org.infinispan.commons.equivalence.AnyEquivalence;
import org.infinispan.commons.equivalence.EquivalentHashMap;
import org.infinispan.commons.io.ByteBuffer;
import org.infinispan.commons.io.ByteBufferFactory;
import org.infinispan.commons.marshall.AdvancedExternalizer;
import org.infinispan.commons.marshall.StreamingMarshaller;
import org.infinispan.commons.util.FastCopyHashMap;
import org.infinispan.configuration.cache.ConfigurationBuilder;
import org.infinispan.configuration.cache.CustomStoreConfigurationBuilder;
import org.infinispan.io.ExpandableMarshalledValueByteStream;
import org.infinispan.manager.EmbeddedCacheManager;
import org.infinispan.marshall.core.MarshalledEntry;
import org.infinispan.persistence.spi.CacheLoader;
import org.infinispan.persistence.spi.InitializationContext;
import org.infinispan.test.SingleCacheManagerTest;
import org.infinispan.test.fwk.TestCacheManagerFactory;
import org.testng.annotations.Test;

 @Test(groups = "functional", testName = "marshall.exts.TestMapDifferentSerializedFormsWithCacheLoader")
public class TestMapDifferentSerializedFormsWithCacheLoader extends SingleCacheManagerTest {

    private static <T> byte[] getSerializedForm(StreamingMarshaller marshaller, AdvancedExternalizer<? super T> externalizer, T object) throws IOException {
       ExpandableMarshalledValueByteStream baos = new ExpandableMarshalledValueByteStream();
       ObjectOutput output = marshaller.startObjectOutput(baos, false, 0);
       // This is written by jboss marshalling to tell to use object tables
       output.writeByte(0x03);
       output.writeByte(externalizer.getId());
       externalizer.writeObject(output, object);
       return baos.getRaw();
    }

   @Override
   protected EmbeddedCacheManager createCacheManager() throws Exception {
      final ConfigurationBuilder config = getDefaultStandaloneCacheConfig(false);
      config
         .persistence().addStore(CustomStoreConfigurationBuilder.class).customStoreClass(ForcedSerializationLoader.class);
      return TestCacheManagerFactory.createCacheManager(config);
   }

   void testExternalizer(Map<String, String> map, Class<? extends AdvancedExternalizer<Map>> externalizer)
         throws InstantiationException, IllegalAccessException {
      Class<?> mapClass = map.getClass();
      map.put(mapClass.toString(), mapClass + "-value");
      ForcedSerializationLoader.externalizerToUse = externalizer.newInstance();
      Map<String, String> resultingMap = (Map<String, String>) cache.get(mapClass);
      // FastCopyHashMap doesn't properly implement equality
      assertSame(map.getClass(), resultingMap.getClass());
      assertEquals(map.size(), resultingMap.size());
      for (Entry<String, String> entry : map.entrySet()) {
         assertEquals(resultingMap.get(entry.getKey()), entry.getValue());
      }
   }

   public void testMapExternalizer() throws InstantiationException, IllegalAccessException {
      testExternalizer(new HashMap<String, String>(), MapExternalizer.class);
   }

   public void testReuseMapExternalizer() throws InstantiationException, IllegalAccessException {
      testExternalizer(new HashMap<String, String>(), MapExternalizer.InstanceReusingMapExternalizer.class);
   }

   public void testTreeMapExternalizer() throws InstantiationException, IllegalAccessException {
      testExternalizer(new TreeMap<String, String>(), MapExternalizer.class);
   }

   public void testReuseTreeMapExternalizer() throws InstantiationException, IllegalAccessException {
      testExternalizer(new TreeMap<String, String>(), MapExternalizer.InstanceReusingMapExternalizer.class);
   }

   public void testEquivalentMapExternalizer() throws InstantiationException, IllegalAccessException {
      testExternalizer(new EquivalentHashMap<String, String>(AnyEquivalence.STRING, AnyEquivalence.STRING), MapExternalizer.class);
   }

   public void testReuseEquivalentMapExternalizer() throws InstantiationException, IllegalAccessException {
      testExternalizer(new EquivalentHashMap<String, String>(AnyEquivalence.STRING, AnyEquivalence.STRING),
            MapExternalizer.InstanceReusingMapExternalizer.class);
   }

   public void testFastCopyMapExternalizer() throws InstantiationException, IllegalAccessException {
      testExternalizer(new FastCopyHashMap<String, String>(), MapExternalizer.class);
   }

   public void testReuseFastCopyMapExternalizer() throws InstantiationException, IllegalAccessException {
      testExternalizer(new FastCopyHashMap<String, String>(), MapExternalizer.InstanceReusingMapExternalizer.class);
   }

   public static class ForcedSerializationLoader implements CacheLoader<Class, Map<String, String>> {
      protected InitializationContext ctx;

      protected static AdvancedExternalizer<Map> externalizerToUse;

      @Override
      public void start() { }

      @Override
      public void stop() { }

      @Override
      public void init(InitializationContext ctx) {
         this.ctx = ctx;
      }

      @Override
      public MarshalledEntry<Class, Map<String, String>> load(Class key) {
         Map<String, String> mapToUse;
         if (key == HashMap.class) {
            mapToUse = new HashMap<String, String>();
         } else if (key == TreeMap.class) {
            mapToUse = new TreeMap<String, String>();
         } else if (key == EquivalentHashMap.class) {
            mapToUse = new EquivalentHashMap<String, String>(AnyEquivalence.STRING, AnyEquivalence.STRING);
         } else if (key == FastCopyHashMap.class) {
            mapToUse = new FastCopyHashMap<String, String>();
         } else {
            throw new IllegalArgumentException("Class " + key + " not supported !");
         }
         mapToUse.put(key.toString(), key + "-value");
         try {
            ByteBufferFactory factory = ctx.getByteBufferFactory();
            byte[] bytes = getSerializedForm(ctx.getMarshaller(), externalizerToUse, mapToUse);
            ByteBuffer buf = factory.newByteBuffer(bytes, 0, bytes.length);
            return ctx.getMarshalledEntryFactory().newMarshalledEntry(key, buf, (ByteBuffer) null);
         } catch (IOException e) {
            throw new CacheException(e);
         }
      }

      @Override
      public boolean contains(Class key) {
         throw new UnsupportedOperationException();
      }
      
   }
}
