package com.redhat.installer.postinstall;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.List;

import com.izforge.izpack.Pack;
import com.izforge.izpack.installer.AutomatedInstallData;
import com.izforge.izpack.util.AbstractUIProcessHandler;
import com.redhat.installer.util.ValidatorConstants;

/**
 * Simple class to set the layers.conf and product.conf file contents to be correct, based upon the matrix here:
 * https://docspace.corp.redhat.com/docs/DOC-146624
 * 
 * @author thauser
 *
 */
public class ProductLayersConfSetter {
														// ProcessPanel window
	private static AutomatedInstallData idata = AutomatedInstallData.getInstance();

	public static void run(AbstractUIProcessHandler handler, String[] args) throws IOException {
		// get the paths of the files in question
		String installPath = idata.getVariable("INSTALL_PATH");
		String installSubPath = idata.getVariable("INSTALL_SUBPATH");
		String basePath = installPath + File.separator + installSubPath;
		String layersPath = basePath + ValidatorConstants.layersConfLoc;
		String productPath = basePath + ValidatorConstants.productConfLoc;

		// read the variables for layers / product.conf
		// the current layers.conf / product.conf contents
		String existingLayers = idata.getVariable(ValidatorConstants.existingLayers);
		//System.out.println("existingLayers = "+existingLayers);
		String existingProduct = idata.getVariable(ValidatorConstants.existingProduct);
		//System.out.println("existingProduct = "+existingProduct);
		// the contents the product.conf / layers.conf should have after
		// installation is done
		String newLayers = getNewLayersString();
		//System.out.println("newLayers = "+newLayers);
		String newProduct = getNewProductString();
		//System.out.println("newProduct = "+newProduct);
		String finalLayers = ""; // hold the value to write to layers.conf
		String finalProduct = ""; // hold the value to write to product.conf
		
		finalProduct = getFinalProductString(existingProduct, newProduct);
		//idata.setVariable("final.product.conf", finalProduct);
		//System.out.println("FinalProduct = " +finalProduct);
		finalLayers = getFinalLayersString(existingLayers, newLayers);
		//idata.setVariable("final.layers.conf", finalLayers);
		//System.out.println("FinalLayers = " +finalLayers);
		
		writeConfFiles(basePath, finalProduct, finalLayers, productPath, layersPath);

	}
	
	/**
	 * General method that uses pack-ids to decide what the product.conf needs to contain. 
	 * Much easier logic than the layers.conf
	 * @return
	 */
	
	private static String getNewProductString(){
		String product = ValidatorConstants.eap;
		List<Pack> installedPacks = idata.selectedPacks;
		
		for (Pack pack : installedPacks){
			if (pack.id.startsWith(ValidatorConstants.soa)){
				product=ValidatorConstants.soa;
				break; // nothing can beat soa
			} else if (pack.id.startsWith(ValidatorConstants.brms)){
				product=ValidatorConstants.brms;
			} else if (pack.id.startsWith(ValidatorConstants.bpms)){
				product=ValidatorConstants.bpms;
			} else if (pack.id.contains(ValidatorConstants.sramp)){
				product=ValidatorConstants.sramp;
			}
		}
		return product;
	}

	/**
	 * General method that uses pack-ids to decide what the layers needs to contain.
	 * Unfortunately, we need to rely on hardcoded pack-ids to decide about what the newLayers needs to contain.
	 * As the amount of installers grows, so will this need to grow
	 * @return
	 */
	private static String getNewLayersString() {
		String layers = "";
		List<Pack> installedPacks = idata.selectedPacks;
		
		/**
		 * Decide if layers has soa. if it does, it goes first always
		 */
		for (Pack pack : installedPacks){
			// special case for RT-Gov
			if (pack.id.startsWith(ValidatorConstants.soa) || pack.id.contains("rt-govs") || pack.id.contains("rt-govc")){
				layers+=ValidatorConstants.soa;
				break;
			}
		}
		
		/**
		 * Decide about sramp. it goes second
		 */
		
		for (Pack pack : installedPacks){
			if (pack.id.startsWith(ValidatorConstants.sramp)){
				if (!layers.isEmpty()){ // layers has stuff in it already
					layers+=",";
				}
				layers+=ValidatorConstants.sramp;
				break; // met the requirements to add sramp to layers. short circuit
			}
		}
		
		/**
		 * BPMS | BRMS go here. they are mutually exclusive, so 
		 * we can combine them here. unfortunately, the pack names are not final yet
		 */
		
		for (Pack pack : installedPacks){
			if (pack.id.startsWith(ValidatorConstants.brms)){
				if (!layers.isEmpty()){
					layers+=",";
				}
				layers+=ValidatorConstants.brms; // a little clever, only will work if pack names are equal to the constants
				break;
			} else if (pack.id.startsWith(ValidatorConstants.bpms)){
				if (!layers.isEmpty()){
					layers+=",";
				}
				layers+=ValidatorConstants.bpms;
				break;
			}
		}
		
		/**
		 * EDS. the last one. 
		 */
		for (Pack pack : installedPacks){
			if (pack.id.startsWith(ValidatorConstants.dv)){ // TODO: may need to change condition
				if (!layers.isEmpty()){
					layers+=",";
				}
				layers+=ValidatorConstants.dv;
				break; // met requirements to add eds to layers. short circuit
			}
		}
		return layers;
	}

	/** 
	 * Writes out the layers.conf and product.conf with the new, correct values
	 * @param finalProduct
	 * @param finalLayers
	 * @param productPath
	 * @param layersPath
	 * @throws IOException 
	 */
	private static void writeConfFiles(String basePath, String finalProduct, String finalLayers, String productPath, String layersPath) throws IOException {
		File productConf = new File(productPath);
		File productBackup = new File(productPath+ValidatorConstants.backupExt);
		File layersConf = new File(layersPath);
		File layersBackup = new File(layersPath+ValidatorConstants.backupExt);
		
		// create backups (quick hacks)
		// TODO: refactor in a more clean way
		productConf.renameTo(productBackup);
		//System.out.println("backup for product written: " + productBackup.getPath());
		layersConf.renameTo(layersBackup);
		//System.out.println("backup for layers written: " + layersBackup.getPath());
		
		BufferedWriter prodOut = new BufferedWriter(new FileWriter(productConf));
		
		prodOut.write("#product.conf written by the platform installer\n");
		prodOut.write("slot="+finalProduct);
		
		BufferedWriter layersOut = new BufferedWriter(new FileWriter(layersConf));
		
		layersOut.write("#layers.conf written by the platform installer\n");
		layersOut.write("layers="+finalLayers);
		
		prodOut.close();
		layersOut.close();
	}

	/**
	 * Decides what the final contents of the new layers.conf needs to contain. There is no error checking here, because
	 * the validators upon the install path need to catch every impossible combination. 
	 * @param existingLayers
	 * @param newLayers
	 * @return
	 */
	private static String getFinalLayersString(String existingLayers, String newLayers) {
		String finalLayers = "";
		boolean brmsInstall = newLayers.equals(ValidatorConstants.brms);
		boolean edsInstall = newLayers.equals(ValidatorConstants.dv);
		boolean soaInstall = newLayers.equals(ValidatorConstants.soa);
		boolean bpmsInstall = newLayers.equals(ValidatorConstants.bpms);
		boolean srampInstall = newLayers.equals(ValidatorConstants.sramp); 
		
		// Stock EAP 
		if (existingLayers.trim().isEmpty()) {
			/*if (soaInstall) {
				finalLayers = ValidatorConstants.soa;
			} else if (brmsInstall) {
				finalLayers = ValidatorConstants.brms;
			}else if (bpmsInstall) {
				finalLayers = ValidatorConstants.bpms;
			}else if (edsInstall) {
				finalLayers = ValidatorConstants.eds;
			}else if (srampInstall) {
				finalLayers = ValidatorConstants.sramp;
			}*/
			finalLayers = newLayers;
		} else if (existingLayers.equals(ValidatorConstants.dv)){ // EDS 
			//installing onto eap + eds system
			if (soaInstall){
				finalLayers = ValidatorConstants.soaEds;
			} else if (brmsInstall){
				finalLayers = ValidatorConstants.brmsEds;
			} else {
				finalLayers = existingLayers;
			}
		}else if (existingLayers.equals(ValidatorConstants.brms)){ // BRMS and combinations
			// installing onto stock brms
			if (soaInstall){
				// installing soa onto brms installation
				finalLayers = ValidatorConstants.soaBrms;
			} else if (edsInstall){
				// installing eds onto brms installation
				finalLayers = ValidatorConstants.brmsEds;
			} else { // this clause should only be reached if we're installing brms onto itself. so just keep it the same. 
				finalLayers = existingLayers; 
			}
		} else if (existingLayers.equals(ValidatorConstants.brmsEds)){ // BRMS + EDS
			if (soaInstall){
				finalLayers = ValidatorConstants.soaBrmsEds;
			} else { // this clause should only be reached if we're installing brms or eds onto the installation, so keep it the same.
		 		finalLayers = existingLayers;
			}
		} else if (existingLayers.equals(ValidatorConstants.bpms)){ // BPMS
			if (soaInstall){
				finalLayers = ValidatorConstants.soaBpms;
			} else { // there should be no way to hit this else, other than bpms ontop of itself (not recommended) Validation in EapExistsvalidator
				finalLayers = existingLayers;
			}
		} else if (existingLayers.equals(ValidatorConstants.soa)){ // SOA derivatives
			if (edsInstall){
				finalLayers = ValidatorConstants.soaEds;
			} else if (brmsInstall){
				finalLayers = ValidatorConstants.soaBrms;
			} else if (bpmsInstall){
				finalLayers = ValidatorConstants.soaBpms;
			} else if (srampInstall){
				finalLayers = ValidatorConstants.soaSramp;
			} else {
				// soa onto soa 
				finalLayers = existingLayers;
			}
		} else if (existingLayers.equals(ValidatorConstants.soaEds)){ // SOA + EDS
			if (brmsInstall){
				finalLayers = ValidatorConstants.soaBrmsEds;
			} else if (srampInstall){
				finalLayers = ValidatorConstants.soaSrampEds;
			} else { // this situation needs to be impossible (validated by IsSupportedPlatformValidator)
				finalLayers = existingLayers;
			}
		} else if (existingLayers.equals(ValidatorConstants.soaBrms)){ // SOA + BRMS
			if (edsInstall){
				finalLayers = ValidatorConstants.soaBrmsEds;
			} else if (srampInstall){
				finalLayers = ValidatorConstants.soaSrampBrms;
			} else {
				finalLayers = existingLayers;
			}
		} else if (existingLayers.equals(ValidatorConstants.soaBrmsEds)){ // SOA + BRMS + EDS
			if (srampInstall){
				finalLayers = ValidatorConstants.soaSrampBrmsEds;
			} else {
				finalLayers = existingLayers;
			}
		} else if (existingLayers.equals(ValidatorConstants.soaSramp)){ // SOA + SRAMP
			if (edsInstall){
				finalLayers = ValidatorConstants.soaSrampEds;
			} else if (brmsInstall){
				finalLayers = ValidatorConstants.soaSrampBrms;
			} else if (bpmsInstall){
				finalLayers = ValidatorConstants.soaSrampBpms;
			} else {
				finalLayers = existingLayers;
			}
		} else if (existingLayers.equals(ValidatorConstants.soaSrampEds)){ // SOA + SRAMP + EDS
			if (brmsInstall){
				finalLayers = ValidatorConstants.soaSrampBrmsEds;
			} else {
				finalLayers = existingLayers;
			}
		} else if (existingLayers.equals(ValidatorConstants.soaSrampBrms)){ // SOA + SRAMP + BRMS
			if (edsInstall){
				finalLayers = ValidatorConstants.soaSrampBrmsEds;
			} else {
				finalLayers = existingLayers;
			}
		} else {
			finalLayers = existingLayers; // situations in which this would be wrong should be impossible
		}
		return finalLayers;
	}

	private static String getFinalProductString(String existingProduct, String newProduct) {
		if (newProduct.equals(ValidatorConstants.soa) || existingProduct.equals(ValidatorConstants.soa)){
			return ValidatorConstants.soa; 
		} else if (existingProduct.equals(ValidatorConstants.eap) && !newProduct.equals(existingProduct)){
			// new product takes priority in these situations
			return newProduct;
		} else if ((existingProduct.equals(ValidatorConstants.brms) || existingProduct.equals(ValidatorConstants.bpms))
                && newProduct.equals(ValidatorConstants.sramp)) { //sramp loses to everything except EAP
            return existingProduct;
        }
        else { // the only other possibility is that we're installing a product with newProduct == ValidatorConstants.eap, thus we can just return the
				 // existing product, since it's guaranteed to be correct
			return existingProduct; // eap in product.conf never beats anything, so it's safe to return the existing straight away
		}	
	}
}
