package com.redhat.installer.postinstall;

import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.HashMap;

import org.jboss.as.cli.CliInitializationException;

import com.izforge.izpack.event.SimpleInstallerListener;
import com.izforge.izpack.installer.AutomatedInstallData;
import com.izforge.izpack.util.AbstractUIProcessHandler;
import com.izforge.izpack.util.AbstractUIProgressHandler;
import com.izforge.izpack.util.CleanupClient;
import com.izforge.izpack.util.Debug;
import com.izforge.izpack.util.Housekeeper;
import com.redhat.installer.postinstall.ProcessPanelHelper.ScriptLauncher;
import com.redhat.installer.util.ServerCommandsHelper;

/**
 * ServerManager is responsible for starting and shutting down the JBoss
 * application server during installation process. It keeps track of server's
 * status (whether it's running or not, it's mode and configuration).
 *
 * It contains three entry points: The run method for calling a start/stop job
 * from process panel, and the startServer and stopServer methods used to start
 * and stop the server from other classes.
 *
 * @author fcanas
 *
 */
public class ServerManager extends SimpleInstallerListener implements
        CleanupClient {
	private static final String POSTINSTALL_SCRIPT = "postinstall-script";
	private static final String SERVER_SCRIPT = "server-script";
    private static final String SHUTDOWN_JOB = "shutdown";
    private static final String STANDALONE_CONFIG = "standalone-config";
    private static final String HOST_CONFIG = "host-config";
    private static final String STARTUP_JOB = "startup";
    private static final String FULL_HA = "full-ha"; //Remove when full-ha bug has been fixed
    private static final int DEFAULT_ATTEMPTS = 30;
    private static final int DEFAULT_HTTP = 8080;
    private static final int DEFAULT_MANAGEMENT = 9999;
    private static final int DEFAULT_MANAGEMENT_HTTP = 9990;
    private static final int DEFAULT_TIMEOUT = 3; 
    private static final String DEFAULT_URL = "http://localhost:";
    private static final String ONFAIL_JOB = "failmode";
    private static final String DEFAULT_PING_URL = "http://localhost:8080";
    private static final int [] DEFAULT_RESPONSE_CODES = new int [] { HttpURLConnection.HTTP_OK };
    private static AutomatedInstallData idata;
    private static ServerCommands serverCommands;
    private static String currentServerMode = "standalone";
    private static String currentStandaloneConfig = "";
    private static String currentHostConfig = "";
    private static boolean isFailMode = false;
    private static boolean isStarting = true;
    private static int [] responseCodes = null;
    private static String pingUrl = null;

    /**
     * Some static variables to keep track of server status.
     */
    private static volatile String STATUS = "down"; // down/up
    private static String CONFIG = ""; // ""/full/ha/full-ha
    private static String MODE = "standalone"; // standalone/domain
    private static volatile String CODE = ""; // the server code, see below:
    
	// Possible server codes we read from output:
    public static final String CODE_START_OK = "JBAS015874";
    public static final String CODE_START_ERROR = "JBAS015875";
    public static final String CODE_START_OK_FULL_HA = "JBAS015888";
    public static final String CODE_STOP_OK = "JBAS015950";
    public static final String CODE_PROCESS_FINISH = "JBAS012015";
    public static final String CODE_HOST_FINISH = "JBAS012010";

    /**
     * Map for strings
     */
    private static HashMap<String, String> portStringMap = new HashMap<String, String>();
    static {
        portStringMap.put("", "");
        portStringMap.put("ha", "h");
        portStringMap.put("full", "f");
        portStringMap.put("full-ha", "fa");
    }

    /**
     * Some public getters for the Server status.
     */
    public static String getStatus() {
        return STATUS;
    }

    public static String getMode() {
        return MODE;
    }

    public static String getConfig() {
        return CONFIG;
    }

    public static boolean isServerUp() {
        return STATUS.equalsIgnoreCase("up");
    }

    public static boolean isServerDown() {
        return STATUS.equalsIgnoreCase("down");
    }

    public static void setServerUp() {
        STATUS = "up";
    }

    public static void setServerCode(String code) {
        CODE = code;
    }

    public static String getServerCode() {
        return CODE;
    }

    public static void setServerDown() {
        STATUS = "down";
    }
    /**
     * Have we read either valid server startup code yet?
     *
     * @return
     */
    private static boolean IsServerCodeReady() {
        return CODE.contains(CODE_START_OK) || CODE.contains(CODE_START_ERROR)
                || CODE.contains(CODE_START_OK_FULL_HA);
    }

    public static String extractCode(String line) {
        String code = "";
        if (line.contains(CODE_START_OK)) {
            code = CODE_START_OK;
        } else if (line.contains(CODE_START_ERROR)) {
            code = CODE_START_ERROR;
        } else if (line.contains(CODE_START_OK_FULL_HA)
                && currentStandaloneConfig.equals(FULL_HA)) {
            code = CODE_START_OK_FULL_HA;
        }

        return code;
    }

    //Check when the server has stopped
    //May want to accept the statuses below also
    //CODE_PROCESS_FINISH
    //CODE_HOST_FINISH
    public static String extractEndCode(String line) {
        String code = "";
        if (line.contains(CODE_STOP_OK)) {
            code = CODE_STOP_OK;
        }

        return code;
    }
    /**
     * @return The http console interface port for current mode and
     *         configuration.
     */
    public static int getHttpPort() {
        return getPort(DEFAULT_HTTP, "http");
    }

    public static String getURL() {
        return DEFAULT_URL;
    }

    /**
     * @return The native cli management interface port for current mode and
     *         configuration.
     */
    public static int getManagementPort() {
        return getPort(DEFAULT_MANAGEMENT, "management-native");
    }

    public static int getManagementHttpPort() {
        return getPort(DEFAULT_MANAGEMENT_HTTP, "management-http");
    }

    /**
     * The process panel run handler. This is the entry point when we call
     * this class from process panel jobs.
     * @param handler
     * @param args
     * @throws Exception
     */
    public static void run(AbstractUIProcessHandler handler, String[] args)
            throws Exception {
        /**
         * Set up argument parser.
         */
        ArgumentParser parser = new ArgumentParser();
        parser.parse(args);

        /**
         * Get the install data.
         */
        idata = AutomatedInstallData.getInstance();

        /**
         * Check to see if this job is being called in failmode.
         */
        isFailMode = parser.propertyIsTrue(ONFAIL_JOB);

        /**
         * Check if we are starting or shutting down server.
         */
        isStarting = parser.hasProperty(SERVER_SCRIPT)
                || parser.hasProperty(POSTINSTALL_SCRIPT);

        if (parser.hasProperty(HOST_CONFIG)) {
            handler.logOutput("[ host-config detected! ]", false);
        }
        if (isStarting) {
            /**
             * Start the server.
             */
            startServer(handler, parser);

        } else {
            /**
             * Shut the server down here.
             */
            shutdownServer(handler, parser);
        }

	}

    /**
     * This method can be called to start the server given the specifics:
     * @param location The path to the server startup script.
     * @param mode	"standalone"|"domain"
     * @param config ""|"ha"|"full"|"full-ha"
     * @param handler Can be null, otherwise the current thread handler.
     */
    public static void startServer(String location, String mode, String config,
            String hostconfig,
            boolean adminOnly, int timeout, int attempts,
            AbstractUIProcessHandler handler) {
        String startingServerString = idata.langpack
                .getString("postinstall.processpanel.startingServer");

        String serverStartedString = idata.langpack
                .getString("postinstall.processpanel.serverStarted");

        String ServerStartedWithErrors = idata.langpack
        		.getString("postinstall.processpanel.serverStartedWithErrors");

        ScriptLauncher serverScript;
        int port = 8080; // default

        /**
         * Generate a ScriptLauncher with the proper params.
         */
        if (adminOnly) {
            if (mode.contains("standalone")) {

                /**
                 * Append the hyphen to the config if it isn't blank.
                 */
                String configstring = (config.equals("") ? config : "-"
                        + config);
                /**
                 * Standalone admin-only mode with specific config.
                 */
                serverScript = new ScriptLauncher(handler, location, "--admin-only",
                        "--server-config=standalone" + configstring + ".xml");
            } else {
                /**
                 * Domain admin-only mode.
                 */
                serverScript = new ScriptLauncher(handler, location,
                        "--admin-only", "--host-config=" + hostconfig + ".xml");
            }
            /**
             * Admin-mode only so we require the management port.
             */
            port = getManagementPort(mode, config);
        } else {
            /**
             * Full server start mode for either standalone or domain.
             */
            port = getHttpPort(mode, config);
            if (mode.contains("standalone")) {
                serverScript = new ScriptLauncher(handler, location);
            } else {
                serverScript = new ScriptLauncher(handler, location,
                        "--host-config=" + hostconfig + ".xml");
            }
        }

        if (handler != null) handler.logOutput(startingServerString, false);

        serverScript.runScript(); // Launching server here.

        /**
         * Forge the URL out of FIRE AND STEEL!!!
         */
        String url = DEFAULT_URL + port;

        /**
         * Ensure server has successfully started and update status, but only
         * ping if it's not in admin-only mode.
         */
        if (waitForServerCodeReady(url, timeout, attempts, handler)) {
            updateStatus("up", mode, config, CODE);
            if (handler != null && CODE.equals(CODE_START_ERROR))
            	handler.logOutput(ServerStartedWithErrors, false);
            else if (handler != null)
                handler.logOutput(serverStartedString, false);
        } else {
            /**
             * We weren't succesful, so don't change status except for marking
             * server as 'down'. TODO: Consider a better approach to handling
             * server startup failures.
             */
            // updateStatus("down", MODE, CONFIG);
        }
    }

    private static boolean waitForServerCodeReady(String url, int timeout,
            int attemptLimit,
            AbstractUIProcessHandler handler) {
        String checkingServerString = idata.langpack
                .getString("postinstall.processpanel.checkingServer");
        String serverNotStartedString = idata.langpack
                .getString("postinstall.processpanel.serverNotStarted");
        String serverStartedString = idata.langpack
                .getString("postinstall.processpanel.serverStarted");

        int currentAttempt = 0;
        int totalTime = timeout * attemptLimit;
        String timeUnit = "seconds";

        if (totalTime > 60){
        	totalTime = totalTime / 60;
        	timeUnit = "minutes";
        }


        if (handler != null)
            ProcessPanelHelper.printToPanel(handler, String.format(checkingServerString, totalTime, timeUnit), false);

        while (!IsServerCodeReady()) {

        	if (handler != null) ProcessPanelHelper.printToPanel(handler, ".", false);

            try {
                Thread.sleep(timeout * 1000);
            } catch (InterruptedException e) {
                // What to do here?!
            }

            currentAttempt++;

            if (currentAttempt > attemptLimit) {
                /**
                 * If we haven't seen the ready code by the time we time out,
                 * poll the http interface one last time to see if should
                 * proceed or fail.
                 */
                if (isLinkAlive(url)) {
                    return true;
                } else {
                    if (handler != null)
                        ProcessPanelHelper.printToPanel(handler, String.format(serverNotStartedString,
                                attemptLimit + 1), true);

                    // Set status to unknown, since the ping verifies the server
                    // didn't finish going up successfully.
                    updateStatus("unknown", MODE, CONFIG, "");
                    return false;
                }
            }

        }
        /**
         * If we break out of loop, it's because server is code ready.
         */
        return true;
    }

    /**
     * Shuts down the currently running server, if any.
     */
    public static void shutDownServer() {
    	try {
			shutdownServer(null, DEFAULT_TIMEOUT, DEFAULT_ATTEMPTS);
		} catch (Exception e) {
			// TODO Auto-generated catch block
		}
    }

    /**
     * Private methods and helpers below.
     */

    /**
     * @return The http console interface port for specified mode and config.
     */
    private static int getHttpPort(String mode, String config) {
        return getPort(mode, config, DEFAULT_HTTP, "http");
    }

    /**
     * @return The native cli management interface port for the specified mode
     *         and configuration.
     */
    private static int getManagementPort(String mode, String config) {

        int port = getPort(mode, config, DEFAULT_MANAGEMENT,
                "management-native");

        /**
         * Domain's management mode port is not affected by the offset.
         */
        return port;
    }

    private static int getPort(int defaultPort, String portName) {
        String serverMode = getMode();
        String standaloneConfig = getConfig();
        return getPort(serverMode, standaloneConfig, defaultPort, portName);
    }

    /**
     * Finds and returns the correct port number based on user's customized port
     * binding configurations, or default port if none is found.
     *
     * @param defaultPort
     * @param portName
     *            The name of the port we need as used in the userInput specs.
     * @return the specified port.
     */
    private static int getPort(String serverMode, String standaloneConfig,
            int defaultPort, String portName) {
        int port = defaultPort;

        String prefix = getPortPrefix(serverMode, standaloneConfig);

        String var = prefix + "." + portName;
        String val = idata.getVariable(var);

        if (val != null) {
            String portString = idata.getVariable(prefix + "." + portName);

            /**
             * Some port variables are in the form of other variables. Yikes!
             */
            if (portString.contains("$")) {

                port = Integer.parseInt(portString.substring(
                        portString.indexOf(':') + 1, portString.length() - 1));

            } else {
                port = Integer.parseInt(portString);
            }

        } else {
            /**
             * Others are simple numbers.
             */
            port = defaultPort;
        }

        /**
         * It says domain here, but this is actually the offset used by the
         * standalone. Not sure how this naming scheme came to be!
         */
        int offset = idata.getVariable("domain.port-offset") != null ? Integer
                .parseInt(idata.getVariable("domain.port-offset")) : 0;

        /**
         * Ignore the offset if we're getting the management-native or
         * management-http port for domain server.
         */
        if (serverMode.contains("domain")
                && (portName.contains("management-native") || portName
                        .contains("management-http"))) {
        	offset = 0;
        }

        return port + offset;

    }

    /**
     * Given the server mode and config, returns the prefix needed to find the
     * server's custom port variables.
     *
     * @param serverMode
     * @param standaloneConfig
     * @return
     */
    private static String getPortPrefix(String serverMode,
            String standaloneConfig) {
        String prefix;

        if (standaloneConfig.equals("")) {
            prefix = serverMode;
        } else {
            prefix = serverMode.contains("standalone") ? serverMode
                + "." + portStringMap.get(standaloneConfig) : serverMode;
        }

        return prefix;
    }

    /**
     * Tests an HTTP connection and returns true if the other side
     * responds with an HTTP_OK.
     * @param URLName
     * @return
     */
    private static boolean isLinkAlive(String URLName) {
    	String url = (pingUrl==null) ? URLName : URLName + pingUrl;
    	int [] ports = (responseCodes == null) ? DEFAULT_RESPONSE_CODES : responseCodes;
        return isLinkAlive(url, ports);

    }

    /**
     *
     * @param URLName
     * @param goodcodes
     *            A list of integers. Each int is an expected HTTP Response
     *            code.
     * @return True if the http interface returned one of the goodcodes.
     *         False otherwise.
     */
    public static boolean isLinkAlive(String URLName, int[] goodcodes) {
        HttpURLConnection urlConn = null;

        try {
            URL url = new URL(URLName);
            urlConn = (HttpURLConnection) url.openConnection();
            urlConn.connect();

            // For debugging:
            /*
            System.out.println("Server Response Code: "
                    + urlConn.getResponseCode());*/

            /**
             * Assume bad response until proven otherwise.
             */
            boolean response = false;

            /**
             * Check the actual response code against the list of desired
             * HTTP response codes.
             */
            for (int code : goodcodes) {
                if (urlConn.getResponseCode() == code) return true;
            }

            return false;
        } catch (IOException e) {
            // System.err.println("Error creating HTTP connection");
            // e.printStackTrace();
            return false;
        }
    }


    private static boolean pingServer(String url,
            AbstractUIProcessHandler handler) {
        return pingServer(url, DEFAULT_TIMEOUT, DEFAULT_ATTEMPTS, handler);
    }

    /**
     * Queries the given URL and returns true if the other side responds
     * after a given number of tries with time outs.
     * @param url
     * @param attemptLimit
     * @param timeout
     * @param handler
     * @return
     */
    private static boolean pingServer(String url, int timeout, int attemptLimit,
            AbstractUIProcessHandler handler) {
        String checkingServerString = idata.langpack
                .getString("postinstall.processpanel.checkingServer");
        String serverNotStartedString = idata.langpack
                .getString("postinstall.processpanel.serverNotStarted");
        String serverStartedString = idata.langpack
                .getString("postinstall.processpanel.serverStarted");
        int currentAttempt = 0;
        
        //System.out.println(serverStatusString());

        /**
         * Here we send the ping to the server. If it doesn't respond, go to
         * sleep and try again. Repeat until we receive a response or we run out
         * of attempts.
         */
        while (!isLinkAlive(url) && !IsServerCodeReady()) {
            if (handler != null)
                ProcessPanelHelper.printToPanel(handler, String.format(checkingServerString,
                    currentAttempt + 1,
                    attemptLimit + 1), false);
            try {
                Thread.sleep(timeout * 1000);
            } catch (InterruptedException e) {
                // What to do here?!
            }

            currentAttempt++;

            if (currentAttempt > attemptLimit) {
                // Timed out without getting server response.
                if (handler != null)
                    ProcessPanelHelper.printToPanel(handler,
                        String.format(serverNotStartedString, attemptLimit + 1),
                        true);
                return false;
            }

        }

        /**
         * Server responded, so return true.
         */
        return true;
    }

    private static String serverStatusString() {
        String status = "Status: " + STATUS + "\n" + "Mode: " + MODE + "\n"
                + "Config: " + CONFIG;

        return status;
    }

    /**
     *
     * @param handler
     * @param parser
     * @throws Exception
     */
    private static void shutdownServer(AbstractUIProcessHandler handler,
            ArgumentParser parser)
            throws Exception {
        /**
         * This variable gets set to true whenever a server is successfully
         * launched.
         */
        // String serverIsUp = idata.getVariable(ProcessPanelHelper.SERVER_UP);

        if (isFailMode)
            ProcessPanelHelper.printToPanel(handler, idata.langpack.getString("ServerManager.onfail.check"), false);

        /**
         * The server is already explicitly down, no need to check.
         */
        if (isServerDown()) {
            ProcessPanelHelper.printToPanel(handler, idata.langpack.getString("ServerManager.nostart"), false);

            return;
        }

        if (isFailMode)
            ProcessPanelHelper.printToPanel(handler, idata.langpack.getString("ServerManager.shutdown"), false);


        shutdownServer(handler,
                DEFAULT_TIMEOUT,
                DEFAULT_ATTEMPTS);

    }

    /**
     *
     * @param handler
     * @param timeout
     * @param attempts
     * @throws Exception
     */
    private static void shutdownServer(AbstractUIProcessHandler handler,
            int timeout, int attempts) throws Exception {
        String servermode = getMode();
        String username = idata.getVariable("postinstall.username");
        String serverShutdownString = idata.langpack
                .getString("postinstall.processpanel.serverStopped");
        char[] password = idata.getVariable("postinstall.password")
                .toCharArray();

        serverCommands = null;

        int port = getManagementPort();

        try {
            serverCommands = ServerCommands.createSession(username, password,
                    port);

        } catch (CliInitializationException e) {
            /**
             * Catching an exception here probably means the server was already
             * down, or at least in a non-receptive state.
             */
            if (handler != null)
                ProcessPanelHelper.printToPanel(handler, idata.langpack.getString("ServerManager.alreadyoff"), false);


        }

        ServerCommandsHelper.connectContext(handler, serverCommands);

        if (servermode.contains("domain")) {
        	/**
            System.out
                    .println("Attempting to shut down domain at port " + port);**/
            serverCommands.shutdownDomainHost();
        } else if (servermode.contains("standalone")) {
        	/**
            System.out.println("Attempting to shut down standalone at port "
                    + port);**/
            serverCommands.shutdownHost();
        } else {
            // No previous server info found.
        	// TODO: How to handle this strange occurrence?
        }
        /**
         * TODO: Possibly add a test to check that the server did indeed shut down.
         */

        long startTime = System.currentTimeMillis();
        while(!isServerDown()){
            // TODO: Add time-out conditions. *Must* guarantee termination.
            if ((System.currentTimeMillis() - startTime) > 10000) break;
            //Wait until server is down before proceeding
        }
        if (handler != null)
            ProcessPanelHelper.printToPanel(handler, serverShutdownString, false);
        updateStatus("down", MODE, CONFIG, CODE_STOP_OK);
    }

    /**
     * Responsible for starting the server.
     *
     * @param handler
     * @param parser
     */
    private static void startServer(AbstractUIProcessHandler handler,
            ArgumentParser parser) {
        String serverLocation;
        boolean adminMode;

        /**
         * Get the server script location and determine if it's admin-only mode
         * or not.
         */
        if (parser.hasProperty(SERVER_SCRIPT)) {
            serverLocation = parser.getProperty(SERVER_SCRIPT);
            adminMode = false;
        } else if (parser.hasProperty(POSTINSTALL_SCRIPT)) {
            serverLocation = parser.getProperty(POSTINSTALL_SCRIPT);
            adminMode = true;
        } else {
            ProcessPanelHelper.printToPanel(handler, idata.langpack.getString("ServerManager.invalidargs"), false);
            return;
        }

        if (serverLocation == null) {
            /**
             * Panic, because we don't have a server startup script.
             */
            ProcessPanelHelper.printToPanel(handler, idata.langpack.getString("ServerManager.noscript"), false);

            return;
        }

        /**
         * Check if this is domain or standalone.
         */
        if (serverLocation.contains("domain")) {
            currentServerMode = "domain";
        } else if (serverLocation.contains("standalone")) {
            currentServerMode = "standalone";
        } else {
            /**
             * No correct startup script given. This is a cause for failure.
             */
            ProcessPanelHelper.printToPanel(handler, idata.langpack.getString("ServerManager.wrongarg"), false);
            return;
        }

        /**
         * Get the configuration if standalone.
         */
        if (currentServerMode.contains("standalone")) {
            currentStandaloneConfig = parser.getProperty(STANDALONE_CONFIG);
            if (currentStandaloneConfig == null)
                currentStandaloneConfig = "";
        }

        /**
         * Get the configuration if domain.
         */
        if (currentServerMode.contains("domain")) {
            if (parser.hasProperty(HOST_CONFIG)) {
                currentHostConfig = parser.getProperty(HOST_CONFIG);
                if (handler != null)
                    ProcessPanelHelper.printToPanel(handler,
                            "[ host-config detected: " + currentHostConfig
                                    + "]", false);
            } else {
                currentHostConfig = "host"; // Default configuration for domain
                                            // mode.
            }
        }

        /**
         * Check for explicit ping interfaces and response codes.
         */
        if (parser.hasProperty("url-ext")) {
        	pingUrl = parser.getProperty("url-ext");
        }

        if(parser.hasProperty("url-resp")) {
        	responseCodes = new int [] { HttpURLConnection.HTTP_OK,
        				Integer.parseInt(parser.getProperty("url-resp"))
        			};
        }

        /**
         * Get timeout and attempts.
         */
        int timeout = (parser.hasProperty("timeout")) ? Integer.parseInt(parser.getProperty("timeout")) : DEFAULT_TIMEOUT;
        int attempts = (parser.hasProperty("attempts")) ? Integer.parseInt(parser.getProperty("attempts")) : DEFAULT_TIMEOUT;

        /**
         * Start the server with the parameters gathered from the job spec.
         */
        startServer(serverLocation, currentServerMode, currentStandaloneConfig,
                currentHostConfig,
                adminMode, timeout, attempts,
                handler);

        /**
         * Restore to defaults.
         */
        pingUrl = null;
        responseCodes = null;
    }

    /**
     * Records the new status for the server.
     * @param status (up or down)
     * @param mode	(standalone or domain)
     * @param config ("", ha, full, full-ha)
     */
    private static void updateStatus(String status, String mode, String config,
            String code) {
        STATUS = status;
        MODE = mode;
        CONFIG = config;
        CODE = code;
        //System.out.println("Recording Server Status:");
        //System.out.println(serverStatusString());
    }

    /**
     * Run this to register to the Housekeeper. We do this after all
     * installation of packs.
     *
     * @param idata
     * @param npacks
     * @param handler
     * @throws Exception
     */
    @Override
    public void beforePacks(AutomatedInstallData idata, Integer x,
            AbstractUIProgressHandler handler) throws Exception {
        Debug.log("Registering ServerManager for Cleanup.");
        Housekeeper.getInstance().registerForCleanup(this);
    }

    /**
     * This method will be run at shutdown when the user prematurely quits an
     * installation.
     */
    public void cleanUp() {
        idata = AutomatedInstallData.getInstance();

        String aborted = idata.getVariable("install.aborted");

        Debug.log("Housekeeper: Running Server Manager Clean Up.");

        if ((aborted == null && !Boolean.parseBoolean(aborted))
                || (!isServerUp())) {
            // clean exit, so remove the post install user and call it a day.
            // System.out.println("Server is not up. CleanUp aborted.");
            Debug.log("Housekeeper: Server Not Up.");
            PostInstallUserHelper.removePostInstallUser();
            return;
        }

        try {
            Debug.log("Housekeeper: Server Up. Shutting Down.");
            shutdownServer(null, DEFAULT_TIMEOUT,
                    DEFAULT_ATTEMPTS);

        } catch (Exception e) {
            // This likely threw an exception because there is no server to shut
            // down or it already went down.

        } finally {
            // Actually, nothing to see here.
        }
    }
}
