"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SegmentAdapter = exports.SegmentProperties = void 0;
const tslib_1 = require("tslib");
/**
 * Copyright (c) 2019-2021 Red Hat, Inc.
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *   Red Hat, Inc. - initial API and implementation
 */
const cli_ux_1 = require("cli-ux");
const countries_and_timezones_1 = require("countries-and-timezones");
const fs = require("fs-extra");
const lodash_1 = require("lodash");
const os = require("os");
const osLocale = require("os-locale");
const path = require("path");
const uuid_1 = require("uuid");
const util_1 = require("../../util");
const Analytics = require('analytics-node');
var SegmentProperties;
(function (SegmentProperties) {
    SegmentProperties.Telemetry = 'segment.telemetry';
})(SegmentProperties = exports.SegmentProperties || (exports.SegmentProperties = {}));
/**
 * Class with help methods which help to connect segment and send telemetry data.
 */
class SegmentAdapter {
    constructor(segmentConfig, segmentId) {
        const { segmentWriteKey } = segmentConfig, options = tslib_1.__rest(segmentConfig, ["segmentWriteKey"]);
        this.segment = new Analytics(segmentWriteKey, options);
        this.id = segmentId;
    }
    /**
     * Returns anonymous id to identify and track crwctl events in segment
     * Check if exists an anonymousId in file: $HOME/.redhat/anonymousId and if not generate new one in this location
     */
    static getAnonymousId() {
        const anonymousIdPath = path.join(os.homedir(), '.redhat', 'anonymousId');
        let anonymousId = uuid_1.v4();
        try {
            if (fs.existsSync(anonymousIdPath)) {
                anonymousId = fs.readFileSync(anonymousIdPath, 'utf8');
            }
            else {
                if (!fs.existsSync(anonymousIdPath)) {
                    fs.mkdirSync(path.join(os.homedir(), '.redhat'));
                }
                fs.writeFileSync(anonymousIdPath, anonymousId, { encoding: 'utf8' });
            }
        }
        catch (error) {
            cli_ux_1.cli.debug(`Failed to store anonymousId ${error}`);
        }
        return anonymousId.trim();
    }
    /**
     * Identify anonymous user in segment before start to track
     * @param anonymousId Unique identifier
     */
    identifySegmentEvent(anonymousId) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            this.segment.identify({
                anonymousId,
                traits: yield this.getSegmentIdentifyTraits(),
            });
        });
    }
    /**
     * Create a segment track object which includes command properties and some crwctl filtred properties
     * @param options crwctl information like command or flags.
     * @param segmentID crwctl ID generated only if telemetry it is 'on'
     */
    trackSegmentEvent(options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            this.segment.track({
                anonymousId: this.id,
                event: options.command.replace(':', ' '),
                context: yield this.getSegmentEventContext(),
                properties: Object.assign(Object.assign({}, lodash_1.pick(options.flags, ['platform', 'installer'])), { command: options.command, version: util_1.getProjectVersion() }),
                // Property which indicate segment will integrate with all configured destinations.
                integrations: {
                    All: true,
                },
            });
        });
    }
    // Returns basic info about identification in segment
    getSegmentIdentifyTraits() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return {
                timezone: Intl.DateTimeFormat().resolvedOptions().timeZone,
                os_name: os.platform(),
                os_version: os.release(),
                os_distribution: yield util_1.getDistribution(),
                locale: osLocale.sync().replace('_', '-'),
            };
        });
    }
    /**
     * Returns segment event context. Include platform info or countries from where the app was executed
     * More info: https://segment.com/docs/connections/spec/common/#context
     */
    getSegmentEventContext() {
        var _a;
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return {
                ip: '0.0.0.0',
                locale: osLocale.sync().replace('_', '-'),
                app: {
                    name: util_1.getProjectName(),
                },
                os: {
                    name: os.platform(),
                    version: os.release(),
                },
                location: {
                    country: ((_a = countries_and_timezones_1.getTimezone(Intl.DateTimeFormat().resolvedOptions().timeZone)) === null || _a === void 0 ? void 0 : _a.country) || 'XX',
                },
                timezone: Intl.DateTimeFormat().resolvedOptions().timeZone,
            };
        });
    }
}
exports.SegmentAdapter = SegmentAdapter;
//# sourceMappingURL=segment-adapter.js.map