"use strict";
/*********************************************************************
 * Copyright (c) 2019 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 **********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const command_1 = require("@oclif/command");
const cli_ux_1 = require("cli-ux");
const Listrq = require("listr");
const notifier = require("node-notifier");
const che_1 = require("../../api/che");
const kube_1 = require("../../api/kube");
const common_flags_1 = require("../../common-flags");
const che_2 = require("../../tasks/che");
const api_1 = require("../../tasks/platforms/api");
class Delete extends command_1.Command {
    run() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { flags } = this.parse(Delete);
            const { args } = this.parse(Delete);
            const ctx = {};
            ctx.workspaces = [];
            const apiTasks = new api_1.ApiTasks();
            const cheTasks = new che_2.CheTasks(flags);
            const cheHelper = new che_1.CheHelper(flags);
            const kubeHelper = new kube_1.KubeHelper(flags);
            const tasks = new Listrq(undefined, { renderer: 'silent' });
            tasks.add(apiTasks.testApiTasks(flags, this));
            tasks.add(cheTasks.verifyCheNamespaceExistsTask(flags, this));
            tasks.add(cheTasks.retrieveEclipseCheUrl(flags));
            tasks.add(cheTasks.checkEclipseCheStatus());
            tasks.add({
                title: `Get workspace by id '${args.workspace}'`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const workspace = yield cheHelper.getWorkspace(ctx.cheURL, args.workspace, flags['access-token']);
                    ctx.infrastructureNamespace = workspace.attributes.infrastructureNamespace;
                    task.title = `${task.title}... done`;
                })
            });
            tasks.add({
                title: `Delete workspace by id '${args.workspace}'`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield cheHelper.deleteWorkspace(ctx.cheURL, args.workspace, flags['access-token']);
                    cli_ux_1.cli.log(`Workspace with id '${args.workspace}' deleted.`);
                    task.title = `${task.title}... done`;
                })
            });
            tasks.add({
                title: 'Verify if namespace exists',
                enabled: () => flags['delete-namespace'],
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    task.title = `${task.title} '${ctx.infrastructureNamespace}'`;
                    if (ctx.infrastructureNamespace === flags.chenamespace) {
                        cli_ux_1.cli.warn(`It is not possible to delete namespace '${ctx.infrastructureNamespace}' since it is used for CodeReady Workspaces deployment.`);
                        return;
                    }
                    ctx.infrastructureNamespaceExists = yield kubeHelper.namespaceExist(ctx.infrastructureNamespace);
                    if (ctx.infrastructureNamespaceExists) {
                        task.title = `${task.title}... found`;
                    }
                    else {
                        task.title = `${task.title}... not found`;
                    }
                })
            });
            tasks.add({
                title: 'Delete namespace',
                skip: ctx => !ctx.infrastructureNamespaceExists,
                enabled: () => flags['delete-namespace'],
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    task.title = `${task.title} '${ctx.infrastructureNamespace}'`;
                    yield kubeHelper.deleteNamespace(ctx.infrastructureNamespace);
                    cli_ux_1.cli.log(`Namespace '${ctx.infrastructureNamespace}' deleted.`);
                    task.title = `${task.title}... done`;
                })
            });
            try {
                yield tasks.run(ctx);
            }
            catch (error) {
                this.error(error);
            }
            notifier.notify({
                title: 'crwctl',
                message: 'Command workspace:delete has completed successfully.'
            });
            this.exit(0);
        });
    }
}
exports.default = Delete;
Delete.description = 'delete a user\'s workspace';
Delete.flags = {
    help: command_1.flags.help({ char: 'h' }),
    chenamespace: common_flags_1.cheNamespace,
    'delete-namespace': command_1.flags.boolean({
        description: 'Indicates that a Kubernetes namespace where workspace was created will be deleted as well',
        default: false
    }),
    'access-token': common_flags_1.accessToken,
    'skip-kubernetes-health-check': common_flags_1.skipKubeHealthzCheck
};
Delete.args = [
    {
        name: 'workspace',
        description: 'The workspace id to delete',
        required: true
    }
];
//# sourceMappingURL=delete.js.map