"use strict";
/*********************************************************************
 * Copyright (c) 2019-2020 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 **********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
exports.CheApiClient = void 0;
const tslib_1 = require("tslib");
const axios_1 = require("axios");
const cli_ux_1 = require("cli-ux");
const https = require("https");
const util_1 = require("../util");
/**
 * Singleton responsible for calls to Che API.
 */
let instance;
class CheApiClient {
    constructor(cheApiEndpoint) {
        this.defaultCheResponseTimeoutMs = 3000;
        this.cheApiEndpoint = cheApiEndpoint;
        // Make axios ignore untrusted certificate error for self-signed certificate case.
        const httpsAgent = new https.Agent({ rejectUnauthorized: false });
        this.axios = axios_1.default.create({
            httpsAgent
        });
    }
    static getInstance(cheApiEndpoint) {
        cheApiEndpoint = this.normalizeCheApiEndpointUrl(cheApiEndpoint);
        if (!instance || instance.cheApiEndpoint !== cheApiEndpoint) {
            instance = new CheApiClient(cheApiEndpoint);
        }
        return instance;
    }
    static normalizeCheApiEndpointUrl(url) {
        if (url) {
            if (!url.includes('://')) {
                url = 'https://' + url;
            }
            const u = new URL(url);
            url = 'https://' + u.host + u.pathname;
            if (url.endsWith('/')) {
                url = url.slice(0, -1);
            }
            return url;
        }
    }
    /**
     * Checks whether provided url really points to Che server API.
     * Throws an exception if it's not.
     */
    checkCheApiEndpointUrl(responseTimeoutMs = this.defaultCheResponseTimeoutMs) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const response = yield this.axios.get(`${this.cheApiEndpoint}/system/state`, { timeout: responseTimeoutMs });
                if (response.data && response.data.status) {
                    return;
                }
            }
            catch (_a) {
                throw new Error(`E_CHE_API_URL_NO_RESPONSE - Failed to connect to "${this.cheApiEndpoint}". Is it the right url?`);
            }
            throw new Error(`E_CHE_API_WRONG_URL - Provided url "${this.cheApiEndpoint}" is not Che API url`);
        });
    }
    isCheServerReady(responseTimeoutMs = this.defaultCheResponseTimeoutMs) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const id = this.axios.interceptors.response.use(response => response, (error) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                if (error.config && error.response && (error.response.status === 404 || error.response.status === 503)) {
                    yield util_1.sleep(500);
                    return this.axios.request(error.config);
                }
                return Promise.reject(error);
            }));
            try {
                yield this.axios.get(`${this.cheApiEndpoint}/system/state`, { timeout: responseTimeoutMs });
                return true;
            }
            catch (_a) {
                return false;
            }
            finally {
                this.axios.interceptors.response.eject(id);
            }
        });
    }
    getCheServerStatus(responseTimeoutMs = this.defaultCheResponseTimeoutMs) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const endpoint = `${this.cheApiEndpoint}/system/state`;
            let response = null;
            try {
                response = yield this.axios.get(endpoint, { timeout: responseTimeoutMs });
            }
            catch (error) {
                throw this.getCheApiError(error, endpoint);
            }
            if (!response || response.status !== 200 || !response.data || !response.data.status) {
                throw new Error('E_BAD_RESP_CHE_API');
            }
            return response.data.status;
        });
    }
    startCheServerShutdown(accessToken = '', responseTimeoutMs = this.defaultCheResponseTimeoutMs) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const endpoint = `${this.cheApiEndpoint}/system/stop?shutdown=true`;
            const headers = accessToken ? { Authorization: accessToken } : null;
            let response = null;
            try {
                response = yield this.axios.post(endpoint, null, { headers, timeout: responseTimeoutMs });
            }
            catch (error) {
                if (error.response && error.response.status === 409) {
                    return;
                }
                else {
                    throw this.getCheApiError(error, endpoint);
                }
            }
            if (!response || response.status !== 204) {
                throw new Error('E_BAD_RESP_CHE_API');
            }
        });
    }
    waitUntilCheServerReadyToShutdown(intervalMs = 500, timeoutMs = 60000) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const iterations = timeoutMs / intervalMs;
            for (let index = 0; index < iterations; index++) {
                let status = yield this.getCheServerStatus();
                if (status === 'READY_TO_SHUTDOWN') {
                    return;
                }
                yield cli_ux_1.cli.wait(intervalMs);
            }
            throw new Error('ERR_TIMEOUT');
        });
    }
    /**
     * Returns list of all workspaces of the user.
     */
    getAllWorkspaces(accessToken) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const all = [];
            const itemsPerPage = 30;
            let skipCount = 0;
            let workspaces;
            do {
                workspaces = yield this.getWorkspaces(skipCount, itemsPerPage, accessToken);
                all.push(...workspaces);
                skipCount += workspaces.length;
            } while (workspaces.length === itemsPerPage);
            return all;
        });
    }
    /**
     * Returns list of workspaces in given range.
     * If lst of all workspaces is needed, getAllWorkspaces should be used insted.
     */
    getWorkspaces(skipCount = 0, maxItems = 30, accessToken) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const endpoint = `${this.cheApiEndpoint}/workspace?skipCount=${skipCount}&maxItems=${maxItems}`;
            const headers = { 'Content-Type': 'text/yaml' };
            if (accessToken && accessToken.length > 0) {
                headers.Authorization = accessToken;
            }
            try {
                const response = yield this.axios.get(endpoint, { headers });
                if (response && response.data) {
                    return response.data;
                }
                else {
                    throw new Error('E_BAD_RESP_CHE_SERVER');
                }
            }
            catch (error) {
                throw this.getCheApiError(error, endpoint);
            }
        });
    }
    getWorkspaceById(workspaceId, accessToken) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const endpoint = `${this.cheApiEndpoint}/workspace/${workspaceId}`;
            const headers = { 'Content-Type': 'text/yaml' };
            if (accessToken) {
                headers.Authorization = accessToken;
            }
            try {
                const response = yield this.axios.get(endpoint, { headers });
                return response.data;
            }
            catch (error) {
                if (error.response.status === 404) {
                    throw new Error(`Workspace ${workspaceId} not found. Please use the command workspace:list to get list of the existed workspaces.`);
                }
                throw this.getCheApiError(error, endpoint);
            }
        });
    }
    deleteWorkspaceById(workspaceId, accessToken) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const endpoint = `${this.cheApiEndpoint}/workspace/${workspaceId}`;
            const headers = {};
            if (accessToken) {
                headers.Authorization = accessToken;
            }
            try {
                yield this.axios.delete(endpoint, { headers });
            }
            catch (error) {
                if (error.response.status === 404) {
                    throw new Error(`Workspace ${workspaceId} not found. Please use the command workspace:list to get list of the existed workspaces.`);
                }
                else if (error.response.status === 409) {
                    throw new Error('Cannot delete a running workspace. Please stop it using the command workspace:stop and try again');
                }
                throw this.getCheApiError(error, endpoint);
            }
        });
    }
    startWorkspace(workspaceId, debug, accessToken) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            let endpoint = `${this.cheApiEndpoint}/workspace/${workspaceId}/runtime`;
            if (debug) {
                endpoint += '?debug-workspace-start=true';
            }
            let response;
            const headers = {};
            if (accessToken) {
                headers.Authorization = accessToken;
            }
            try {
                response = yield this.axios.post(endpoint, undefined, { headers });
            }
            catch (error) {
                if (error.response && error.response.status === 404) {
                    throw new Error(`E_WORKSPACE_NOT_EXIST - workspace with "${workspaceId}" id doesn't exist`);
                }
                else {
                    throw this.getCheApiError(error, endpoint);
                }
            }
            if (!response || response.status !== 200 || !response.data) {
                throw new Error('E_BAD_RESP_CHE_API');
            }
        });
    }
    stopWorkspace(workspaceId, accessToken) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const endpoint = `${this.cheApiEndpoint}/workspace/${workspaceId}/runtime`;
            let response;
            const headers = {};
            if (accessToken) {
                headers.Authorization = accessToken;
            }
            try {
                response = yield this.axios.delete(endpoint, { headers });
            }
            catch (error) {
                if (error.response && error.response.status === 404) {
                    throw new Error(`E_WORKSPACE_NOT_EXIST - workspace with "${workspaceId}" id doesn't exist`);
                }
                else {
                    throw this.getCheApiError(error, endpoint);
                }
            }
            if (!response || response.status !== 204) {
                throw new Error('E_BAD_RESP_CHE_API');
            }
        });
    }
    createWorkspaceFromDevfile(devfileContent, accessToken) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const endpoint = `${this.cheApiEndpoint}/workspace/devfile`;
            const headers = { 'Content-Type': 'text/yaml' };
            if (accessToken) {
                headers.Authorization = accessToken;
            }
            let response;
            try {
                response = yield this.axios.post(endpoint, devfileContent, { headers });
            }
            catch (error) {
                if (error.response) {
                    if (error.response.status === 400) {
                        throw new Error(`E_BAD_DEVFILE_FORMAT - Message: ${error.response.data.message}`);
                    }
                    if (error.response.status === 409) {
                        let message = '';
                        if (error.response.data) {
                            message = error.response.data.message;
                        }
                        throw new Error(`E_CONFLICT - Message: ${message}`);
                    }
                }
                throw this.getCheApiError(error, endpoint);
            }
            if (response && response.data) {
                return response.data;
            }
            else {
                throw new Error('E_BAD_RESP_CHE_SERVER');
            }
        });
    }
    isAuthenticationEnabled(responseTimeoutMs = this.defaultCheResponseTimeoutMs) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const endpoint = `${this.cheApiEndpoint}/keycloak/settings`;
            let response = null;
            try {
                response = yield this.axios.get(endpoint, { timeout: responseTimeoutMs });
            }
            catch (error) {
                if (error.response && (error.response.status === 404 || error.response.status === 503)) {
                    return false;
                }
                else {
                    throw this.getCheApiError(error, endpoint);
                }
            }
            if (!response || response.status !== 200 || !response.data) {
                throw new Error('E_BAD_RESP_CHE_API');
            }
            return true;
        });
    }
    getCheApiError(error, endpoint) {
        if (error.response) {
            const status = error.response.status;
            if (status === 403) {
                return new Error(`E_CHE_API_FORBIDDEN - Endpoint: ${endpoint} - Message: ${JSON.stringify(error.response.data.message)}`);
            }
            else if (status === 401) {
                return new Error(`E_CHE_API_UNAUTHORIZED - Endpoint: ${endpoint} - Message: ${JSON.stringify(error.response.data)}`);
            }
            else if (status === 404) {
                return new Error(`E_CHE_API_NOTFOUND - Endpoint: ${endpoint} - Message: ${JSON.stringify(error.response.data)}`);
            }
            else {
                // The request was made and the server responded with a status code
                // that falls out of the range of 2xx
                return new Error(`E_CHE_API_UNKNOWN_ERROR - Endpoint: ${endpoint} -Status: ${error.response.status}`);
            }
        }
        else if (error.request) {
            // The request was made but no response was received
            // `error.request` is an instance of XMLHttpRequest in the browser and an instance of
            // http.ClientRequest in node.js
            return new Error(`E_CHE_API_NO_RESPONSE - Endpoint: ${endpoint} - Error message: ${error.message}`);
        }
        else {
            // Something happened in setting up the request that triggered an Error
            return new Error(`E_CHECTL_UNKNOWN_ERROR - Endpoint: ${endpoint} - Message: ${error.message}`);
        }
    }
}
exports.CheApiClient = CheApiClient;
//# sourceMappingURL=che-api-client.js.map