"use strict";
/*********************************************************************
 * Copyright (c) 2019 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 **********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const client_node_1 = require("@kubernetes/client-node");
const command_1 = require("@oclif/command");
const flags_1 = require("@oclif/parser/lib/flags");
const yaml = require("js-yaml");
const devfile_1 = require("../../api/devfile");
const kube_1 = require("../../api/kube");
let kube;
const stringLitArray = (arr) => arr;
const languages = stringLitArray(['java', 'typescript', 'go', 'python', 'c#']);
const editors = stringLitArray(['theia-next', 'theia-1.0.0']);
const LanguagesComponents = new Map([
    ['java', { type: devfile_1.TheEndpointName.ChePlugin, alias: 'java-ls', id: 'redhat/java/latest' }],
    ['typescript', { type: devfile_1.TheEndpointName.ChePlugin, alias: 'typescript-ls', id: 'che-incubator/typescript/latest' }],
    ['go', { type: devfile_1.TheEndpointName.ChePlugin, alias: 'go-ls', id: 'ms-vscode/go/latest' }],
    ['python', { type: devfile_1.TheEndpointName.ChePlugin, alias: 'python-ls', id: 'ms-python/python/latest' }],
    ['c#', { type: devfile_1.TheEndpointName.ChePlugin, alias: 'csharp-ls', id: 'redhat-developer/che-omnisharp-plugin/latest' }],
]);
const EditorComponents = new Map([
    ['theia-next', { type: devfile_1.TheEndpointName.CheEditor, alias: 'theia-editor', id: 'eclipse/che-theia/next' }],
    ['theia-1.0.0', { type: devfile_1.TheEndpointName.CheEditor, alias: 'theia-editor', id: 'eclipse/che-theia/1.0.0' }]
]);
class Generate extends command_1.Command {
    run() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { flags } = this.parse(Generate);
            kube = new kube_1.KubeHelper(flags);
            const notifier = require('node-notifier');
            let name = flags.name || 'chectl-generated';
            let devfile = {
                apiVersion: '1.0.0',
                metadata: {
                    name
                }
            };
            if (flags['git-repo'] !== undefined) {
                const repo = {
                    type: 'git',
                    location: flags['git-repo']
                };
                const project = {
                    source: repo,
                    name: flags['git-repo'].split('/').pop() || 'git-project'
                };
                if (devfile.projects) {
                    devfile.projects.push(project);
                }
                else {
                    devfile.projects = [project];
                }
            }
            if (flags.dockerimage !== undefined) {
                const component = {
                    alias: `${flags.dockerimage.replace(/[\.\/:]/g, '-').substring(0, 20)}`,
                    type: devfile_1.TheEndpointName.Dockerimage,
                    image: `${flags.dockerimage}`,
                    memoryLimit: '512M',
                    mountSources: true,
                    command: ['tail'],
                    args: ['-f', '/dev/null']
                };
                if (devfile.components) {
                    devfile.components.push(component);
                }
                else {
                    devfile.components = [component];
                }
            }
            if (flags.selector !== undefined) {
                let k8sList = {
                    kind: 'List',
                    apiVersion: 'v1',
                    metadata: {
                        name: `${flags.selector}`
                    },
                    items: new Array()
                };
                const deployments = yield this.getDeploymentsBySelector(flags.selector, flags.namespace);
                const services = yield this.getServicesBySelector(flags.selector, flags.namespace);
                const ingresses = yield this.getIngressesBySelector(flags.selector, flags.namespace);
                const pvcs = yield this.getPersistentVolumeClaimsBySelector(flags.selector, flags.namespace);
                deployments.forEach((element) => {
                    k8sList.items.push(element);
                });
                services.forEach((element) => {
                    k8sList.items.push(element);
                });
                ingresses.forEach((element) => {
                    k8sList.items.push(element);
                });
                pvcs.forEach((element) => {
                    k8sList.items.push(element);
                });
                const component = {
                    type: devfile_1.TheEndpointName.Kubernetes,
                    alias: `${flags.selector}`,
                    referenceContent: `${yaml.safeDump(k8sList, { skipInvalid: true })}`
                };
                if (devfile.components) {
                    devfile.components.push(component);
                }
                else {
                    devfile.components = [component];
                }
            }
            if (flags.plugin !== undefined) {
                if (devfile.components) {
                    devfile.components.push(JSON.parse(flags.plugin));
                }
                else {
                    devfile.components = [JSON.parse(flags.plugin)];
                }
            }
            if (flags.language !== undefined) {
                if (languages.indexOf(flags.language) === -1) {
                    this.error(`Language ${flags.language} is not supported. Supported languages are ${languages}`);
                }
                const components = this.getPluginsByLanguage(flags.language);
                if (devfile.components && components) {
                    devfile.components.push(components);
                }
                else if (components) {
                    devfile.components = [components];
                }
            }
            if (flags.editor !== undefined) {
                if (editors.indexOf(flags.editor) === -1) {
                    this.error(`Editor ${flags.editor} is not supported. Supported editors are ${editors}`);
                }
                const components = EditorComponents.get(flags.editor);
                if (devfile.components && components) {
                    devfile.components.push(components);
                }
                else if (components) {
                    devfile.components = [components];
                }
            }
            if (flags.command !== undefined && devfile.components && devfile.components.length > 0) {
                let workdir = '/projects/';
                if (devfile.projects && devfile.projects.length > 0) {
                    workdir += devfile.projects[0].name;
                }
                const command = {
                    name: `${flags.command}`,
                    actions: [
                        {
                            type: 'exec',
                            command: `${flags.command}`,
                            component: `${devfile.components[0].alias}`,
                            workdir
                        }
                    ]
                };
                if (devfile.commands) {
                    devfile.commands.push(command);
                }
                else {
                    devfile.commands = [command];
                }
            }
            // Add header
            this.log('# Generated by crwctl (see https://github.com/redhat-developer/codeready-workspaces-chectl):');
            // only arguments after devfile:generate
            const index = process.argv.indexOf('devfile:generate');
            const updatedArgs = process.argv.slice(index).map(arg => {
                if (arg.indexOf(' ') >= 0) {
                    return arg.replace(/(.*?)=(.*)/g, '$1=\"$2\"');
                }
                else {
                    return arg;
                }
            });
            this.log(`# crwctl ${updatedArgs.join(' ')}`);
            this.log(yaml.safeDump(devfile));
            notifier.notify({
                title: 'crwctl',
                message: 'Command devfile:generate has completed successfully.'
            });
            this.exit(0);
        });
    }
    getPluginsByLanguage(language) {
        return LanguagesComponents.get(language);
    }
    getDeploymentsBySelector(labelSelector, namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            let items = new Array();
            const k8sDeployList = yield kube.getDeploymentsBySelector(labelSelector, namespace);
            k8sDeployList.items.forEach((item) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                let deployment = new client_node_1.V1Deployment();
                deployment.apiVersion = 'apps/v1';
                deployment.kind = 'Deployment';
                deployment.metadata = new client_node_1.V1ObjectMeta();
                deployment.metadata.labels = Object.assign({}, item.metadata.labels);
                deployment.metadata.name = item.metadata.name;
                deployment.spec = new client_node_1.V1DeploymentSpec();
                deployment.spec.selector = item.spec.selector;
                deployment.spec.template = new client_node_1.V1PodTemplateSpec();
                deployment.spec.template.metadata = new client_node_1.V1ObjectMeta();
                deployment.spec.template.metadata.labels = Object.assign({}, item.spec.template.metadata.labels);
                deployment.spec.template.metadata.name = item.spec.template.metadata.name;
                deployment.spec.template.spec = item.spec.template.spec;
                yield items.push(deployment);
            }));
            return items;
        });
    }
    getServicesBySelector(labelSelector, namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            let items = new Array();
            const k8sServicesList = yield kube.getServicesBySelector(labelSelector, namespace);
            k8sServicesList.items.forEach((item) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                let service = new client_node_1.V1Service();
                service.kind = 'Service';
                service.apiVersion = 'v1';
                service.metadata = new client_node_1.V1ObjectMeta();
                service.metadata.labels = Object.assign({}, item.metadata.labels);
                service.metadata.name = item.metadata.name;
                service.spec = new client_node_1.V1ServiceSpec();
                service.spec.type = item.spec.type;
                service.spec.selector = item.spec.selector;
                service.spec.ports = new Array();
                item.spec.ports.forEach(port => {
                    let svcPort = new client_node_1.V1ServicePort();
                    svcPort.port = port.port;
                    service.spec.ports.push(svcPort);
                });
                yield items.push(service);
            }));
            return items;
        });
    }
    getIngressesBySelector(labelSelector, namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            let items = new Array();
            const k8sIngressesList = yield kube.getIngressesBySelector(labelSelector, namespace);
            k8sIngressesList.items.forEach((item) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                let ingress = new client_node_1.ExtensionsV1beta1Ingress();
                ingress.kind = 'Ingress';
                ingress.apiVersion = 'extensions/v1beta1';
                ingress.metadata = new client_node_1.V1ObjectMeta();
                ingress.metadata.labels = Object.assign({}, item.metadata.labels);
                ingress.metadata.name = item.metadata.name;
                ingress.spec = item.spec;
                yield items.push(ingress);
            }));
            return items;
        });
    }
    getPersistentVolumeClaimsBySelector(labelSelector, namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            let items = new Array();
            const k8sPVCsList = yield kube.getPersistentVolumeClaimsBySelector(labelSelector, namespace);
            k8sPVCsList.items.forEach((item) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                let pvc = new client_node_1.V1PersistentVolumeClaim();
                pvc.kind = 'PersistentVolumeClaim';
                pvc.apiVersion = 'v1';
                pvc.metadata = new client_node_1.V1ObjectMeta();
                pvc.metadata.labels = Object.assign({}, item.metadata.labels);
                pvc.metadata.name = item.metadata.name;
                pvc.spec = new client_node_1.V1PersistentVolumeClaimSpec();
                pvc.spec.accessModes = item.spec.accessModes;
                pvc.spec.resources = item.spec.resources;
                yield items.push(pvc);
            }));
            return items;
        });
    }
}
exports.default = Generate;
Generate.description = 'generate and print a devfile to stdout given some Kubernetes resources and other workspaces features (project, language-support, commands etc...)';
Generate.flags = {
    help: command_1.flags.help({ char: 'h' }),
    name: flags_1.string({
        description: 'Workspace name',
        default: '',
        env: 'WSNAME',
        required: false,
    }),
    dockerimage: flags_1.string({
        description: 'dockerimage component to include in the Devfile',
        env: 'DOCKERIMAGE',
        required: false,
    }),
    namespace: flags_1.string({
        description: 'Kubernetes namespace where the resources are defined',
        default: '',
        env: 'NAMESPACE',
        required: false,
    }),
    editor: flags_1.string({
        description: `Specify the editor component. Currently supported editors: ${editors}`,
        env: 'EDITOR',
        required: false,
    }),
    selector: flags_1.string({
        description: 'label selector to filter the Kubernetes resources. For example --selector="app.kubernetes.io/name=employee-manager"',
        env: 'SELECTOR',
        required: false,
    }),
    language: flags_1.string({
        description: `Add support for a particular language. Currently supported languages: ${languages}`,
        env: 'LANGUAGE_SUPPORT',
        required: false,
    }),
    plugin: flags_1.string({
        description: 'CodeReady Workspaces plugin to include in the workspace. The format is JSON. For example this is a valid CodeReady Workspaces plugin specification: {"type": "TheEndpointName.ChePlugin", "alias": "java-ls", "id": "redhat/java/0.38.0"}',
        env: 'CHE_PLUGIN',
        required: false,
    }),
    'git-repo': flags_1.string({
        description: 'Source code git repository to include in the workspace',
        env: 'GIT_REPO',
        required: false,
    }),
    command: flags_1.string({
        description: 'Command to include in the workspace',
        env: 'COMMAND',
        required: false,
    }),
};
//# sourceMappingURL=generate.js.map