package org.jbpm.gd.jpdl.ui.properties;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.FocusListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetWidgetFactory;
import org.jbpm.gd.jpdl.model.Decision;
import org.jbpm.gd.jpdl.model.Handler;
import org.jbpm.gd.jpdl.model.JpdlElementFactory;
import org.jbpm.gd.jpdl.ui.dialog.ChooseDelegationClassDialog;

public class DecisionConfigurationComposite implements SelectionListener, FocusListener {
	
	public static DecisionConfigurationComposite create(TabbedPropertySheetWidgetFactory widgetFactory, Composite parent) {
		DecisionConfigurationComposite result = new DecisionConfigurationComposite();
		result.widgetFactory = widgetFactory;
		result.parent = parent;
		result.create();
		return result;
	}
	
	private TabbedPropertySheetWidgetFactory widgetFactory;
	private Composite parent;
	private Decision decision;
	
	private Group mainInfoGroup;
	private Label decisionConfigurationTypeLabel;
	private CCombo decisionConfigurationTypeCombo;
	private Group delegationConfigurationInfoGroup;
	private DelegationConfigurationComposite delegationConfigurationComposite;
	private Group expressionInfoGroup;
	private Label expressionLabel;
	private Text expressionText;
	
	private String selectedDecisionType;
		
	private DecisionConfigurationComposite() {}
	
	public void setDecision(Decision decision) {
		if (this.decision == decision) return;
		unhookListeners();
		this.decision = decision;
		if (decision == null) {
			clearControls();
		} else {
			updateControls();
			hookListeners();
		}
	}
	
	private void hookListeners() {
		expressionText.addFocusListener(this);
		expressionText.addSelectionListener(this);
	}
	
	private void unhookListeners() {
		expressionText.removeFocusListener(this);
		expressionText.removeSelectionListener(this);
	}
	
	private void clearControls() {
		decisionConfigurationTypeCombo.setText("");
		delegationConfigurationComposite.setDelegation(null);
		expressionText.setText("");
	}
	
	private void updateControls() {
		if (decision.getHandler() != null) {
			decisionConfigurationTypeCombo.setText("delegation");
			delegationConfigurationComposite.setDelegation(decision.getHandler());	
			delegationConfigurationInfoGroup.setVisible(true);
		} else if (decision.getExpression() != null) {
			decisionConfigurationTypeCombo.setText("expression");
			expressionText.setText(decision.getExpression());
			expressionInfoGroup.setVisible(true);
		} else {
			decisionConfigurationTypeCombo.setText("");
			delegationConfigurationInfoGroup.setVisible(false);
			expressionInfoGroup.setVisible(false);
		}
	}
	
	private void create() {
		createMainInfoGroup();
		createExpressionInfoGroup();
		createDelegationInfoGroup();
		initializeLayouts();
	}
	
	private void createMainInfoGroup() {
		mainInfoGroup = widgetFactory.createGroup(parent, "Decision Configuration");
		mainInfoGroup.setLayout(new FormLayout());
		decisionConfigurationTypeLabel = widgetFactory.createLabel(mainInfoGroup, "Configuration Type");
		decisionConfigurationTypeCombo = widgetFactory.createCCombo(mainInfoGroup, SWT.BORDER);
		decisionConfigurationTypeCombo.setItems(new String[] {"", "expression", "delegation"});
		decisionConfigurationTypeCombo.setEditable(false);
		decisionConfigurationTypeCombo.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				handleConfigurationTypeComboSelected();
			}
		});
	}
	
	private void createDelegationInfoGroup() {
		delegationConfigurationInfoGroup = widgetFactory.createGroup(parent, "Delegation Information");
		delegationConfigurationInfoGroup.setLayout(new FormLayout());
		delegationConfigurationInfoGroup.setVisible(false);
		delegationConfigurationComposite = DelegationConfigurationComposite.create(widgetFactory, delegationConfigurationInfoGroup);
		delegationConfigurationComposite.setChooseDelegationClassDialog(createChooseDecisionHandlerDialog());
	}
	
	private ChooseDelegationClassDialog createChooseDecisionHandlerDialog() {
		return new ChooseDelegationClassDialog(
				null, 
				"org.jbpm.graph.node.DecisionHandler",
				"Choose Decision Handler",
				"Choose a decision handler from the list");
	}
	
	private void createExpressionInfoGroup() {
		expressionInfoGroup = widgetFactory.createGroup(parent, "Expression Information");
		expressionInfoGroup.setLayout(new FormLayout());
		expressionInfoGroup.setVisible(false);
		expressionLabel = widgetFactory.createLabel(expressionInfoGroup, "Expression");
		expressionText = widgetFactory.createText(expressionInfoGroup, "");
	}
	
	private void initializeLayouts() {
		decisionConfigurationTypeLabel.setLayoutData(createConfigurationTypeLabelLayoutData());
		decisionConfigurationTypeCombo.setLayoutData(createConfigurationTypeComboLayoutData());
		mainInfoGroup.setLayoutData(createMainInfoGroupLayoutData());
		delegationConfigurationInfoGroup.setLayoutData(createDelegationInfoGroupLayoutData());
		expressionInfoGroup.setLayoutData(createDelegationInfoGroupLayoutData());
		expressionLabel.setLayoutData(createExpressionLabelLayoutData());
		expressionText.setLayoutData(createExpressionTextLayoutData());
	}
	
	private FormData createExpressionLabelLayoutData() {
		FormData result = new FormData();
		result.top = new FormAttachment(0, 5);
		result.left = new FormAttachment(0, 5);
		return result;
	}
	
	private FormData createExpressionTextLayoutData() {
		FormData result = new FormData();
		result.top = new FormAttachment(0, 5);
		result.left = new FormAttachment(expressionLabel, 5);
		result.right = new FormAttachment(100, -5);
		result.bottom = new FormAttachment(100, -5);
		return result;
	}
	
	private FormData createConfigurationTypeLabelLayoutData() {
		FormData result = new FormData();
		result.top = new FormAttachment(0, 7);
		result.left = new FormAttachment(0, 5);
		result.top = new FormAttachment(0, 5);
		return result;
	}
	
	private FormData createConfigurationTypeComboLayoutData() {
		FormData result = new FormData();
		result.top = new FormAttachment(0, 5);
		result.left = new FormAttachment(decisionConfigurationTypeLabel, 5);
		result.bottom = new FormAttachment(100, -5);
		return result;
	}
	
	private FormData createMainInfoGroupLayoutData() {
		FormData result = new FormData();
		result.left = new FormAttachment(0, 5);
		result.top = new FormAttachment(0, 5);
		result.right = new FormAttachment(100, -5);
		return result;
	}
	
	private FormData createDelegationInfoGroupLayoutData() {
		FormData result = new FormData();
		result.left = new FormAttachment(0, 5);
		result.top = new FormAttachment(mainInfoGroup, 5);
		result.bottom = new FormAttachment(100, -5);
		result.right = new FormAttachment(100, -5);
		return result;
	}

	private void updateExpressionText() {
		decision.setExpression(expressionText.getText());
	}
	
	private void updateControlVisibility() {
		delegationConfigurationInfoGroup.setVisible("delegation".equals(decisionConfigurationTypeCombo.getText()));
		expressionInfoGroup.setVisible("expression".equals(decisionConfigurationTypeCombo.getText()));
	}
	
	private void handleConfigurationTypeComboSelected() {
		String selection = decisionConfigurationTypeCombo.getText();
		if (selection.equals(selectedDecisionType)) return;
		selectedDecisionType = selection;
		updateControlVisibility();
		if (decision != null) {
			removeExpressionAndHandlerInfo();
			updateExpressionAndHandlerInfo();
		}
	}

	private void updateExpressionAndHandlerInfo() {
		if ("expression".equals(selectedDecisionType)) {
			decision.setExpression(expressionText.getText());
		} else if ("delegation".equals(selectedDecisionType)) {
			Handler handler = (Handler)JpdlElementFactory.INSTANCE.createById("org.jbpm.gd.jpdl.core.handler");
			decision.setHandler(handler);
			delegationConfigurationComposite.updateDelegationInfo(handler);
		}
	}
	
	private void removeExpressionAndHandlerInfo() {
		decision.setExpression(null);
		decision.setHandler(null);
	}

	public void focusLost(FocusEvent e) {
		if (e.widget == expressionText) {
			updateExpressionText();
		}
	}

	public void focusGained(FocusEvent e) {
	}
	
	public void widgetDefaultSelected(SelectionEvent e) {
		if (e.widget == expressionText) {
			updateExpressionText();
		}
	}

	public void widgetSelected(SelectionEvent e) {
	}
	
}
