package org.jboss.eap.util.xp.patch.stream.manager;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamConstants;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;

import org.jboss.staxmapper.XMLElementReader;
import org.jboss.staxmapper.XMLExtendedStreamReader;
import org.jboss.staxmapper.XMLMapper;

/**
 * @author <a href="mailto:kabir.khan@jboss.com">Kabir Khan</a>
 */
public class PatchXml implements XMLStreamConstants, XMLElementReader<PatchXml.Validator> {
    private static final XMLMapper MAPPER = XMLMapper.Factory.create();
    private static final PatchXml XML = new PatchXml();
    private static final XMLInputFactory INPUT_FACTORY = XMLInputFactory.newInstance();

    static {
        MAPPER.registerRootElement(new QName(Namespace.PATCH_1_0.getNamespace(), "patch"), XML);
        MAPPER.registerRootElement(new QName(Namespace.PATCH_1_1.getNamespace(), "patch"), XML);
        MAPPER.registerRootElement(new QName(Namespace.PATCH_1_2.getNamespace(), "patch"), XML);
    }

    public enum Namespace {

        PATCH_1_0("urn:jboss:patch:1.0"),
        PATCH_1_1("urn:jboss:patch:1.1"),
        PATCH_1_2("urn:jboss:patch:1.2"),
        UNKNOWN(null),
        ;

        private final String namespace;
        Namespace(String namespace) {
            this.namespace = namespace;
        }

        public String getNamespace() {
            return namespace;
        }

        static Map<String, Namespace> elements = new HashMap<String, Namespace>();
        static {
            for(Namespace element : Namespace.values()) {
                if(element != UNKNOWN) {
                    elements.put(element.namespace, element);
                }
            }
        }

        static Namespace forUri(String name) {
            final Namespace element = elements.get(name);
            return element == null ? UNKNOWN : element;
        }
    }

    private PatchXml() {
        //
    }

    public static void validateBasePatch(final InputStream stream) throws XMLStreamException {
        Validator validator = parse(stream);
        validator.validateBasePatch();
    }

    public static void validateXpPatch(final InputStream stream) throws XMLStreamException {
        Validator validator = parse(stream);
        validator.validateXpPatch();
    }

    private static Validator parse(final InputStream stream) throws XMLStreamException {
        XMLStreamReader reader = getXMLInputFactory().createXMLStreamReader(stream);
        try {
            final Validator validator = new Validator();
            MAPPER.parseDocument(validator, reader);
            return validator;
        } finally {
            reader.close();
        }
    }

    private static XMLInputFactory getXMLInputFactory() throws XMLStreamException {
        final XMLInputFactory inputFactory = INPUT_FACTORY;
        setIfSupported(inputFactory, XMLInputFactory.IS_VALIDATING, Boolean.FALSE);
        setIfSupported(inputFactory, XMLInputFactory.SUPPORT_DTD, Boolean.FALSE);
        return inputFactory;
    }

    private static void setIfSupported(final XMLInputFactory inputFactory, final String property, final Object value) {
        if (inputFactory.isPropertySupported(property)) {
            inputFactory.setProperty(property, value);
        }
    }



    @Override
    public void readElement(XMLExtendedStreamReader reader, Validator validator) throws XMLStreamException {
        final int count = reader.getAttributeCount();
        for (int i = 0; i < count; i++) {
            final String value = reader.getAttributeValue(i);
            switch (reader.getAttributeLocalName(i)) {
                case "id": {
                    validator.setPatchId(value);
                }
                break;
            }
        }

        while (reader.hasNext() && reader.nextTag() != END_ELEMENT) {
            final String localName = reader.getLocalName();
            switch (localName) {
                case "element": {
                    parsePatchElement(reader, validator);
                }
                break;
                default:
                    consumeChildren(reader);
            }
        }
    }

    private void parsePatchElement(XMLExtendedStreamReader reader, Validator validator) throws XMLStreamException {
        String id = null;
        final int count = reader.getAttributeCount();
        for (int i = 0; i < count; i++) {
            final String value = reader.getAttributeValue(i);
            switch (reader.getAttributeLocalName(i)) {
                case "id": {
                    id = value;
                }
                break;
            }
        }

        String layerName = null;
        while (reader.hasNext() && reader.nextTag() != END_ELEMENT) {
            final String localName = reader.getLocalName();
            switch (localName) {
                case "upgrade": {
                    layerName = getLayerFromUpgradeElement(reader);
                }
                break;
                default:
                    consumeChildren(reader);
            }
        }

        validator.addLayerIdAndName(id, layerName);
    }

    private String getLayerFromUpgradeElement(XMLExtendedStreamReader reader) throws XMLStreamException {
        String layerName = null;
        final int count = reader.getAttributeCount();
        for (int i = 0; i < count; i++) {
            final String value = reader.getAttributeValue(i);
            switch (reader.getAttributeLocalName(i)) {
                case "name": {
                    layerName = value;
                }
                break;
            }
        }

        while (reader.hasNext() && reader.nextTag() != END_ELEMENT) {
            consumeChildren(reader);
        }

        return layerName;
    }

    public void consumeChildren(XMLExtendedStreamReader reader) throws XMLStreamException {

        while (reader.hasNext()) {
            int type = reader.next();
            switch (type) {
                case START_ELEMENT:
                    consumeChildren(reader);
                    break;
                case END_ELEMENT:
                    return;
                default:
                    // Do nothing, easy!
            }
        }
    }



    static class Validator {
        private static final Pattern BASE_PATCH_ID = Pattern.compile("jboss-eap-7\\.\\d+\\.\\d+\\.CP");
        private static final Pattern XP_PATCH_ID = Pattern.compile("jboss-eap-xp-\\d+\\.\\d+\\.\\d+\\.CP");
        private String patchId;
        private final List<String> layerIds = new ArrayList<>();
        private final List<String> layerNames = new ArrayList<>();

        public Validator() {
        }

        public void setPatchId(String patchName) {
            this.patchId = patchName;
        }

        public void addLayerIdAndName(String layerId, String layerName) {
            layerIds.add(layerId);
            layerNames.add(layerName);
        }

        public void validateBasePatch() {
            validatePatch(ManagerArgsParser.ARG_BASE_PATCH, BASE_PATCH_ID, "layer-base-", "base");
        }

        public void validateXpPatch() {
            validatePatch(ManagerArgsParser.ARG_XP_PATCH, XP_PATCH_ID, "layer-", "microprofile");
        }

        private void validatePatch(String patchArg, Pattern patchIdPattern, String layerIdPrefix, String layer) {
            if (patchId == null || !patchIdPattern.matcher(patchId).matches()) {
                throw ManagerLogger.LOGGER.invalidPatchIdForPatch(patchArg, patchId);
            }

            if (layerIds.size() != 1 || layerNames.size() != 1) {
                // layerNames will be the same size as layerIds according to the patching xsd
                throw ManagerLogger.LOGGER.patchElementsNotEqualsOneInPatch(patchArg);
            }

            if (!layerIds.get(0).equals(layerIdPrefix + patchId)) {
                throw ManagerLogger.LOGGER.badPatchElementIdInPatch(patchArg, layerIds.get(0));
            }

            if (!layerNames.get(0).equals(layer)) {
                throw ManagerLogger.LOGGER.badUpgradeElementNameInPatch(patchArg, layerNames.get(0), layer);
            }
        }
    }
}
