package org.jboss.eap.util.xp.patch.stream.manager;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.jboss.dmr.ModelNode;

/**
 * @author <a href="mailto:kabir.khan@jboss.com">Kabir Khan</a>
 */
public class RemoveManagerAction extends ManagerAction {
    private final FileSet removeFileSet;
    private final Path jbossHome;
    private final Path modulesDir;
    private final ServerWrapper serverWrapper;
    private boolean error;

    public RemoveManagerAction(ManagerStatus status, boolean supportPolicyAccepted, Path jbossHome, Path modulesDir) {
        super(status, supportPolicyAccepted);
        this.removeFileSet = status.getFileSet();
        this.jbossHome = jbossHome;
        this.modulesDir = modulesDir;
        serverWrapper = new ServerWrapper(jbossHome, modulesDir);
    }

    @Override
    ManagerState getToManagerState() {
        return ManagerState.CLEAN;
    }

    @Override
    ManagerCommand getCommand() {
        return ManagerCommand.REMOVE;
    }

    @Override
    void doExecute() throws Exception {
        try {
            System.out.println(ManagerLogger.LOGGER.checkingForPatchesToRollback());
            int numberPatches = removePatchStreamPatches();
            if (numberPatches == 0) {
                System.out.println(ManagerLogger.LOGGER.noPatchesToRemove());
            }
        } catch (Exception e) {
            System.err.println(ManagerLogger.LOGGER.errorRemovingPatches());
            e.printStackTrace();
        }
        removeFiles();
        System.out.println(ManagerLogger.LOGGER.successfullyRemovedExpansionPack());
    }

    private int removePatchStreamPatches() throws Exception {
        int numberPatches = 0;
        for (String streamName : ManagerActionUtils.getManagedPatchStreamNames()) {
            // TODO if these have inter-dependencies we will need to do some TBD checks. For now it is fine since there is only one
            numberPatches += removePatchesForStream(streamName);
        }
        return numberPatches;
    }

    private int removePatchesForStream(String streamName) throws Exception {
        int numberPatches = 0;
        while (true) {
            try {
                serverWrapper.start();

                ModelNode idNode = serverWrapper.execute(
                        new OperationBuilder("read-attribute")
                                .param("name", "cumulative-patch-id")
                                .addr("core-service", "patching")
                                .addr("patch-stream", streamName)
                                .build());

                String id = idNode.asString();

                if (id.equals("base")) {
                    break;
                }

                System.out.println(ManagerLogger.LOGGER.rollingBackPatch(id, streamName));
                serverWrapper.execute(
                        new OperationBuilder("rollback")
                                .param("patch-id", id)
                                .param("reset-configuration", "false")
                                .addr("core-service", "patching")
                                .addr("patch-stream", streamName)
                                .build());
                numberPatches++;
                System.out.println(ManagerLogger.LOGGER.rolledBackPatchSuccessfully());
            } catch (Exception e) {

            } finally {
                serverWrapper.stop();
                serverWrapper.resetRestartRequiredStatusInPatching();
            }
        }
        return numberPatches;
    }

    private void removeFiles() throws Exception {
        List<String> moduleLayers = ManagerActionUtils.getManagedModuleLayerNames();
        List<String> patchNamePatterns = ManagerActionUtils.getManagedPatchNamePatterns();
        remove(removeFileSet.getLayersConf());
        for (Path p : removeFileSet.getModuleLayerDirs()) {
            remove(p);
        }
        for (Path p : removeFileSet.getPatchStreamFiles()) {
            remove(p);
        }
        for (String layer : moduleLayers) {
            Path p = jbossHome.resolve(".installation/layers/" + layer);
            remove(p);
        }
        for (String namePattern : patchNamePatterns) {
            Pattern pattern = Pattern.compile(namePattern);
            Path p = jbossHome.resolve(".installation/patches");
            if (Files.exists(p)) {
                try (Stream<Path> stream = Files.list(p)) {
                    List<Path> paths = stream.collect(Collectors.toList());
                    for (Path path : paths) {
                        if (pattern.matcher(path.getFileName().toString()).matches()) {
                            remove(path);
                        }
                    }
                }
            }
        }
    }

    private void remove(Path path) {
        if (path == null) {
            return;
        }
        if (Files.exists(path)) {
            try {
                if (Files.isDirectory(path)) {
                    Files.walkFileTree(path, new FileVisitors.DeleteDirectory());
                } else {
                    Files.delete(path);
                }
            } catch (IOException e) {
                error = true;
                System.err.println(ManagerLogger.LOGGER.errorDeletingStateWillBeInconsistent(path));
                e.printStackTrace();
            }
        }
    }
}
