/*
 * JBoss, Home of Professional Open Source
 * Copyright 2020, JBoss Inc., and individual contributors as indicated
 * by the @authors tag.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.eap.util.xp.patch.stream.manager;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

import org.jboss.as.controller.client.Operation;
import org.jboss.dmr.ModelNode;

/**
 * @author <a href="mailto:kabir.khan@jboss.com">Kabir Khan</a>
 */
public class SetupManagerAction extends ManagerAction {

    private static final String PATCH_STREAM_FILE_CONTENTS =
            "current-version=0.0.0\npatches=\ncumulative-patch-id=base\ninstalled-patches=";
    private static final String SERVER_TARGET_STREAMS = "server-target-streams";
    private static final String SERVER_TARGET_LAYERS = "server-target-layers";

    private final FileSet toFileSet;
    private final URL jarUrl;
    private final Path jbossHome;
    private final Path serverConfigDir;
    private final Path basePatch;
    private final Path xpPatch;
    private final List<Path> createdLayers = new ArrayList<>();
    private final List<Path> createdPatchStreams = new ArrayList<>();
    private final ServerWrapper serverWrapper;


    private SetupManagerAction(ManagerStatus status, boolean supportPolicyAccepted, URL jarUrl, Path jbossHome,
                               Path modulesDir, Path serverConfigDir, Path basePatch, Path xpPatch) {
        super(status, supportPolicyAccepted);
        this.toFileSet = status.getToFileSet(getToManagerState());
        this.jarUrl = jarUrl;
        this.jbossHome = jbossHome;
        this.serverConfigDir = serverConfigDir;
        this.basePatch = basePatch;
        this.xpPatch = xpPatch;
        serverWrapper = new ServerWrapper(jbossHome, modulesDir);
    }

    static SetupManagerAction create(ManagerStatus status, boolean supportPolicyAccepted, URL jarUrl, Path jbossHome,
                                     Path modulesDir, Path serverConfigDir, Path basePatch, Path xpPatch) throws Exception {
        return new SetupManagerAction(status, supportPolicyAccepted, jarUrl, jbossHome,
                modulesDir, serverConfigDir, basePatch, xpPatch);
    }

    @Override
    public ManagerState getToManagerState() {
        return ManagerState.INSTALLED;
    }

    @Override
    public ManagerCommand getCommand() {
        return ManagerCommand.SETUP;
    }

    @Override
    public void doExecute() throws Exception {

        Path tempDir = Files.createTempDirectory("wildfly-mp-manager");
        Path addedConfigs = tempDir.resolve("added-configs");
        try {
            unzipSelfToTemp(jarUrl, addedConfigs);
            install(addedConfigs);
        } finally {
            Files.walkFileTree(tempDir, new FileVisitors.DeleteDirectory());
        }
    }

    private void install(Path addedConfigs) throws Exception {
        System.out.println(ManagerLogger.LOGGER.startingSetup());

        // Set up the base manager stuff
        checkPatches();
        boolean success = false;
        try {
            setupLayersAndPatchStreams();
            copyAddedConfigs(addedConfigs);

            System.out.println(ManagerLogger.LOGGER.startingEmbeddedServer(jbossHome));
            serverWrapper.start();
            checkVersion();
            checkCumulativePatchId();
            applyPatches();
            success = true;
        } catch (Exception e) {
            System.err.println(ManagerLogger.LOGGER.errorSettingUpExpansionPackRollingBackAttemptedChanges());
            e.printStackTrace();
            if (Files.exists(toFileSet.getLayersConf())) {
                try {
                    Files.delete(toFileSet.getLayersConf());
                } catch (IOException ex) {
                    System.err.println(ManagerLogger.LOGGER.errorDeletingStateWillBeInconsistent(toFileSet.getLayersConf()));
                    ex.printStackTrace();
                }

                for (Path layerDir : createdLayers) {
                    if (Files.exists(layerDir)) {
                        try {
                            Files.walkFileTree(layerDir, new FileVisitors.DeleteDirectory());
                        } catch (IOException ex) {
                            System.err.println(ManagerLogger.LOGGER.errorDeletingStateWillBeInconsistent(layerDir));
                            ex.printStackTrace();
                        }
                    }
                }

                for (Path streamFile : createdPatchStreams) {
                    if (Files.exists(streamFile)) {
                        try {
                            Files.delete(streamFile);
                        } catch (IOException ex) {
                            System.err.println(ManagerLogger.LOGGER.errorDeletingStateWillBeInconsistent(streamFile));
                            ex.printStackTrace();
                        }
                    }
                }
            }
        } finally {
            serverWrapper.close();
            if (success) {
                System.out.println(ManagerLogger.LOGGER.expansionPackPreparedSuccessfully());
                if (xpPatch == null) {
                    System.out.println(ManagerLogger.LOGGER.applyXpPatchForFunctionality());
                }
            }
        }
    }

    private void unzipSelfToTemp(URL url, Path addedConfigs) throws Exception {
        byte[] buffer = new byte[1024];
        File file = new File(url.toURI());
        //boolean foundPatch = false;
        try (ZipInputStream zin = new ZipInputStream(new BufferedInputStream(new FileInputStream(file)))) {
            ZipEntry entry = zin.getNextEntry();
            while (entry != null) {
                try {
                    if (entry.isDirectory()) {
                        continue;
                    }
                    if (entry.getName().startsWith("added-configs") && !entry.isDirectory()) {
                        Path path = addedConfigs.getParent().resolve(entry.getName());
                        if (!Files.exists(path.getParent())) {
                            Files.createDirectories(path.getParent());
                        }
                        writeToFile(path, buffer, zin);
                    }
                } finally {
                    zin.closeEntry();
                    entry = zin.getNextEntry();
                }
            }
        }
    }

    private void writeToFile(Path path, byte[] buffer, ZipInputStream zin) throws Exception {
        try (OutputStream out = new BufferedOutputStream(new FileOutputStream(path.toFile()))) {
            int len;
            while ((len = zin.read(buffer)) > 0) {
                out.write(buffer, 0, len);
            }
        }
    }

    private void checkPatches() throws Exception {
        if (basePatch != null) {
            checkPatch(basePatch);
        }
        if (xpPatch != null) {
            checkPatch(xpPatch);
        }
    }

    private void checkPatch(Path patchFile) throws Exception {
        boolean base = patchFile == basePatch;
        String patchArg = base ? ManagerArgsParser.ARG_BASE_PATCH : ManagerArgsParser.ARG_XP_PATCH;
        try (InputStream in = findPatchXmlInZip(patchFile)) {
            if (in == null) {
                throw ManagerLogger.LOGGER.noPatchXmlFoundInPatchFile(patchArg, patchFile);
            }

            if (base) {
                PatchXml.validateBasePatch(in);
            } else {
                PatchXml.validateXpPatch(in);
            }
        }
    }

    private InputStream findPatchXmlInZip(Path patchFile)  {
        byte[] buffer = new byte[1024];
        try (ZipInputStream zin = new ZipInputStream(new BufferedInputStream(new FileInputStream(patchFile.toFile())))) {
            ZipEntry entry = zin.getNextEntry();
            while (entry != null) {
                try {
                    if (!entry.isDirectory()) {
                        if (entry.getName().equals("patch.xml")) {
                            try (ByteArrayOutputStream out = new ByteArrayOutputStream()) {
                                int len;
                                while ((len = zin.read(buffer)) > 0) {
                                    out.write(buffer, 0, len);
                                }
                                return new ByteArrayInputStream(out.toByteArray());
                            }
                        }
                    }
                } finally {
                    zin.closeEntry();
                    entry = zin.getNextEntry();
                }
            }
        } catch (IOException e) {
            throw ManagerLogger.LOGGER.patchFileIsNotAnArchive(e, patchFile);
        }

        return null;
    }

    private void applyPatches() throws Exception {
        // Now install the patches if needed
        if (basePatch != null) {
            applyBasePatch();
        }

        if (xpPatch != null) {
            if (basePatch != null) {
                // Need to stop server and reset the restart-required flag
                serverWrapper.stop();
                serverWrapper.resetRestartRequiredStatusInPatching();
                serverWrapper.start();
            }
            applyXpPatch();
        }
    }

    private void applyBasePatch() throws Exception {
        if (basePatch == null) {
            return;
        }
        System.out.println(ManagerLogger.LOGGER.applyingJBossEAPPatch(basePatch));
        applyPatch(basePatch);
    }

    private void applyXpPatch() throws Exception {
        if (xpPatch == null) {
            return;
        }
        System.out.println(ManagerLogger.LOGGER.applyingExpansionPackPatch(xpPatch));
        applyPatch(xpPatch);
    }

    private void applyPatch(Path patch) throws Exception {
        ModelNode operation = new OperationBuilder("patch")
                .addr("core-service", "patching")
                .param("override-modules", "false")
                .param("override-all", "false")
                .param("input-stream-index", "0")
                .build();

        final org.jboss.as.controller.client.OperationBuilder operationBuilder = org.jboss.as.controller.client.OperationBuilder.create(operation);
        operationBuilder.addFileAsAttachment(patch.toFile());
        serverWrapper.execute(operationBuilder.build());
    }

    private void copyAddedConfigs(Path addedConfigs) throws Exception {
        System.out.println(ManagerLogger.LOGGER.copyingAddedConfigs());
        if (!Files.exists(addedConfigs)) {
            return;
        }

        try (Stream<Path> stream = Files.list(addedConfigs)) {
            for (Path config : stream.collect(Collectors.toList())) {
                Path target = serverConfigDir.resolve(config.getFileName());
                if (!Files.exists(target)) {
                    System.out.println(ManagerLogger.LOGGER.copyinConfigFile(target));
                    Files.copy(config, target);
                } else {
                    Path renamed = target.getParent().resolve(target.getFileName().toString() + "." + System.currentTimeMillis());
                    System.out.println(ManagerLogger.LOGGER.copyingConfigFileAlreadyExists(target, renamed));
                    Files.copy(config, renamed);
                }
            }
        }
    }

    private void checkVersion() throws Exception {
        if (basePatch != null) {
            System.out.println(ManagerLogger.LOGGER.skipServerVersionCheck());
        } else {
            System.out.println(ManagerLogger.LOGGER.checkServerVersion());
            ModelNode versionNode = serverWrapper.execute(
                    new OperationBuilder("read-attribute")
                            .param("name", "product-version")
                            .build());
            String version = versionNode.asString();
            if (version.equals("7.3.0.GA")) {
                throw ManagerLogger.LOGGER.incompatibleServerVersionApplyBasePatchFirst(version);
            }
        }
    }

    private void checkCumulativePatchId() throws Exception {
        System.out.println(ManagerLogger.LOGGER.checkCumulativePatchId());
        ModelNode versionNode = serverWrapper.execute(
                new OperationBuilder("read-attribute")
                        .addr("core-service", "patching")
                        .param("name", "cumulative-patch-id")
                        .build());
        String id = versionNode.asString();
        if (!id.equals("base")) {
            Pattern pattern = Pattern.compile("jboss-eap-7\\.3\\.\\d+\\.CP");
            if (!pattern.matcher(id).matches()) {
                throw ManagerLogger.LOGGER.badCumulativePatchId(id);
            }
        }
    }

    private void setupLayersAndPatchStreams() throws Exception {
        System.out.println(ManagerLogger.LOGGER.settingUpLayersAndStreams());

        StringBuilder layersConfContents = new StringBuilder("layers=");
        boolean first = true;

        for (Path layerDir : toFileSet.getModuleLayerDirs()) {
            Files.createDirectories(layerDir);
            createdLayers.add(layerDir);

            if (!first) {
                layersConfContents.append(",");
            } else {
                first = false;
            }
            layersConfContents.append(layerDir.getFileName().toString());
        }
        layersConfContents.append("\n");

        Files.write(toFileSet.getLayersConf(), layersConfContents.toString().getBytes(StandardCharsets.UTF_8));

        for (Path streamFile : toFileSet.getPatchStreamFiles()) {
            Files.createFile(streamFile);
            createdPatchStreams.add(streamFile);
            Files.write(streamFile, PATCH_STREAM_FILE_CONTENTS.getBytes(StandardCharsets.UTF_8));
        }
    }
}
