/*
 * JBoss, Home of Professional Open Source
 * Copyright 2020, JBoss Inc., and individual contributors as indicated
 * by the @authors tag.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.eap.util.xp.patch.stream.manager;

import java.net.URL;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.logging.Level;
import java.util.logging.LogManager;
import java.util.logging.Logger;

import org.jboss.logmanager.formatters.PatternFormatter;
import org.jboss.logmanager.handlers.ConsoleHandler;

/**
 *
 * @author <a href="mailto:kabir.khan@jboss.com">Kabir Khan</a>
 */
public class ManagerMain {

    public static void main(String[] args) throws Exception {
        configureLogging();
        String version = ManagerManifestConfig.INSTANCE.getManagerVersion();
        System.out.println(ManagerLogger.LOGGER.startMessage(version));


        URL url = ManagerMain.class.getProtectionDomain().getCodeSource().getLocation();
        if (!url.toString().contains(".jar")) {
            throw ManagerLogger.LOGGER.managerNotAJar();
        }

        ManagerAction action = ManagerArgsParser.parseArguments(url, args);
        if (action != null) {
            ManagerAction.Status status = action.execute();

            // return non zero status code on error
            if (!ManagerAction.Status.OK.equals(status)) {
                System.exit(1);
            }
        }
    }

    private static void configureLogging() {
        try {
            // First get the system log manager system property and determine if it's already been set
            if (getSystemProperty("java.util.logging.manager") == null) {
                // We currently package the JBoss Log Manager with this tool. If the log manager is not set we should use
                // it as our default.
                setSystemProperty("java.util.logging.manager", "org.jboss.logmanager.LogManager");

                final LogManager logManager = LogManager.getLogManager();

                // Load the log manager and check to see if we're an instance of the JBoss Log Manager
                if (logManager.getClass() != org.jboss.logmanager.LogManager.class) {
                    System.err.println(ManagerLogger.LOGGER.unknownLogManager(LogManager.getLogManager().getClass().getCanonicalName(),
                            org.jboss.logmanager.LogManager.class.getCanonicalName()));
                } else {
                    final Logger rootLogger = logManager.getLogger("");
                    // If there are any handlers assigned to the root logger we're going to assume logging has already been
                    // configured.
                    if (rootLogger.getHandlers().length == 0) {
                        configureDefaultLogging(rootLogger);
                    }
                }
            }
        } catch (Throwable e) {
            System.err.println(ManagerLogger.LOGGER.failedToConfigureLogManager());
            e.printStackTrace(System.err);
        }
    }

    private static void configureDefaultLogging(final Logger rootLogger) {
        final PrivilegedAction<Object> action = () -> {
            final PatternFormatter formatter = new PatternFormatter("%K{level}[%-5p] [%c] (%t) %s%e%n");
            final ConsoleHandler handler = new ConsoleHandler(ConsoleHandler.Target.SYSTEM_OUT, formatter);
            rootLogger.setLevel(Level.INFO);
            rootLogger.addHandler(handler);
            return null;
        };
        if (System.getSecurityManager() == null) {
            action.run();
        } else {
            AccessController.doPrivileged(action);
        }
    }

    @SuppressWarnings("SameParameterValue")
    private static void setSystemProperty(final String key, final String value) {
        if (System.getSecurityManager() == null) {
            System.setProperty(key, value);
        }
        AccessController.doPrivileged((PrivilegedAction<Object>) () -> {
            System.setProperty(key, value);
            return null;
        });
    }

    @SuppressWarnings("SameParameterValue")
    private static String getSystemProperty(final String key) {
        if (System.getSecurityManager() == null) {
            return System.getProperty(key);
        }

        return AccessController.doPrivileged((PrivilegedAction<String>) () -> System.getProperty(key));
    }
}
