package org.jboss.eap.util.xp.patch.stream.manager.server.wrapper;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.IOException;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import org.jboss.dmr.ModelNode;
import org.jboss.eap.util.xp.patch.stream.manager.ManagerLogger;
import org.jboss.eap.util.xp.patch.stream.manager.ManagerMain;
import org.jboss.eap.util.xp.patch.stream.manager.OperationBuilder;

/**
 * @author <a href="mailto:kabir.khan@jboss.com">Kabir Khan</a>
 */
public class ServerWrapperProxy implements AutoCloseable {
    private static ServerWrapperProxy instance;
    private Path jbossHome;
    private Path modulesDir;

    private volatile ServerSocket socket;
    private volatile Socket clientSocket;
    private volatile DataOutputStream socketOut;
    private volatile DataInputStream socketIn;
    private volatile Process process;

    public ServerWrapperProxy(Path jbossHome, Path modulesDir) {
        this.jbossHome = jbossHome;
        this.modulesDir = modulesDir;
        if (instance != null) {
            instance = this;
            // This is not a user error, rather a guard against our code initialising the
            // wrapper more than once. So no i18n is needed.
            throw new IllegalStateException("ServerWrapper initialised more than once!");
        }
    }

    public void start() throws Exception {
        if (socket != null) {
            // Not a user error, no i18n needed
            throw new IllegalStateException();
        }

        int port;
        boolean success = false;
        while (true) {
            // Open on a random port
            port = (int)(Math.random() * 65535);
            if (port <= 1024) {
                // For these we need root access
                continue;
            }
            try {

                socket = new ServerSocket(port);
                break;
            } catch (IOException e) {
            }
        }
        CountDownLatch latch = new CountDownLatch(1);
        Thread acceptTimeoutThread = new Thread(new Runnable() {
            @Override
            public void run() {
                try {
                    clientSocket = socket.accept();
                    System.out.println(ManagerLogger.LOGGER.externalProcessConnected());
                    socketOut = new DataOutputStream(new BufferedOutputStream(clientSocket.getOutputStream()));
                    socketIn = new DataInputStream(new BufferedInputStream(clientSocket.getInputStream()));
                    latch.countDown();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        });
        acceptTimeoutThread.start();


        try {
            System.out.println(ManagerLogger.LOGGER.tryingToStartExternalProcessWithEmbeddedServer());
            process = new ProcessBuilder(
                    javaExecutable(),
                    //"-agentlib:jdwp=transport=dt_socket,server=y,suspend=y,address=5005",
                    "-classpath",
                    jarLocation(),
                    ServerWrapperRemote.class.getName(),
                    jbossHome.toAbsolutePath().toString(),
                    modulesDir.toAbsolutePath().toString(),
                    String.valueOf(port))
                    .inheritIO()
                    .start();
        } catch (IOException e) {
            throw e;
        }

        if (process.isAlive()) {
            if (!latch.await(10, TimeUnit.SECONDS)) {
                throw ManagerLogger.LOGGER.externalProcessDidNotConnectOnTime();
            }
            sendCommand(Protocol.START_CMD);
        }
    }

    private String javaExecutable() {
        StringBuilder sb = new StringBuilder();
        sb.append(System.getProperty("java.home"))
                .append(File.separator).append("bin")
                .append(File.separator).append("java");
        if (File.separator.equals("\\")) {
            sb.append(".exe");
        }
        return sb.toString();
    }

    private String jarLocation() throws URISyntaxException {
        // To develop on this and be able to debug in an IDE we need to hardcode the path
        //if (true) {
        //    return "/Users/kabir/sourcecontrol/jboss-eap7/jboss-eap-xp-patch-stream-manager/target/jboss-eap-xp-patch-stream-manager-3.0.0.Final-SNAPSHOT.jar",
        //}
        URL url = ManagerMain.class.getProtectionDomain().getCodeSource().getLocation();
        Path path = Paths.get(url.toURI());
        return path.toAbsolutePath().toString();
    }

    public ModelNode execute(ModelNode operation) throws IOException, ServerException {
        String response = sendCommand(Protocol.formatWithPayload(Protocol.EXECUTE_OPERATION_CMD, operation.toJSONString(true)));
        return getResult(ModelNode.fromJSONString(response));
    }

    private ModelNode getResult(ModelNode result) {
        if (result.get("outcome").asString().equals("failed")) {
            throw new RuntimeException(result.get("failure-description").asString());
        }
        return result.get("result");
    }

    public void applyPatch(Path patch) throws Exception {
        sendCommand(Protocol.formatWithPayload(Protocol.APPLY_PATCH_CMD, patch.toAbsolutePath().toString()));
    }

    @Override
    public void close() throws Exception {
        stop();
    }

    public void stop() throws Exception {
        if (process != null && process.isAlive()) {
            sendCommand(Protocol.STOP_CMD);
            process.destroy();
            process = null;
        }
        if (socketOut != null) {
            Protocol.safeClose(socketOut);
            socketOut = null;
        }
        if (socketIn != null) {
            Protocol.safeClose(socketIn);
            socketIn = null;
        }
        if (clientSocket != null) {
            Protocol.safeClose(clientSocket);
            clientSocket = null;
        }
        if (socket != null) {
            Protocol.safeClose(socket);
            socket = null;
        }
    }

    private String sendCommand(String command) throws IOException, ServerException {
        if (clientSocket == null || clientSocket.isClosed()) {
            throw new IllegalStateException();
        }
        if (socketOut == null) {
            throw new IllegalStateException();
        }
        socketOut.writeUTF(command);
        socketOut.flush();

        String response = socketIn.readUTF();
        return Protocol.checkResponseAndExtractAnyPayload(response);
    }

//    public static void main(String[] args) throws Exception {
//        Path jbossHome = Paths.get("/Users/kabir/VirtualBox VMs/Shared/jboss-eap-7.4");
//        Path modulesDir = jbossHome.resolve("modules");
//        ServerWrapperProxy proxy = new ServerWrapperProxy(jbossHome, modulesDir);
//        proxy.start();
//
//        ModelNode result = proxy.execute(
//                new OperationBuilder("read-attribute")
//                        //.addr("core-service", "patching")
//                        .param("name", "product-name")
//                        .build());
//
//        proxy.stop();
//    }
}
