/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.ejb3.stateful;

import java.lang.reflect.Method;
import org.jboss.aop.Dispatcher;
import org.jboss.aop.advice.Interceptor;
import org.jboss.aop.util.MethodHashing;
import org.jboss.aop.util.PayloadKey;
import org.jboss.aspects.remoting.ClusterConstants;
import org.jboss.aspects.remoting.FamilyWrapper;
import org.jboss.aspects.remoting.InvokeRemoteInterceptor;
import org.jboss.ejb3.Container;
import org.jboss.ejb3.ProxyUtils;
import org.jboss.ejb3.remoting.ClusteredIsLocalInterceptor;
import org.jboss.ejb3.remoting.IsLocalInterceptor;
import org.jboss.ejb3.session.BaseSessionRemoteProxy;
import org.jboss.ha.framework.interfaces.LoadBalancePolicy;

/**
 * InvocationHandler for a clustered home SFSB proxy.
 * 
 * @author <a href="mailto:galder.zamarreno@jboss.com">Galder Zamarreno</a>
 */
public class StatefulHomeClusteredProxy extends BaseSessionRemoteProxy
{   
   /** The serialVersionUID */
   private static final long serialVersionUID = 5682364426659525065L;
   
   protected FamilyWrapper family;
   protected LoadBalancePolicy lbPolicy;
   protected String partitionName;

   public StatefulHomeClusteredProxy(Container container, Interceptor[] interceptors, FamilyWrapper family, LoadBalancePolicy lbPolicy, String partitionName)
   {
      super(container, interceptors);
      this.family = family;
      this.lbPolicy = lbPolicy;
      this.partitionName = partitionName;
   }

   public StatefulHomeClusteredProxy(String containerId, String containerGuid, Interceptor[] interceptors, FamilyWrapper family, LoadBalancePolicy lbPolicy, String partitionName)
   {
      super(containerId, containerGuid, interceptors);
      this.family = family;
      this.lbPolicy = lbPolicy;
      this.partitionName = partitionName;
   }

   public StatefulHomeClusteredProxy()
   {
   }

   public Object invoke(Object proxy, Method method, Object[] args)
           throws Throwable
   {
      long hash = MethodHashing.calculateHash(method);
      Object ret = ProxyUtils.handleCallLocally(hash, proxy, this, method, args);
      if (ret != null)
      {
         return ret;
      }
      
      ret = handleEjb21CallLocally(method, args);
      if (ret != null)
      {
         return ret;
      }

      StatefulRemoteInvocation sri = new StatefulRemoteInvocation(interceptors, hash, method, method, null, null);
      sri.setArguments(args);
      sri.setInstanceResolver(metadata);
      sri.getMetaData().addMetaData(Dispatcher.DISPATCHER, Dispatcher.OID, containerId, PayloadKey.AS_IS);
      sri.getMetaData().addMetaData(ClusterConstants.CLUSTERED_REMOTING, ClusterConstants.CLUSTER_FAMILY_WRAPPER, family, PayloadKey.AS_IS);
      sri.getMetaData().addMetaData(ClusterConstants.CLUSTERED_REMOTING, ClusterConstants.LOADBALANCE_POLICY, lbPolicy, PayloadKey.AS_IS);
      sri.getMetaData().addMetaData(InvokeRemoteInterceptor.REMOTING, InvokeRemoteInterceptor.SUBSYSTEM, "AOP", PayloadKey.AS_IS);
      sri.getMetaData().addMetaData(ClusteredIsLocalInterceptor.PARTITION_NAME, ClusteredIsLocalInterceptor.PARTITION_NAME, partitionName, PayloadKey.TRANSIENT);
      sri.getMetaData().addMetaData(IsLocalInterceptor.IS_LOCAL, IsLocalInterceptor.GUID, containerGuid, PayloadKey.AS_IS);
      
      return sri.invokeNext();
   }

   public String toString()
   {
      return containerId.toString() + ":Home:Clustered";
   }

   public Object getAsynchronousProxy(Object proxy)
   {
      throw new RuntimeException("NOT IMPLEMENTED");
   }

   private Object handleEjb21CallLocally(Method method, Object[] args)
   {
      if (method.equals(ProxyUtils.GET_HOME_HANDLE))
      {
         return homeHandle;
      }
      else if (method.equals(ProxyUtils.GET_EJB_METADATA))
      {
         return ejbMetaData;
      }

      return null;
   }
}
