/*
Copyright 2016 Red Hat, Inc.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
 */

package org.jboss.as.domain.http.server;

import static org.jboss.as.domain.http.server.Constants.INTERNAL_SERVER_ERROR;
import static org.jboss.as.domain.http.server.DomainUtil.safeClose;

import java.io.IOException;
import java.io.PrintWriter;
import java.util.Queue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.atomic.AtomicLongFieldUpdater;

import org.jboss.com.sun.net.httpserver.HttpExchange;
import org.jboss.com.sun.net.httpserver.HttpHandler;

/**
 * Represents a limit on a number of running requests. Adapted from the Undertow class of the same name.
 *
 * @author Stuart Douglas
 * @author Brian Stansberry
 */
class RequestLimit {
    @SuppressWarnings("unused")
    private volatile long state;

    private static final AtomicLongFieldUpdater<RequestLimit> stateUpdater = AtomicLongFieldUpdater.newUpdater(RequestLimit.class, "state");

    private static final long MASK_MAX = longBitMask(32, 63);
    private static final long MASK_CURRENT = longBitMask(0, 30);

    /**
     * The handler that will be invoked if the queue is full.
     */
    private volatile HttpHandler failureHandler = new HttpHandler() {
        @Override
        public void handle(HttpExchange httpExchange) throws IOException {
            httpExchange.sendResponseHeaders(513, -1);
        }
    };

    private final Queue<SuspendedRequest> queue;


    RequestLimit(int maximumConcurrentRequests) {
        this(maximumConcurrentRequests, -1);
    }

    /**
     * Construct a new instance. The maximum number of concurrent requests must be at least one.
     *
     * @param maximumConcurrentRequests the maximum concurrent requests
     * @param queueSize                 The maximum number of requests to queue
     */
    RequestLimit(int maximumConcurrentRequests, int queueSize) {
        if (maximumConcurrentRequests < 1) {
            throw new IllegalArgumentException("Maximum concurrent requests must be at least 1");
        }
        state = (maximumConcurrentRequests & 0xFFFFFFFFL) << 32;

        this.queue = new LinkedBlockingQueue<SuspendedRequest>(queueSize <= 0 ? Integer.MAX_VALUE : queueSize);
    }

    public void handleRequest(final HttpExchange exchange, final DomainApiHandler next) throws IOException {
        long oldVal, newVal;
        do {
            oldVal = state;
            final long current = oldVal & MASK_CURRENT;
            final long max = (oldVal & MASK_MAX) >> 32L;
            if (current >= max) {
                if (!queue.offer(new SuspendedRequest(exchange, next))) {
                    failureHandler.handle(exchange);
                }
                return;
            }
            newVal = oldVal + 1;
        } while (!stateUpdater.compareAndSet(this, oldVal, newVal));
        handleStreamRead(next, exchange);
    }

    private void decrementRequests() {
        stateUpdater.decrementAndGet(this);
    }

    private void handleStreamRead(DomainApiHandler next, HttpExchange exchange) {

        try {
            while (next != null) {
                DomainApiHandler current = next;
                next = null;
                current.safeHandle(exchange);
                final SuspendedRequest task = queue.poll();
                if (task != null) {
                    next = task.next;
                    exchange = task.exchange;
                }
            }
        } finally {
            decrementRequests();
        }
    }

    private static long longBitMask(int low, int high) {
        return (high == 63 ? 0L : (1L << (long) high + 1L)) - (1L << (long) low);
    }

    private static final class SuspendedRequest {
        final HttpExchange exchange;
        final DomainApiHandler next;

        private SuspendedRequest(HttpExchange exchange, DomainApiHandler next) {
            this.exchange = exchange;
            this.next = next;
        }
    }
}
