/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2016, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.as.test.integration.logging.misc;

import org.jboss.as.test.integration.logging.syslogserver.BlockedAllProtocolsSyslogServerEventHandler;
import static java.net.HttpURLConnection.HTTP_OK;
import java.net.URI;
import java.net.URL;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.TimeUnit;
import org.jboss.arquillian.container.test.api.Deployment;
import org.jboss.arquillian.container.test.api.OperateOnDeployment;
import org.jboss.arquillian.container.test.api.RunAsClient;
import org.jboss.arquillian.junit.Arquillian;
import org.jboss.arquillian.test.api.ArquillianResource;
import org.jboss.as.arquillian.api.ServerSetup;
import org.jboss.as.arquillian.api.ServerSetupTask;
import org.jboss.as.arquillian.container.ManagementClient;
import org.jboss.as.controller.PathAddress;
import static org.jboss.as.controller.descriptions.ModelDescriptionConstants.SUBSYSTEM;
import org.jboss.as.controller.operations.common.Util;
import org.jboss.as.test.integration.logging.misc.ReconnectSyslogServerTestCase.ReconnectSyslogServerSetupTask;
import org.jboss.as.test.integration.logging.operations.DefaultLoggingServlet;
import org.jboss.as.test.integration.logging.syslogserver.TCPSyslogServerConfig;
import org.jboss.as.test.integration.logging.syslogserver.UDPSyslogServerConfig;
import org.jboss.as.test.integration.security.common.Utils;
import org.jboss.as.test.shared.TimeoutUtil;
import org.jboss.dmr.ModelNode;
import org.jboss.shrinkwrap.api.ShrinkWrap;
import org.jboss.shrinkwrap.api.spec.WebArchive;
import org.junit.Assert;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.productivity.java.syslog4j.SyslogConstants;
import org.productivity.java.syslog4j.server.SyslogServer;
import org.productivity.java.syslog4j.server.SyslogServerConfigIF;
import org.productivity.java.syslog4j.server.SyslogServerEventIF;
import org.productivity.java.syslog4j.server.SyslogServerIF;

/**
 * Test case whether logging to syslog through UDP and TCP is possible if syslog server is restarted once and twice.
 *
 * Regression test case for https://bugzilla.redhat.com/show_bug.cgi?id=1295660.
 *
 * @author olukas
 */
@RunWith(Arquillian.class)
@RunAsClient
@ServerSetup(ReconnectSyslogServerSetupTask.class)
public class ReconnectSyslogServerTestCase {

    private static final int SYSLOG_UDP_PORT = 10514;
    private static final int SYSLOG_TCP_PORT = 10515;

    private static final String DEPLOYMENT = "logging-deployment";

    private static final String SYSLOG_UDP_HANDLER_NAME = "SYSLOG_UDP_HANDLER";
    private static final String SYSLOG_TCP_HANDLER_NAME = "SYSLOG_TCP_HANDLER";

    private static SyslogServerIF udpServer;
    private static SyslogServerIF tcpServer;

    @ArquillianResource
    private ManagementClient managementClient;

    private static final int ADJUSTED_SECOND = TimeoutUtil.adjust(1000);

    @Test
    @OperateOnDeployment(DEPLOYMENT)
    public void testReconnectSyslogServer(@ArquillianResource URL url) throws Exception {
        final URI servletUri = new URL(url.toExternalForm() + "/logger").toURI();
        final String host = Utils.getHost(managementClient);

        final BlockingQueue<SyslogServerEventIF> udpQueue = BlockedAllProtocolsSyslogServerEventHandler.getQueue("udp");
        final BlockingQueue<SyslogServerEventIF> tcpQueue = BlockedAllProtocolsSyslogServerEventHandler.getQueue("tcp");
        udpQueue.clear();
        tcpQueue.clear();

        // logging before syslog restart
        makeOneLog(servletUri);
        SyslogServerEventIF udpSyslogEvent = udpQueue.poll(5 * ADJUSTED_SECOND, TimeUnit.MILLISECONDS);
        Assert.assertNotNull("No message was logged into the UDP syslog", udpSyslogEvent);
        SyslogServerEventIF tcpSyslogEvent = tcpQueue.poll(5 * ADJUSTED_SECOND, TimeUnit.MILLISECONDS);
        Assert.assertNotNull("No message was logged into the TCP syslog", tcpSyslogEvent);

        stopSyslogServers();

        makeOneLogs_syslogIsOffline(servletUri);
        udpSyslogEvent = udpQueue.poll(1 * ADJUSTED_SECOND, TimeUnit.MILLISECONDS);
        Assert.assertNull("Message was logged into the UDP syslog even if syslog server should be stopped", udpSyslogEvent);
        tcpSyslogEvent = tcpQueue.poll(1 * ADJUSTED_SECOND, TimeUnit.MILLISECONDS);
        Assert.assertNull("Message was logged into the TCP syslog even if syslog server should be stopped", tcpSyslogEvent);

        startSyslogServers(host);

        // logging after first syslog restart
        makeOneLog(servletUri);
        udpSyslogEvent = udpQueue.poll(5 * ADJUSTED_SECOND, TimeUnit.MILLISECONDS);
        Assert.assertNotNull("No message was logged into the UDP syslog after first syslog server restart", udpSyslogEvent);
        tcpSyslogEvent = tcpQueue.poll(5 * ADJUSTED_SECOND, TimeUnit.MILLISECONDS);
        Assert.assertNotNull("No message was logged into the TCP syslog after first syslog server restart", tcpSyslogEvent);

        stopSyslogServers();

        makeOneLogs_syslogIsOffline(servletUri);
        udpSyslogEvent = udpQueue.poll(1 * ADJUSTED_SECOND, TimeUnit.MILLISECONDS);
        Assert.assertNull("Message was logged into the UDP syslog even if syslog server should be stopped", udpSyslogEvent);
        tcpSyslogEvent = tcpQueue.poll(1 * ADJUSTED_SECOND, TimeUnit.MILLISECONDS);
        Assert.assertNull("Message was logged into the TCP syslog even if syslog server should be stopped", tcpSyslogEvent);

        startSyslogServers(host);

        // logging after second syslog restart
        makeOneLog(servletUri);
        udpSyslogEvent = udpQueue.poll(5 * ADJUSTED_SECOND, TimeUnit.MILLISECONDS);
        Assert.assertNotNull("No message was logged into the UDP syslog after second syslog server restart", udpSyslogEvent);
        tcpSyslogEvent = tcpQueue.poll(5 * ADJUSTED_SECOND, TimeUnit.MILLISECONDS);
        Assert.assertNotNull("No message was logged into the TCP syslog after second syslog server restart", tcpSyslogEvent);

    }

    @Deployment(name = DEPLOYMENT)
    public static WebArchive deployment() {
        final WebArchive war = ShrinkWrap.create(WebArchive.class, DEPLOYMENT + ".war");
        war.addClasses(DefaultLoggingServlet.class);
        return war;
    }

    private void makeOneLog(URI uri) throws Exception {
        Utils.makeCall(uri, HTTP_OK);
    }

    /**
     * This is expected behavior when syslog is offline. When syslog goes down then at least two TCP messages are needed for
     * success reconnect and even number of UDP messages is needed as well.
     *
     * @param uri
     * @throws Exception
     */
    private void makeOneLogs_syslogIsOffline(URI uri) throws Exception {
        makeOneLog(uri);
        makeOneLog(uri);
    }

    private static SyslogServerIF createAndStartSyslogInstance(SyslogServerConfigIF config, String host, int port, String protocol) {
        config.setUseStructuredData(true);
        config.setHost(host);
        config.setPort(port);
        config.addEventHandler(new BlockedAllProtocolsSyslogServerEventHandler(protocol));
        SyslogServerIF syslogServer = SyslogServer.createInstance(protocol, config);
        SyslogServer.getThreadedInstance(protocol);
        return syslogServer;
    }

    private static void startSyslogServers(String host) throws InterruptedException {
        udpServer = createAndStartSyslogInstance(
                new UDPSyslogServerConfig(),
                host,
                SYSLOG_UDP_PORT,
                SyslogConstants.UDP);
        tcpServer = createAndStartSyslogInstance(
                new TCPSyslogServerConfig(),
                host,
                SYSLOG_TCP_PORT,
                SyslogConstants.TCP);
        // reconnection timeout is 5sec for TCP syslog handler
        Thread.sleep(6 * ADJUSTED_SECOND);
    }

    private static void stopSyslogServers() throws InterruptedException {
        SyslogServer.shutdown();
        udpServer.setThread(null);
        udpServer.getConfig().removeAllEventHandlers();
        tcpServer.setThread(null);
        tcpServer.getConfig().removeAllEventHandlers();
    }

    static class ReconnectSyslogServerSetupTask implements ServerSetupTask {

        @Override
        public void setup(ManagementClient managementClient, String containerId) throws Exception {
            // clear created server instances (TCP/UDP)
            SyslogServer.shutdown();

            String host = Utils.getHost(managementClient);

            startSyslogServers(host);

            //setup EAP for TCP syslog
            PathAddress tcpSyslogAddress = PathAddress.pathAddress()
                    .append(SUBSYSTEM, "logging")
                    .append("custom-handler", SYSLOG_TCP_HANDLER_NAME);
            ModelNode tcpSyslogHandler = Util.createAddOperation(tcpSyslogAddress);
            tcpSyslogHandler.get("class").set("org.jboss.logmanager.handlers.SyslogHandler");
            tcpSyslogHandler.get("module").set("org.jboss.logmanager");
            tcpSyslogHandler.get("formatter").set("%-5p [%c] (%t) %s%E%n");
            tcpSyslogHandler.get("encoding").set("ISO-8859-1");
            tcpSyslogHandler.get("enabled").set("true");
            ModelNode tcpSyslogProperties = tcpSyslogHandler.get("properties");
            tcpSyslogProperties.get("appName").set("JBossEAP");
            tcpSyslogProperties.get("facility").set("LOCAL_USE_5");
            tcpSyslogProperties.get("serverHostname").set(host);
            tcpSyslogProperties.get("hostname").set("-");
            tcpSyslogProperties.get("port").set(SYSLOG_TCP_PORT);
            tcpSyslogProperties.get("syslogType").set("RFC5424");
            tcpSyslogProperties.get("protocol").set("TCP");
            tcpSyslogProperties.get("messageDelimiter").set("-");
            tcpSyslogProperties.get("useMessageDelimiter").set("true");
            Utils.applyUpdate(tcpSyslogHandler, managementClient.getControllerClient());

            //setup EAP for UDP syslog
            PathAddress udpSyslogAddress = PathAddress.pathAddress()
                    .append(SUBSYSTEM, "logging")
                    .append("syslog-handler", SYSLOG_UDP_HANDLER_NAME);
            ModelNode udpSyslogHandler = Util.createAddOperation(udpSyslogAddress);
            udpSyslogHandler.get("enabled").set("true");
            udpSyslogHandler.get("app-name").set("JBossEAP");
            udpSyslogHandler.get("facility").set("LOCAL_USE_5");
            udpSyslogHandler.get("server-address").set(host);
            udpSyslogHandler.get("hostname").set("-");
            udpSyslogHandler.get("port").set(SYSLOG_UDP_PORT);
            udpSyslogHandler.get("syslog-format").set("RFC5424");
            Utils.applyUpdate(udpSyslogHandler, managementClient.getControllerClient());

            // add logging category
            PathAddress categoryLoggerAddress = PathAddress.pathAddress()
                    .append(SUBSYSTEM, "logging")
                    .append("logger", DefaultLoggingServlet.class.getPackage().getName());
            ModelNode registerTcpSyslogHandler = Util.createAddOperation(categoryLoggerAddress);
            registerTcpSyslogHandler.get("level").set("INFO");
            registerTcpSyslogHandler.get("handlers").add(SYSLOG_TCP_HANDLER_NAME);
            registerTcpSyslogHandler.get("handlers").add(SYSLOG_UDP_HANDLER_NAME);
            Utils.applyUpdate(registerTcpSyslogHandler, managementClient.getControllerClient());

        }

        @Override
        public void tearDown(ManagementClient managementClient, String containerId) throws Exception {
            PathAddress categoryLoggerAddress = PathAddress.pathAddress()
                    .append(SUBSYSTEM, "logging")
                    .append("logger", DefaultLoggingServlet.class.getPackage().getName());
            Utils.applyUpdate(Util.createRemoveOperation(categoryLoggerAddress), managementClient.getControllerClient());

            PathAddress udpAddress = PathAddress.pathAddress()
                    .append(SUBSYSTEM, "logging")
                    .append("syslog-handler", SYSLOG_UDP_HANDLER_NAME);
            Utils.applyUpdate(Util.createRemoveOperation(udpAddress), managementClient.getControllerClient());

            PathAddress tcpAddress = PathAddress.pathAddress()
                    .append(SUBSYSTEM, "logging")
                    .append("custom-handler", SYSLOG_TCP_HANDLER_NAME);
            Utils.applyUpdate(Util.createRemoveOperation(tcpAddress), managementClient.getControllerClient());

            stopSyslogServers();
        }
    }
}
