/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2022 Red Hat, Inc., and individual contributors
 * as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.installer.common;

import org.jboss.installer.core.LoggerUtils;

import java.awt.Font;
import java.awt.FontFormatException;
import java.awt.GraphicsEnvironment;
import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;

public class FontResources {

    private static final List<String> UNSUPPORTED_LOCALS = Arrays.asList("zh", "ja");
    private static Font openSansRegular;
    private static Font openSansLight;
    private static Font openSansLightSubHeading;
    private static Font regularBaseFont;
    private static Font monospaceFont;
    private static Font titleBaseFont;
    private static Locale locale;

    private static void loadOpenSansRegular() throws IOException, FontFormatException {
        InputStream is = FontResources.class.getResourceAsStream("/fonts/opensans-regular.ttf");
        openSansRegular = Font.createFont(0, is).deriveFont(13.0f);
        GraphicsEnvironment.getLocalGraphicsEnvironment().registerFont(openSansRegular);
    }

    private static void loadOpenSansLight() throws IOException, FontFormatException {
        InputStream is = FontResources.class.getResourceAsStream("/fonts/opensans-light.ttf");
        openSansLight = Font.createFont(0, is).deriveFont(24.0f);
        GraphicsEnvironment.getLocalGraphicsEnvironment().registerFont(openSansLight);
        openSansLightSubHeading = openSansLight.deriveFont(19.0f);
        GraphicsEnvironment.getLocalGraphicsEnvironment().registerFont(openSansLightSubHeading);
    }

    private static void loadRegularBaseFont() throws IOException, FontFormatException {
        regularBaseFont = new Font(Font.DIALOG, Font.PLAIN, 12);
        GraphicsEnvironment.getLocalGraphicsEnvironment().registerFont(regularBaseFont);
    }

    private static void loadMonospaceFont() throws IOException, FontFormatException {
        monospaceFont = new Font(Font.MONOSPACED, Font.PLAIN, 12);
        GraphicsEnvironment.getLocalGraphicsEnvironment().registerFont(monospaceFont);
    }

    private static void loadTitleBaseFont() throws IOException, FontFormatException {
        titleBaseFont = new Font(Font.DIALOG, Font.PLAIN, 24);
        GraphicsEnvironment.getLocalGraphicsEnvironment().registerFont(titleBaseFont);
    }

    public static void setLocale(Locale locale) {
        FontResources.locale = locale;
    }

    public static Font getOpenSansRegular() {
        if(locale == null || UNSUPPORTED_LOCALS.contains(locale.getLanguage())){
            return getRegularBaseFont();
        }

        if (openSansRegular == null) {
            try {
                loadOpenSansRegular();
            } catch (IOException | FontFormatException e) {
                LoggerUtils.systemLog.error("Unable to load fonts", e);
            }
        }
        return openSansRegular;
    }

    public static Font getOpenSansLight() {
        if(locale == null || UNSUPPORTED_LOCALS.contains(locale.getLanguage())){
            return getTitleBaseFont();
        }
        if (openSansLight == null) {
            try {
                loadOpenSansLight();
            } catch (IOException | FontFormatException e) {
                LoggerUtils.systemLog.error("Unable to load fonts", e);
            }
        }
        return openSansLight;
    }

    public static Font getOpenSansLightSubHeading() {
        if(locale == null || UNSUPPORTED_LOCALS.contains(locale.getLanguage())){
            return getTitleBaseFont();
        }
        if (openSansLightSubHeading == null) {
            try {
                loadOpenSansLight();
            } catch (IOException | FontFormatException e) {
                LoggerUtils.systemLog.error("Unable to load fonts", e);
            }
        }
        return openSansLightSubHeading;
    }

    public static Font getTitleBaseFont() {
        if(titleBaseFont == null) {
            try {
                loadTitleBaseFont();
            }
            catch (FontFormatException | IOException e) {
                LoggerUtils.systemLog.error("Unable to load fonts", e);
            }
        }
        return titleBaseFont;
    }

    public static Font getRegularBaseFont() {
        if(regularBaseFont == null) {
            try {
                loadRegularBaseFont();
            }
            catch (IOException | FontFormatException e) {
                LoggerUtils.systemLog.error("Unable to load fonts", e);
            }
        }
        return regularBaseFont;
    }

    public static Font getMonospaceFont() {
        if(monospaceFont == null) {
            try {
                loadMonospaceFont();
            }
            catch (IOException | FontFormatException e) {
                LoggerUtils.systemLog.error("Unable to load fonts", e);
            }
        }
        return monospaceFont;
    }
}
