/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2021 Red Hat, Inc., and individual contributors
 * as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.installer.screens;

import org.jboss.installer.core.InstallationData;
import org.jboss.installer.core.InstallerRuntimeException;
import org.jboss.installer.core.LanguageUtils;
import org.jboss.installer.navigation.ScreenNavState;

import javax.swing.BorderFactory;
import javax.swing.ButtonGroup;
import javax.swing.JEditorPane;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JScrollPane;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.event.ItemEvent;
import java.io.IOException;
import java.net.URL;

public class LicenseScreen extends DefaultScreen {

    public static final String NAME = "LicenseScreen";

    public static final String TITLE_KEY = "license_screen.title";
    public static final String SIDE_NAV_KEY = "license_screen.side_nav";
    public static final String ACCEPT_LICENSE_KEY = "license_screen.accept_license";
    public static final String DO_NOT_ACCEPT_LICENSE_KEY = "license_screen.do_not_accept_license";
    public static final String JBOSS_EULA_FILENAME = "jboss_eula.html";
    public static final String ERROR_MISSING_LICENSE_FILE_KEY = "license_screen.error.missing_license_file";
    private final ScreenNavState screenNavState;
    private final JRadioButton acceptButton = createRadioButton(langUtils.getString(ACCEPT_LICENSE_KEY), false);
    private final JRadioButton rejectButton = createRadioButton(langUtils.getString(DO_NOT_ACCEPT_LICENSE_KEY), true);
    private boolean licenseAccepted = false;

    public LicenseScreen(ScreenNavState navState, LanguageUtils langUtils) {
        super(langUtils);
        screenNavState = navState;
    }

    @Override
    public String getTitle() {
        return langUtils.getString(TITLE_KEY);
    }

    @Override
    public String getName() {
        return NAME;
    }

    @Override
    public String getSideNavName() {
        return langUtils.getString(SIDE_NAV_KEY);
    }

    @Override
    public JPanel getContent() {
        JPanel content = new JPanel();
        content.setLayout(new GridBagLayout());
        final GridBagConstraints c = initializeConstraints();
        c.insets = FIELD_ROW_INSET;

        final JPanel selectionPanel = getSelectionPanel();

        acceptButton.addItemListener(itemEvent -> {
            screenNavState.setNextEnabled(itemEvent.getStateChange() == ItemEvent.SELECTED);
            licenseAccepted = acceptButton.isSelected();
        });

        JEditorPane licenseText = new JEditorPane();
        try {
            licenseText.setPage(findLicenseFile());
        } catch (IOException e) {
            throw new InstallerRuntimeException(langUtils.getString(ERROR_MISSING_LICENSE_FILE_KEY), e);
        }
        licenseText.setEditable(false);
        licenseText.setFocusable(false);

        c.weightx = 1;
        c.weighty = 10;
        c.fill = GridBagConstraints.BOTH;
        final JScrollPane scrollPane = new JScrollPane(licenseText);
        scrollPane.createVerticalScrollBar();
        scrollPane.setBorder(BorderFactory.createEmptyBorder(0,0,0,0));
        // hack to ensure the textArea's scroll pane isn't taken over by the screen's scroll pane
        scrollPane.setPreferredSize(new Dimension(100, 200));
        content.add(scrollPane, c);

        c.weighty = 1;
        c.fill = GridBagConstraints.HORIZONTAL;
        c.gridy++;
        content.add(selectionPanel, c);

        return content;
    }

    private JPanel getSelectionPanel() {
        ButtonGroup licenseAcceptance = new ButtonGroup();
        licenseAcceptance.add(acceptButton);
        licenseAcceptance.add(rejectButton);

        final JPanel selectionPanel = new JPanel(new GridBagLayout());
        final GridBagConstraints c = initializeConstraints();
        c.anchor = GridBagConstraints.WEST;
        selectionPanel.add(acceptButton, c);
        c.gridy++;
        selectionPanel.add(rejectButton, c);
        return selectionPanel;
    }

    @Override
    public void load(InstallationData installationData) {
        screenNavState.setNextEnabled(licenseAccepted);
    }

    private URL findLicenseFile()  {
        URL url = this.getClass().getClassLoader().getResource(JBOSS_EULA_FILENAME);
        if (url == null) {
            throw new InstallerRuntimeException(langUtils.getString(ERROR_MISSING_LICENSE_FILE_KEY));
        }
        return url;
    }
}
