/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2022 Red Hat, Inc., and individual contributors
 * as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.installer.validators;

import org.apache.commons.lang3.StringUtils;
import org.jboss.installer.core.LanguageUtils;
import org.jboss.installer.core.ValidationResult;
import org.jboss.installer.postinstall.task.secdom.DatabaseSecurity;

import static org.jboss.installer.postinstall.task.secdom.DatabaseSecurity.JdbcConfig.MapperType.Digest;
import static org.jboss.installer.postinstall.task.secdom.DatabaseSecurity.JdbcConfig.MapperType.SaltedDigest;
import static org.jboss.installer.screens.domain.DatabaseSecurityDomainPanel.SQL_QUERY_LABEL;
import static org.jboss.installer.screens.domain.SelectorPanel.GROUP_SELECTOR_LABEL;
import static org.jboss.installer.screens.domain.SelectorPanel.PASSWORD_SELECTOR_LABEL;
import static org.jboss.installer.screens.domain.SelectorPanel.HASH_ENC_LABEL;
import static org.jboss.installer.screens.domain.SelectorPanel.SALT_SELECTOR_LABEL;
import static org.jboss.installer.screens.domain.SelectorPanel.SALT_ENCODING_LABEL;
import static org.jboss.installer.validators.CertificateSecurityDomainValidator.GENERIC_VALIDATION_EMPTY_FIELD;

public class DataBaseSecurityDomainValidator {

    private final LanguageUtils langUtils;
    private final IntRangeValidator intRangeValidator;

    public DataBaseSecurityDomainValidator(LanguageUtils langUtils) {
        this.langUtils = langUtils;
        this.intRangeValidator = new IntRangeValidator(langUtils, 1);
    }

    public ValidationResult validate(DatabaseSecurity.JdbcConfig config) {
        if (StringUtils.isEmpty(config.getSqlQuery())) {
            return ValidationResult.error(langUtils.getString(GENERIC_VALIDATION_EMPTY_FIELD, langUtils.getString(SQL_QUERY_LABEL)));
        }
        // plain
        ValidationResult res = validateIntIndex(config.getPasswordIndex(), langUtils.getString(PASSWORD_SELECTOR_LABEL));
        if (res.getResult() != ValidationResult.Result.OK) {
            return res;
        }

        res = validateIntIndex(config.getGroupIndex(), langUtils.getString(GROUP_SELECTOR_LABEL));
        if (res.getResult() != ValidationResult.Result.OK) {
            return res;
        }

        // simple-digest
        if ((config.getMapperType() == Digest || config.getMapperType() == SaltedDigest) && StringUtils.isEmpty(config.getHashEncoding())) {
            return ValidationResult.error(langUtils.getString(GENERIC_VALIDATION_EMPTY_FIELD, langUtils.getString(HASH_ENC_LABEL)));

        }

        //salted-simple-digest
        if (config.getMapperType() == SaltedDigest) {
            res = validateIntIndex(config.getSaltIndex(), langUtils.getString(SALT_SELECTOR_LABEL));
            if (res.getResult() != ValidationResult.Result.OK) {
                return res;
            }
            if (StringUtils.isEmpty(config.getEncodingAlgorithm())) {
                return ValidationResult.error(langUtils.getString(GENERIC_VALIDATION_EMPTY_FIELD, langUtils.getString(HASH_ENC_LABEL)));
            }
            if (StringUtils.isEmpty(config.getSaltEncoding())) {
                return ValidationResult.error(langUtils.getString(GENERIC_VALIDATION_EMPTY_FIELD, langUtils.getString(SALT_ENCODING_LABEL)));
            }
        }
        return ValidationResult.ok();
    }

    private ValidationResult validateIntIndex(String text, String label) {
        if (StringUtils.isEmpty(text)) {
            return ValidationResult.error(langUtils.getString(GENERIC_VALIDATION_EMPTY_FIELD, label));
        } else {
            ValidationResult res = intRangeValidator.validate(text);
            if (res.getResult() != ValidationResult.Result.OK) {
                return res;
            }
        }
        return ValidationResult.ok();
    }
}
