/*
 * Copyright 2022 Red Hat, Inc. and/or its affiliates
 * and other contributors as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.installer.validators;

import org.jboss.installer.core.ValidationResult;
import org.jboss.installer.test.utils.MockLanguageUtils;
import org.junit.Assert;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

import java.io.File;

import static org.assertj.core.api.Assertions.assertThat;
import static org.jboss.installer.screens.DefaultScreen.HOME_DIR;
import static org.jboss.installer.screens.QuickstartsMavenRepositorySetupScreen.DEFAULT_MAVEN_REPOSITORY;
import static org.jboss.installer.validators.MavenRepositoryValidator.LOCAL_REPO_PATH_ERROR_KEY;
import static org.jboss.installer.validators.MavenRepositoryValidator.REMOTE_REPO_PATH_ERROR_KEY;

public class MavenRepositoryValidatorTest {

    private final MavenRepositoryValidator validator = new MavenRepositoryValidator(new MockLanguageUtils());

    @Rule
    public TemporaryFolder temp = new TemporaryFolder();

    @Test
    public void validateRemoteFileTest() {
        String notExistentRemoteLocation = "https://not-existant-remote-location.com/";
        Assert.assertNull(validator.validate(DEFAULT_MAVEN_REPOSITORY).getMessage());
        Assert.assertEquals(REMOTE_REPO_PATH_ERROR_KEY, validator.validate(notExistentRemoteLocation).getMessage());
    }

    @Test
    public void validateLocalFileTest() {
        String notExistentLocalDirectory = "/not/existent/local/dir";
        Assert.assertNull(validator.validate(HOME_DIR).getMessage());
        Assert.assertEquals(LOCAL_REPO_PATH_ERROR_KEY, validator.validate(notExistentLocalDirectory).getMessage());
    }

    @Test
    public void validateRedirectLinks(){
        String incompleteLink = "https://maven.repository.redhat.com/ga";
        Assert.assertEquals(validator.validate(incompleteLink).getResult().toString(), "OK");
    }

    @Test
    public void validateInvalidPath() throws Exception {
        String invalidPath = temp.newFolder("test").toString() + File.separator + "invalid\0name";
        final ValidationResult result = validator.validate(invalidPath);
        Assert.assertEquals(ValidationResult.Result.ERROR, result.getResult());
        assertThat(result.getMessage())
                .startsWith("generic.invalid_char");
    }

    @Test
    public void validateLocalFileUrlTest() throws Exception {
        final File testRepo = temp.newFolder("test-repo");
        final String testRepoUri = testRepo.toURI().toString();
        System.out.println(testRepoUri);
        Assert.assertEquals(ValidationResult.Result.OK, validator.validate(testRepoUri).getResult());
        Assert.assertEquals(ValidationResult.Result.OK, validator.validate(testRepoUri.replace("file:/", "file:///")).getResult());
    }
}
